#' Define a Markov Model
#' 
#' Combine information on parameters, transition matrix and 
#' states defined through \code{\link{define_parameters}}, 
#' \code{\link{define_matrix}} and 
#' \code{\link{define_state_list}} respectively.
#' 
#' This function checks whether the objects are compatible
#' in the same model (same state names...).
#' 
#' @param parameters Optional. An object generated by 
#'   \code{\link{define_parameters}}.
#' @param transition_matrix An object generated by 
#'   \code{\link{define_matrix}}.
#' @param states An object generated by 
#'   \code{\link{define_state_list}}.
#'   
#' @return An object of class \code{uneval_model} (a list 
#'   containing the unevaluated parameters, matrix and
#'   states).
#'   
#' @export
#' 
#' @examples
#' 
#' param <- define_parameters(
#'   a = 1,
#'   b = 4 * markov_cycle,
#'   c = a + b
#' )
#' 
#' mat <- define_matrix(
#'   state_names = c("s1", "s2"),
#'   1 / c, 1 - 1/ c,
#'   0, 1
#' )
#' 
#' list_states <- define_state_list(
#'   s1 = define_state(cost = 234),
#'   s2 = define_state(cost = 421)
#' )
#' 
#' define_model(
#'  parameters = param,
#'  transition_matrix = mat,
#'  states = list_states
#' )
#' 
define_model <- function(
  parameters = define_parameters(),
  transition_matrix = define_matrix(),
  states = define_state_list()
) {
  
  stopifnot(
    get_state_number(states) == 0 |
      get_state_number(states) == get_matrix_order(transition_matrix),
    length(
      intersect(
        get_parameter_names(parameters),
        get_state_value_names(states)
      )
    ) == 0,
    
    identical(
      sort(get_state_names(states)),
      sort(get_state_names(transition_matrix))
    )
  )
  
  structure(
    list(
      parameters = parameters,
      transition_matrix = transition_matrix,
      states = states
    ), class = "uneval_model")
}

#' @export
print.uneval_model <- function(x, ...) {
  n_parm <- length(get_parameter_names(get_parameters(x)))
  n_states <- get_state_number(get_states(x))
  n_state_values <- length(get_state_value_names(get_states(x)))
  
  cat(sprintf(
    "An unevaluated Markov model:

    %i parameter%s,
    %i state%s,
    %i state value%s.\n",
    n_parm,
    plur(n_parm),
    n_states,
    plur(n_states),
    n_state_values,
    plur(n_state_values)
  ))
}

#' Get Markov Model Parameters
#' 
#' Works on both unevaluated and evaluated models.
#' 
#' For internal use.
#' 
#' @param x An \code{uneval_model} or \code{eval_model}
#'   object.
#'   
#' @return An \code{uneval_parameters} or
#'   \code{eval_parameters} object.
get_parameters <- function(x){
  UseMethod("get_parameters")
}

get_parameters.default <- function(x){
  x$parameters
}

#' Get Markov Model Transition Matrix
#' 
#' Works on both unevaluated and evaluated models.
#' 
#' For internal use.
#' 
#' @param x An \code{uneval_model} or \code{eval_model}
#'   object.
#'   
#' @return An \code{uneval_matrix} or \code{uneval_matrix}
#'   object.
get_matrix <- function(x){
  UseMethod("get_matrix")
}

get_matrix.default <- function(x){
  x$transition_matrix
}

get_states <- function(x){
  UseMethod("get_states")
}

get_states.default <- function(x){
  x$states
}

get_counts <- function(x){
  UseMethod("get_counts")
}

get_counts.eval_model <- function(x){
  x$counts
}

#' Evaluate Markov Model
#' 
#' Given an unevaluated Markov Model, an initial number of 
#' individual and a number of cycle to compute, returns the 
#' evaluated version of the objects and the count of 
#' individual per state per model cycle.
#' 
#' \code{init} need not be integer. E.g. specifying a vector
#' of type c(Q = 1, B = 0, C = 0, ...) returns the 
#' probabilities for an individual starting in state A to be
#' in each state, per cycle.
#' 
#' @param model An \code{uneval_model} object.
#' @param cycles positive integer. Number of Markov Cycles 
#'   to compute.
#' @param init numeric vector, same length as number of 
#'   model states. Number of individuals in each model state
#'   at the beginning.
#' @param count_args Additional arguments passed as a list
#'   to \code{compute_counts}.
#'   
#' @return An \code{eval_model} object (actually a list of 
#'   evaluated parameters, matrix, states and cycles 
#'   counts).
#' 
#' @examples
#' 
#' \dontrun{
#' param <- define_parameters(
#'   a = markov_cycle + 1 * 2
#' )
#' 
#' mat <- define_matrix(
#'   1-1/a, 1/a,
#'   .1,    .9
#' )
#' 
#' sta <- define_state_list(
#'   A = define_state(cost = 10),
#'   B = define_state(cost = 2)
#' )
#' 
#' mod <- define_model(
#'   parameters = param,
#'   transition_matrix = mat,
#'   states = sta
#' )
#' 
#' eval_model(
#'   mod,
#'   init = c(10, 5),
#'   cycles = 5
#' )
#' }
#' 
eval_model <- function(
  model,
  cycles, 
  init,
  count_args = NULL
) {
  
  stopifnot(
    cycles > 0,
    length(cycles) == 1,
    all(init >= 0)
  )
  
  parameters <- eval_parameters(get_parameters(model),
                                cycles = cycles)
  transition_matrix <- eval_matrix(get_matrix(model),
                                   parameters)
  states <- eval_state_list(get_states(model), parameters)
  
  count_table <- do.call(
    compute_counts,
    c(transition_matrix = list(transition_matrix),
      init = list(init),
      count_args))
  
  
  values <- compute_values(states, count_table)
  
  structure(
    list(
      parameters = parameters,
      transition_matrix = transition_matrix,
      states = states,
      counts = count_table,
      values = values
    ),
    class = c("eval_model"),
    init = init,
    cycles = cycles)
}

get_state_values <- function(x) {
  x$values
}

#' Compute Count of Individual in Each State per Cycle
#' 
#' Given an initial number of individual and an evaluated 
#' transition matrix, returns the number of individual per 
#' state per cycle.
#' 
#' Use the \code{method} argument to specify if transitions
#' are supposed to happen at the beginning or the end of
#' each cycle. Alternatively linear interpolation between 
#' cycles can be performed.
#' 
#' @param transition_matrix An \code{eval_matrix} object.
#' @param init numeric vector, same length as number of 
#'   model states. Number of individuals in each model state
#'   at the beginning.
#' @param method character. One of \code{final},
#'   \code{initial}, \code{exponential} or \code{linear}.
#'   See details.
#' @param round logical scalar. Should individual counts be rounded?
#'   
#' @return A \code{cycle_counts} object.
#' 
compute_counts <- function(
  transition_matrix, init,
  method = c("final", "initial", "exponential", "linear"),
  round = FALSE
) {
  
  stopifnot(
    length(init) == get_matrix_order(transition_matrix)
  )
  
  method <- match.arg(method)
  
  
  list_counts <- Reduce(
    "%*%",
    transition_matrix,
    init,
    accumulate = TRUE
  )
  
  res <- dplyr::as.tbl(
    as.data.frame(
      matrix(
        unlist(list_counts),
        byrow = TRUE,
        ncol = get_matrix_order(transition_matrix)
      )
    )
  )
  
  colnames(res) <- get_state_names(transition_matrix)
  
  n0 <- res[- nrow(res), ]
  n1 <- res[-1, ]
  
  switch(
    method,
    initial = {
      out <- if (round) round(n0) else n0
    },
    final = {
      out <- if (round) round(n1) else n1
    },
    exponential = {
      stop("Unimplemented")
    },
    linear = {
      out <- (n0 + n1) / 2
    },
    {
      stop()
    }
  )
  
  structure(out, class = c("cycle_counts", class(out)))
  
}


#' Compute State Values per Cycle
#' 
#' Given states and counts, computes the total state values
#' per cycle.
#' 
#' @param states An object of class \code{eval_state_list}.
#' @param counts An object of class \code{cycle_counts}.
#'   
#' @return A data.frame of state values, one column per
#'   state value and one row per cycle.
#'   
compute_values <- function(states, counts) {
  
  states_names <- get_state_names(states)
  state_values_names <- get_state_value_names(states)
  
  res <- data.frame(
    markov_cycle = states[[1]]$markov_cycle
  )
  # bottleneck!
  for (state_value in state_values_names) {
    res[state_value] <- 0
    
    for (state in states_names) {
      res[state_value] <-
        res[state_value] +
        counts[, state] * 
        states[[state]][, state_value]
    }
  }
  res
}

get_state_value_names.uneval_model <- function(x) {
  get_state_value_names(get_states(x))
}

get_state_names.uneval_model <- function(x, ...) {
  get_state_names(get_states(x))
}

