#' Run one or more Markov Model
#' 
#' Runs one or more unevaluated Markov Models. When more 
#' than one model is provided, all models should have the 
#' same states and state value names.
#' 
#' 
#' A usual situation where more than one model needs to be 
#' run is when comparing different care startegies.
#' 
#' In order to compute comparisons Markov Models must be 
#' similar (same states and state value names). Thus models 
#' should only differ through parameters, transition matrix 
#' cell values and values attached to states (but not state 
#' value names).
#' 
#' The initial number of individuals in each state and the 
#' number of cycle will be the same for all models.
#' 
#' \code{state_cycle_limit} can be specified in 3 different
#' ways: 1. As a single value: the limit is applied to all
#' states in all models. 2. As a named vector (where names
#' are state names): the limits are applied to the given
#' state names, for all models. 3. As a named list of named
#' vectors: the limits are applied to the given state names
#' for the given models.
#' 
#' Internally this function does 2 operations: first 
#' evaluating parameters, transition matrix, state values 
#' and computing individual counts through 
#' \code{\link{eval_model}}; and then using individual 
#' counts and evaluated state values to compute values at 
#' each cycle through \code{compute_values}.
#' 
#' @param ... One or more \code{uneval_model} object.
#' @param parameters Optional. An object generated by 
#'   \code{\link{define_parameters}}.
#' @param init numeric vector, same length as number of 
#'   model states. Number of individuals in each model state
#'   at the beginning.
#' @param cycles positive integer. Number of Markov Cycles 
#'   to compute.
#' @param cost Names or expression to compute cost on the 
#'   cost-effectiveness plane.
#' @param effect Names or expression to compute effect on 
#'   the cost-effectiveness plane.
#' @param method Counting method.
#' @param list_models List of models, only used by 
#'   \code{run_model_} to avoid using \code{...}.
#' @param state_cycle_limit Optional expansion limit for 
#'   \code{state_cycle}, see details.
#'   
#' @return A list of evaluated models with computed values.
#' @export
#' 
#' @example inst/examples/example_run_model.R
#'   
run_model <- function(...,
                       parameters = define_parameters(),
                       init = c(1000L, rep(0L, get_state_number(get_states(list(...)[[1]])) - 1)),
                       cycles = 1,
                       method = c("life-table", "beginning", "end",
                                  "half-cycle"),
                       cost = NULL, effect = NULL,
                       state_cycle_limit = NULL) {
  
  list_models <- list(...)
  
  method <- match.arg(method)
  
  run_model_(
    list_models = list_models,
    parameters = parameters,
    init = init,
    cycles = cycles,
    method = method,
    cost = lazyeval::lazy_(substitute(cost), env = parent.frame()),
    effect = lazyeval::lazy_(substitute(effect), env = parent.frame()),
    state_cycle_limit = state_cycle_limit
  )
}

#' @export
#' @rdname run_model
run_model_ <- function(list_models,
                        parameters,
                        init,
                        cycles,
                        method,
                        cost, effect,
                        state_cycle_limit) {
  
  if (! is.wholenumber(cycles)) {
    stop("'cycles' must be a whole number.")
  }
  
  if (! all(unlist(lapply(
    list_models,
    function(x) "uneval_model" %in% class(x))))) {
    
    .x <- names(list_models[! unlist(lapply(
      list_models,
      function(x) "uneval_model" %in% class(x)))])
    
    stop(sprintf(
      "Incorrect model object%s: %s.",
      plur(length(.x)),
      paste(.x, collapse = ", ")
    ))
  }
  
  list_ce <- list(
    .cost = cost,
    .effect = effect
  )
  
  ce <- c(
    lazyeval::lazy_dots(),
    list_ce
  )
  
  model_names <- names(list_models)
  
  if (is.null(model_names)) {
    message("No named model -> generating names.")
    model_names <- as.character(utils::as.roman(seq_along(list_models)))
    names(list_models) <- model_names
  }
  
  if (any(model_names == "")) {
    warning("Not all models are named -> generating names.")
    model_names <- as.character(utils::as.roman(seq_along(list_models)))
    names(list_models) <- model_names
  }
  
  if (! list_all_same(lapply(list_models,
                             function(x) sort(get_state_names(x))))) {
    stop("State names differ between models.")
  }
  
  if (! list_all_same(lapply(list_models,
                             function(x) sort(get_state_value_names(x))))) {
    stop("State value names differ between models.")
  }
  
  if (! length(init) == get_state_number(list_models[[1]])) {
    stop(sprintf(
      "Length of 'init' vector (%i) differs from number of states (%i).",
      length(init),
      get_state_number(list_models[[1]])
    ))
  }
  
  if (! any(init > 0)) {
    stop("At least one init count must be > 0.")
  }
  
  if (is.null(names(init)))
    names(init) <- get_state_names(list_models[[1]])
  
  if (! all(sort(names(init)) == sort(get_state_names(list_models[[1]])))) {
    stop("Names of 'init' vector differ from state names.")
  }
  
  state_cycle_limit <- complete_scl(
    state_cycle_limit,
    state_names = get_state_names(list_models[[1]]),
    model_names = names(list_models),
    cycles = cycles
  )
  
  eval_model_list <- list()
  
  for (n in names(list_models)) {
    eval_model_list[[n]] <- eval_model(
      list_models[[n]], 
      parameters = parameters,
      init = init, 
      cycles = cycles,
      method = method,
      expand_limit = state_cycle_limit[[n]]
    )
  }
  
  list_res <- lapply(eval_model_list, get_total_state_values)
  
  for (n in model_names){
    list_res[[n]]$.model_names <- n
  }
  
  res <- Reduce(dplyr::bind_rows, list_res) %>% 
    dplyr::mutate_(.dots = ce)
  
  base_model <- get_base_model(res)
  
  structure(
    res,
    eval_model_list = eval_model_list,
    uneval_model_list = list_models,
    class = c("run_model", class(res)),
    parameters = parameters,
    init = init,
    cycles = cycles,
    method = method,
    ce = ce,
    base_model = base_model
  )
}

get_model_names <- function(x) {
  x$.model_names
}

get_model_count <- function(x) {
  nrow(x)
}

get_total_state_values <- function(x) {
  # faster than as.data.frame or dplyr::as_data_frame
  res <- as.list(colSums((x$values)[- 1]))
  class(res) <- "data.frame"
  attr(res, "row.names") <- c(NA, -1)
  res
}

get_base_model <- function(x, ...) {
  UseMethod("get_base_model")
}

get_base_model.default <- function(x, ...) {
  if (! all(c(".cost", ".effect") %in% names(x))) {
    warning("No effect defined, cannot find base model.")
    return(NULL)
  }
  (x %>% 
    dplyr::arrange_(.dots = list(~ .cost, ~ desc(.effect))) %>% 
    dplyr::slice(1))$.model_names
}

get_base_model.run_model <- function(x, ...) {
  attr(x, "base_model")
}

get_lowest_model <- function(x) {
  x$.model_names[x$.effect == min(x$.effect)][1]
}

#' Get Model Values
#' 
#' Given a result from \code{\link{run_model}}, return 
#' cost and effect values for a specific model.
#' 
#' @param x Result from \code{\link{run_model}}.
#' @param m Model name or index.
#' @param ...	further arguments passed to or from other
#'   methods.
#'   
#' @return A data frame of values per state.
#' @export
get_values <- function(x, ...) {
  UseMethod("get_values")
}

#' @rdname get_values
#' @export
get_values.run_model <- function(x, m = 1, ...) {
  check_model_index(x, m, ...)
  get_values(attr(x, "eval_model_list")[[m]])
}

#' @rdname get_values
#' @export
get_values.eval_model <- function(x, ...) {
  x$values
}

#' @rdname get_values
#' @export
get_values.list <- function(x, ...) {
  x$values
}

#' @rdname get_values
#' @export
get_values.updated_models <- function(x, m, ...) {
  get_values(attributes(x)$combined_models, m, ...)
}

#' @rdname get_values
#' @export
get_values.combined_models <- function(x, m, ...){
  x <- attributes(x)$eval_model_list
  x$.model_names <- names(x)
  check_model_index(x, m)
  get_values(x[[m]])
}

#' Get State Membership Counts
#' 
#' Given a result from \code{\link{run_model}}, return 
#' state membership counts for a specific model.
#' 
#' @param x Result from \code{\link{run_model}}.
#' @param m Model name or index.
#' @param ...	further arguments passed to or from other
#'   methods.
#'   
#' @return A data frame of counts per state.
#' @export
get_counts <- function(x, ...) {
  UseMethod("get_counts")
}

#' @rdname get_counts
#' @export
get_counts.run_model <- function(x, m = 1, ...) {
  check_model_index(x, m, ...)
  get_counts(attr(x, "eval_model_list")[[m]])
}

#' @rdname get_counts
#' @export
get_counts.eval_model <- function(x, ...) {
  x$counts
}

#' @rdname get_counts
#' @export
get_counts.list <- function(x, ...) {
  x$counts
}

#' @rdname get_counts
#' @export
get_counts.updated_models <- function(x, m, ...) {
  get_counts(attributes(x)$combined_models, m, ...)
}

#' @rdname get_counts
#' @export
get_counts.combined_models <- function(x, m, ...){
  x <- attributes(x)$eval_model_list
  x$.model_names <- names(x)
  check_model_index(x, m)
  get_counts(x[[m]])
}

#' Get Initial State Values
#' 
#' @param x x Result from \code{\link{run_model}}.
#'   
#' @return A vector of initial state values.
#' @export
get_init <- function(x) {
  attr(x, "init")
}
