% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/input-data.R
\name{input_mats}
\alias{input_mats}
\title{Input matrices for a statistical model}
\usage{
input_mats(
  X,
  strategy_id,
  n_strategies,
  patient_id,
  n_patients,
  line = NULL,
  n_lines = NULL,
  state_id = NULL,
  n_states = NULL,
  transition_id = NULL,
  n_transitions = NULL,
  time_id = NULL,
  time_intervals = NULL,
  n_times = NULL,
  time_reset = NULL,
  time_fun = NULL
)
}
\arguments{
\item{X}{A list of input matrices for predicting the values of each parameter in a statistical model. May also be
a list of lists of input matrices when a list of separate models is fit (e.g., with \link{flexsurvreg_list}).}

\item{strategy_id}{A numeric vector of integers denoting the treatment strategy represented by each row
in \code{X}.}

\item{n_strategies}{A scalar denoting the number of unique treatment strategies.}

\item{patient_id}{A numeric vector of integers denoting the patient represented by each row
in \code{X}.}

\item{n_patients}{A scalar denoting the number of unique patients.}

\item{line}{A numeric vector of integers denoting the treatment line represented by each row
in \code{X}. Not supported by currently available models.}

\item{n_lines}{A \code{data.table} denoting the number of treatment lines associated
with each treatment strategy. Should contain a column, "strategy_id", and a column,
"N". Not supported by currently available models.}

\item{state_id}{A numeric vector of integers denoting the health state represented by each row
in \code{X}.}

\item{n_states}{A scalar denoting the number of unique health states.}

\item{transition_id}{A numeric vector denoting the 
health state transition represented by each row in \code{X}. This must only be specified when
estimating the health state transitions with a joint likelihood function. If independent
models are fit for each transition, then separate \code{X} matrices must be specified
for each transition. Note that this is not currently supported but
will be supported once \code{hesim} provides support for state transition modeling.}

\item{n_transitions}{A scalar denoting the number of unique transitions. 
Not supported by currently available models.}

\item{time_id}{A numeric vector of integers denoting a unique time interval.}

\item{time_intervals}{A \code{data.table} denotes unique time intervals. Must 
contain the columns \code{time_id}, \code{time_start}, and \code{time_stop}.
\code{time_start} is the starting time of an interval and \code{time_stop} is
the stopping time of an interval. Time intervals are closed on the left and
open on the right, and in the final interval, \code{time_stop} is equal to 
infinity.}

\item{n_times}{A scalar denoting the number of time intervals. Equal to the
number of rows in \code{time_intervals}.}

\item{time_reset}{If \code{TRUE}, then time intervals reset each time a patient enters a new health 
state. This is relevant if, for example, costs vary over time within health states. 
If \code{FALSE} then time intervals are based on time since the start of the simulation.}

\item{time_fun}{A pointer to a C++ functor that can be used to update \code{X} as a function
of time in a simulation model. Not currently supported.}
}
\description{
Create an object of class "input_mats", which contains inputs matrices
for simulating a statistical model. Consists of (i) input matrices, \code{X},
(ii) ID variables indexing the rows of each matrix in \code{X}, and (iii) the dimensions of
the \code{X} matrices. More details are provided under "Details" below. Note that an "input_mats" 
object should be created using \code{\link{create_input_mats}}.
}
\details{
Each row of each matrix \code{X} is an input vector, \eqn{x_{hijk}}, where \eqn{h} denotes
a health-related index, \eqn{i} indexes a patient, \eqn{j} indexes a treatment line,
and \eqn{k} is a treatment strategy. A health-related index is either a health state
(e.g., \code{state_id}) or a transition between health states (e.g., \code{transition_id}).
In some cases, the health-related index \eqn{h} can be suppressed and separate models
can be fit for each health index. This is, for instance, the case in a partitioned survival 
model where separate models are fit for each survival endpoint. Likewise, models 
can be fit without multiple treatment lines as would, again, be the case in a 
partitioned survival analysis where sequential treatment would be incorporated by 
adding additional health states rather than by using the index \eqn{j}.

The rows of the matrices in \code{X} must be sorted in a manner consistent with the ID variables.
The sorting order should be the same as specified in \code{\link{expand.hesim_data}}; that is,
the rows of \code{X} must be sorted by: (i) \code{strategy_id}, (ii) \code{line}, 
(iii) \code{patient_id}, and (iv) the health-related ID variable (either \code{state_id} or
 \code{transition_id}).
}
\examples{
strategies <- data.frame(strategy_id = c(1, 2))
patients <- data.frame(patient_id = seq(1, 3), 
                          age = c(45, 47, 60),
                          female = c(1, 0, 0),
                          group = factor(c("Good", "Medium", "Poor")))
hesim_dat <- hesim_data(strategies = strategies,
                        patients = patients)

dat <- expand(hesim_dat, by = c("strategies", "patients"))
input_mats <- input_mats(X = list(mu = model.matrix(~ age, dat)),
                         strategy_id = dat$strategy_id,
                         n_strategies = length(unique(dat$strategy_id)),
                         patient_id = dat$patient_id,
                        n_patients = length(unique(dat$patient_id)))
print(input_mats)
}
\seealso{
\code{\link{create_input_mats}}
}
