% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hkevp.fit.R
\name{hkevp.fit}
\alias{hkevp.fit}
\title{Fitting procedure of the HKEVP}
\usage{
hkevp.fit(y, sites, knots, niter, nburn, nthin, trace, spatial.covar, quiet,
  GEV.vary, latent.correlation.type, GEV.init, alpha.init, tau.init, A.init,
  B.init, range.init, sill.init, constant.GEV.prior, latent.beta.prior,
  alpha.prior, tau.prior, sill.prior, range.prior, GEV.random.walk,
  range.random.walk, tau.random.walk, alpha.random.walk, A.random.walk,
  B.random.walk)
}
\arguments{
\item{y}{A numerical matrix of real values.
Observed block maxima process: each column corresponds to a site position whose coordinates are given by \code{sites} and each row represents a block of observations (e.g. a specific year).}

\item{sites}{A numerical matrix of real values.
Coordinates of the sites: each row corresponds to a position and each column to a spatial coordinate.}

\item{knots}{A numerical matrix of real values.
Coordinates of the knots: each row corresponds to a knot position and each column to a spatial coordinate. If missing, \code{knots} is equal to \code{sites}. See details for how to place the knots.}

\item{niter}{A positive integer.
Number of iterations before the algorithm stops.}

\item{nburn}{A positive integer.
Number of iterations to be discarded to assess convergence. Half of \code{niter} is taken by default.}

\item{nthin}{A positive integer.
Size of the thinning, 1 by default.}

\item{trace}{A positive integer.
If \code{quiet} is FALSE, the log-likelihood of the model is displayed each block of \code{trace} iterations, which serves to observe the progression of the algorithm.}

\item{spatial.covar}{A numerical matrix.
Spatial covariates associated to the sites. Each row corresponds to a position and each column to a spatial covariate. The first column should be filled with ones to represent the intercept. See details.}

\item{quiet}{Logical.
Whether or not the progression of the routine may be displayed. If TRUE, nothing is displayed (quiet mode) and if FALSE (default), log-likelihood is given every block of \code{trace} iterations and acceptance rates are shown at the end of the algorithm.}

\item{GEV.vary}{A logical vector of size three.
Each element corresponds to a GEV parameter (respectively the location, the scale and the shape) and indicates if it is spatially-varying (TRUE) or spatially-constant (FALSE, by default for the shape parameter). In the latter case, the value of the GEV parameter is the same for each site.}

\item{latent.correlation.type}{A character string.
Indicates the form of the correlation functions of the latent Gaussian processes associated to the GEV parameters. Must be one of \code{"expo"}, \code{"gauss"}, \code{"mat32"} (By default) and \code{"mat52"}, respectively corresponding to the exponential, Gaussian, Matern-3/2 and Matern-5/2 correlation functions.}

\item{GEV.init}{A matrix of numerical values.
Initial values for the GEV parameters. Must have three columns and same number of rows as in \code{sites}. If missing, default values are used (see details).}

\item{alpha.init}{A numerical value in (0,1].
Initial value of the dependence parameter \eqn{\alpha}.}

\item{tau.init}{A positive numerical value.
Initial value of the bandwith parameter \eqn{\tau}. By default, the minimum distance between knots is taken. Should be lower than twice the maximum distance between sites (see details).}

\item{A.init}{A positive numerical value.
Initial value of the random effect \eqn{A}. The same initial state is considered for each knot and for each observation of the process.}

\item{B.init}{A numerical value in (0,1).
Initial value of the auxiliary variable \eqn{B} used to update the random effect \eqn{A}. See appendices in \cite{Reich and Shaby (2012)} or the method of \cite{Stephenson (2009)}.}

\item{range.init}{A positive numerical value.
Initial value of the range parameter of the latent Gaussian processes associated to the GEV parameters. Must be lower than twice the maximum distance between sites. By default, one third of this maximum distance is taken as initial value. See details.}

\item{sill.init}{A positive numerical value.
Initial value of the sill parameter of the latent Gaussian processes associated to the GEV parameters.}

\item{constant.GEV.prior}{A numerical matrix with two lines and three columns.
Normal prior hyperparameters for the spatially-constant GEV parameters. The first (resp. second) line corresponds to the mean (resp. standard deviation). Each column corresponds to a GEV parameter.}

\item{latent.beta.prior}{A numerical value.
Standard deviation of the normal prior associated to the \eqn{\beta} parameters in the mean of the latent processes.}

\item{alpha.prior}{A two-dimensional vector of positive values.
Parameters of the Beta prior for \eqn{\alpha}. By default, the uniform distribution is considered.}

\item{tau.prior}{A two-dimensional vector of positive values.
Parameters of the Beta prior for \eqn{\tau} over the interval \eqn{(0, 2D_{max})}, where \eqn{D_{max}} is the maximum distance between sites. By default, a Beta(2,5) distribution is considered.}

\item{sill.prior}{A two-dimensional vector of positive values.
Parameters of the Inverse-Gamma conjugate prior for the sill parameter of the latent Gaussian processes. The prior is the same for all spatially-varying GEV parameters.}

\item{range.prior}{A two-dimensional vector of positive values.
Parameters of the Beta prior for the range parameter of the latent Gaussian processes. The Beta prior is considered over the interval \eqn{(0, 2D_{max})}, where \eqn{D_{max}} is the maximum distance between sites (see details). The prior is the same for all spatially-varying GEV parameters.}

\item{GEV.random.walk}{A three-dimensional vector of positive values.
Standard deviations of the normal random walk for each GEV parameters.}

\item{range.random.walk}{A three-dimensional vector of positive values.
Standard deviations of the normal random walk for the range parameter of the latent Gaussian processes.}

\item{tau.random.walk}{A positive value.
Standard deviation of the normal random walk for the bandwith parameter \eqn{\tau}.}

\item{alpha.random.walk}{A positive value.
Standard deviation of the normal random walk for the dependence parameter \eqn{\alpha}.}

\item{A.random.walk}{A positive value.
Standard deviation of the normal random walk for the random effect \eqn{A}.}

\item{B.random.walk}{A positive value.
Standard deviation of the normal random walk for the auxiliary variable \eqn{B}.}
}
\value{
A named list with twelve elements:
\itemize{
\item{\code{GEV}: a three-dimensional array of real values. Markov chains associated to the GEV parameters per site. The dimensions of the array correspond respectively to the sites positions, the three GEV parameters and the states of the Markov chains.}
\item{\code{alpha}: a column vector. Markov chain associated to the dependence parameter \eqn{\alpha}.}
\item{\code{tau}: a column vector. Markov chain associated to the dependence parameter \eqn{\tau}.}
\item{\code{A}: a three-dimensional array of real values. Markov chains associated to the positive stable random effect per site and per block. The dimensions correspond respectively to the indices of blocks, the knots positions and the states of the Markov chains.}
\item{\code{llik}: a column vector. Log-likelihood of the model for each step of the algorithm.}
\item{\code{time}: a positive value. Time (in sec) spent for the fit.}
\item{\code{spatial}: a named list with four elements linked to the GEV spatially-varying parameters:
\itemize{
\item{\code{vary}: a column vector of length three. Numerical version of the argument \code{GEV.vary}.}
\item{\code{beta}: a three-dimensional array of real values. The dimensions correspond respectively to the states of the Markov chains, the associated spatial covariates and the three GEV parameters}
\item{\code{sills}: a three column matrix of real values. Markov chains associated to the sills in the correlation functions of the latent Gaussian processes. Each column corresponds to one GEV parameter.}
\item{\code{ranges}: a three column matrix of real values. Markov chains associated to the ranges in the correlation functions of the latent Gaussian processes. Each column corresponds to one GEV parameter.}
}}
\item{\code{knots}: the set of knots.}
\item{\code{spatial.covar}}: the spatial covariates.
\item{\code{latent.correlation.type}: the type of correlation function that describe the latent Gaussian processes.}
\item{\code{nstep}: a positive integer. Number of states \code{niter-nburn} obtained at the end of the routine.}
}
}
\description{
Metropolis-within-Gibbs algorithm that returns posterior distribution (as Markov chains) for the marginal and dependence parameters of the HKEVP.
The implementation has been done such that user has to provide the minimum input parameters as possible, but convergence of the chains should be assessed, using for instance \code{\link{mcmc.plot}}. Initial states, prior hyperparameters and magnitude of random walks can be set by the experimented user.
}
\details{
\itemize{
\item{The positive integer arguments \code{niter}, \code{nburn} and \code{nthin} control the length of the resulting chains in the MCMC algorithm. The routine operates over a loop of size \code{niter} and for each step, the elements of the Markov chains are updated \code{nthin} times to account for thinning procedure. At the end, the first \code{nburn} iterations are then discarded. 
As a result, the length of the Markov chains given by the \code{hkevp.fit} procedure is \code{niter-nburn}. The \code{hkevp.fit} function updates the Markov chains \code{niter*nthin} times.}

\item{The spatial covariates for each position are given by the \code{spatial.covar} matrix argument. These values are used in the linear modelling of the mean of the latent Gaussian processes associated to the GEV parameters. If this argument is given by the user, the first column must be filled with ones to represent the intercept and other columns should be standardized. If not given, the default spatial covariates used are the intercept and the sites positions.}

\item{Perhaps the most restrictive feature when fitting the HKEVP to observed data comes from the choice of the knots.
A trade-off may be done here, since too few knots may result in a poor fit while too many knots increases drastically the computational burden of the algorithm.
A possible advice for the choice of \code{knots} is to take a gridded network of positions over the studied region, by using for instance the routine \code{expand.grid} from R base package.
If this argument is not furnished, the knots will coincide with the site positions.}

\item{If the initial values of the GEV parameters are not given through \code{GEV.init}, the median of the observed data are taken for the location parameter (by site if \code{GEV.vary[1]} is TRUE, or over all data if \code{GEV.vary[1]} is FALSE). For the scale and shape parameters, the default initial values are respectively 1 and 0.01.}

\item{A change has been made from the original MCMC procedure described in \cite{Reich and Shaby (2012)} concerning the bandwidth parameter \eqn{\tau}. Instead of being defined over the positive real values and with a vague inverse-Gamma prior, we chose to restrict this parameter to the interval \eqn{(0,2D_{max}]}, where \eqn{D_{max}} is the maximum distance between sites. A Beta(2,5) prior is set by default on this interval. For this reason, the initial value \code{tau.init} should be between 0 and \eqn{2D_{max}}. A similar remark can be done for the range parameters of the latent Gaussian processes and the associated initial values \code{range.init}}

\item{In the appendix of \cite{Reich and Shaby (2012)} about the MCMC algorithm, an adaptive procedure taking place in the burn-in period is presented, but it has not been implemented in this function. Instead, acceptance rates can be displayed with the function \code{\link{mcmc.accept}}.}}
}
\examples{
# Simulation of HKEVP:
sites <- as.matrix(expand.grid(1:3,1:3))
knots <- sites
mu <- sites[,1]*10
sigma <- 3
xi <- .2
alpha <- .4
tau <- 1
ysim <- hkevp.rand(10, sites, knots, mu, sigma, xi, alpha, tau)

# HKEVP fit:
fit <- hkevp.fit(ysim, sites, knots, niter = 100, nburn = 50, quiet = FALSE)





}
\author{
Quentin Sebille
}
\references{
Reich, B. J., & Shaby, B. A. (2012). A hierarchical max-stable spatial model for extreme precipitation. The annals of applied statistics, 6(4), 1430. <DOI:10.1214/12-AOAS591>

Stephenson, A. G. (2009) High-dimensional parametric modelling of multivariate extreme events. Aust. N. Z. J Stat, 51, 77-88. <DOI:10.1111/j.1467-842X.2008.00528.x>
}

