\name{coreVennTree}
\alias{coreVennTree}
\title{
Phylogeny with Branches Colored Based on Venn Diagram
}
\description{
Plots a phylogeny with branches colored based on the compartment that they are associated with in the Venn diagram generated by the \code{corePhyloVenn} function.
}
\usage{
coreVennTree(x, grouping, core_fraction,
mode = 'branch',rooted=TRUE, ordered_groups=NULL,branch_color=NULL,
remove_zeros=TRUE,plot.chronogram=FALSE)
}
\arguments{
\item{x}{
  (Required) Microbial community data. This must be in the form of a phyloseq object and must contain, at a minimum, an OTU abundance table and a phylogeny.
}
\item{grouping}{
(Required) A vector specifying which samples belong to which habitat type.
}
\item{core_fraction}{
(Required) The fraction of samples that a microbial taxon must be found in to be considered part of the 'core' microbiome.
}
\item{mode}{
Whether to build a tip-based ('tip') or a branch-based ('branch') phylogeny. The default is 'branch'.
}
\item{rooted}{
Whether to include the root of the phylogeny. The default is TRUE, meaning that the root is necessarily included in all phylogenies. This requires that the input tree be rooted.
}
\item{ordered_groups}{
When provided, specifies the order in which different habitats should be plotted. This order is matched to the color order specified in \code{fill_color} (see below).
}
\item{branch_color}{
A vector specifying what colors to use for branches associated with each of the different habitat combinations in the Venn diagram. This vector must be as long as the number of possible habitat combinations (number of compartments in the Venn diagram plus one for branches not included in the core of any habitats). When no colors are specified or a vector of the wrong length is specified, the default is to use a range of colors from blue to red.
}
\item{remove_zeros}{
Whether or not to remove taxa that are missing from all samples prior to drawing the phylogeny. The default is TRUE.
}
\item{plot.chronogram}{
Whether to plot a phylogeny or a chronogram. The default is FALSE.
}

}
\details{
\code{coreVennTree} generates a phylogeny with branches colored according to the compartments of an associated Venn diagram as generated using the \code{coreVenn} function. For more details, see Bewick and Camper (2025).
}
\value{
This function returns a color coded plot of the phylogeny. When a vector of colors is specified, the color key is printed out in the console.
}

\references{
Bewick, S.A. and Benjamin T. Camper. "Phylogenetic Measures of the Core Microbiome" <doi:TBD>

McMurdie, Paul J., and Susan Holmes. "phyloseq: an R package for reproducible interactive analysis and graphics of microbiome census data." PloS one 8.4 (2013): e61217.

McMurdie, Paul J., and Susan Holmes. "Phyloseq: a bioconductor package for handling and analysis of high-throughput phylogenetic sequence data." Biocomputing 2012. 2012. 235-246.
}

\examples{
#Test with enterotype dataset
library(phyloseq)
library(ape)
library(phytools)
library(ggplot2)
data(enterotype)

set.seed(1)

#Generate an example tree and label it with the names of the microbial taxa
enterotype_tree<-rtree(length(taxa_names(enterotype)))
enterotype_tree$tip.label<-taxa_names(enterotype)

#keep only those samples with gender identified
gendered<-which(!(is.na(sample_data(enterotype)$Gender)))
enterotypeMF<-prune_samples(sample_names(enterotype)[gendered],enterotype)

#Create a phyloseq object with a tree
example_phyloseq<-phyloseq(otu_table(enterotypeMF),phy_tree(as.phylo(enterotype_tree)))

#Define the groups
bygender<-sample_data(enterotypeMF)$Gender

#Define the colors for group combinations
clist<-c('black','red','orange','yellow')

#Plot the tree
coreVennTree(example_phyloseq,grouping=bygender,0.5,branch_color=clist)

}
