% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reportingheterogeneity.R
\name{getLevels}
\alias{getLevels}
\title{Summarize the adjusted and the original self-rated response levels}
\usage{
getLevels(
  model,
  formula = model$thresh.formula,
  data = model$frame,
  sep = "_",
  decreasing.levels = model$decreasing.levels,
  sort.flag = FALSE,
  weight.original = TRUE
)
}
\arguments{
\item{model}{a fitted \code{hopit} model.}

\item{formula}{a formula containing the grouping variables. It is by default set to threshold formula.}

\item{data}{data used to fit the model.}

\item{sep}{a separator for the level names.}

\item{decreasing.levels}{a logical indicating whether self-reported health classes are ordered in increasing order.}

\item{sort.flag}{a logical indicating whether to sort the levels.}

\item{weight.original}{a logical indicating if use survey weights for calcualtion of original responses.}
}
\value{
a list with the following components:
\item{original}{ frequencies of original response levels for selected groups/categories.}
\item{adjusted}{ frequencies of adjusted response levels (Jurges 2007 method) for selected groups/categories.}
\item{N.original}{ the number of original response levels for selected groups/categories.}
\item{N.adjusted}{ the number of adjusted response levels (Jurges 2007 method) for selected groups/categories.}
\item{categories}{ selected groups/categories used in summary.}
\item{tab}{ an original vs. an adjusted contingency table.}
\item{mat}{ a matrix with columns: grouping variables, original response levels, adjusted response levels.
Each row corresponds to a single individual from the data used to fit the model.}
}
\description{
Summarize the adjusted and the original self-rated response levels.
}
\examples{
# DATA
data(healthsurvey)

# the order of response levels decreases from the best health to
# the worst health; hence the hopit() parameter decreasing.levels
# is set to TRUE
levels(healthsurvey$health)

# fit a model
model1 <- hopit(latent.formula = health ~ hypertension + high_cholesterol +
                heart_attack_or_stroke + poor_mobility + very_poor_grip +
                depression + respiratory_problems +
                IADL_problems + obese + diabetes + other_diseases,
              thresh.formula = ~ sex + ageclass + country,
              decreasing.levels = TRUE,
              control = list(trace = FALSE),
              data = healthsurvey)

# Example 1 ---------------------

# calculate a summary by country
hl <- getLevels(model=model1, formula=~ country, sep=' ')
plot(hl, las=1, mar = c(3,2,1.5,0.5))

# differences between frequencies of original and adjusted health levels
round(100*(hl$original - hl$adjusted),2)

# extract good and bad health levels (combined levels)
Org <- cbind(bad = rowSums(hl$original[,1:2]),
             good = rowSums(hl$original[,4:5]))
Adj <- cbind(bad = rowSums(hl$adjusted[,1:2]),
             good = rowSums(hl$adjusted[,4:5]))
round(100*(Org - Adj),2)

# plot the differences
barplot(t(Org - Adj), beside = TRUE, density = 20, angle = c(-45, 45),
        col = c('pink4', 'green2'),
        ylab = 'Original - adjusted reported health frequencies')
abline(h = 0); box()
legend('top', c('Bad health','Good health'),
       density = 20, angle = c(-45, 45),
       fill = c('pink4', 'green2'), bty = 'n', cex = 1.2)

# in country X, bad health seems to be over-reported while good health
# is under-reported; in country Z, good health is highly over-reported.

# Example 2 ---------------------

# summary by gender and age
hl <- getLevels(model = model1, formula=~ sex + ageclass, sep=' ')
plot(hl)

# differences between frequencies of original and adjusted health levels
round(100*(hl$original - hl$adjusted),2)

# extract good health levels (combined "Very good" and "Excellent" levels)
Org <- rowSums(hl$original[,4:5])
Adj <- rowSums(hl$adjusted[,4:5])
round(100*(Org - Adj),2)

pmar <- par('mar'); par(mar = c(9.5, pmar[2:4]))
barplot(Org-Adj,
        ylab = 'Original - adjusted reported good health frequencies',
        las = 3,
        density = 20, angle = c(45, -45), col = c('blue', 'orange'))
abline(h = 0); box(); par(mar = pmar)
legend('top', c('Man','Woman'), density = 20, angle = c(-45, 45),
       fill = c('blue', 'orange'), bty = 'n', cex = 1.2)

# results show that women in general tend to over-report good health,
# while men aged 50-59 greatly under-report good health.

# more examples can be found in the description of the boot_hopit() function.
}
\references{
\insertRef{Jurges2007}{hopit}\cr\cr
\insertRef{OKSUZYAN2019}{hopit}
}
\seealso{
\code{\link{getCutPoints}}, \code{\link{latentIndex}}, \code{\link{standardiseCoef}}, \code{\link{hopit}}.
}
\author{
Maciej J. Danko
}
