% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sat_vapor_pressure.R
\encoding{UTF-8}
\name{kin_visc_water}
\alias{kin_visc_water}
\title{Kinematic Viscosity for liquid Water}
\source{
\enumerate{
   \item r - How to not run an example using roxygen2? - Stack Overflow answered and edited by samkart on Jul 9 2017. (Also see the additional comments in response to the answer.) See \url{https://stackoverflow.com/questions/12038160/how-to-not-run-an-example-using-roxygen2}.
   \item devtools - Issues in R package after CRAN asked to replace dontrun by donttest - Stack Overflow answered by Hong Ooi on Sep 1 2020. (Also see the additional comments in response to the answer.) See \url{https://stackoverflow.com/questions/63693563/issues-in-r-package-after-cran-asked-to-replace-dontrun-by-donttest}.
}
}
\usage{
kin_visc_water(
  rho,
  mu,
  rho_units = c("kg/m^3", "lbm/ft^3", "slug/ft^3"),
  mu_units = c("Pa*s or kg/m/s", "lbf*s/ft^2", "slug/ft/s")
)
}
\arguments{
\item{rho}{numeric vector that contains the water density}

\item{mu}{numeric vector that contains the water dynamic viscosity}

\item{rho_units}{character vector that contains the unit for the density of
water [options are kg/m^3, lbm/ft^3, or slug/ft^3]}

\item{mu_units}{character vector that contains the unit for the dynamic
viscosity of water [options are Pa*s or kg/m/s, lbf*s/ft^2, or slug/ft/s]}
}
\value{
the kinematic viscosity as a numeric vector. The units are not
  returned.
}
\description{
This function solves for the kinematic viscosity of water using only
the water density and the dynamic viscosity.
}
\details{
The simplified equation is expressed as

\deqn{\\nu = \frac{\\mu}{\\rho}}

\describe{
  \item{\emph{\\nu}}{Water Kinematic Viscosity (m^2/s or ft^2/s)}
  \item{\emph{\\rho}}{Water Density (mass divided by volume), slug/ft^3}
  \item{\emph{\\mu}}{Water Dynamic viscosity, slug/ft/s}
}
}
\note{
Note: 1 lbf = 1 slug * 1 ft/sec^2, thus 1 slug = 1 lbf * sec^2 / 1 ft
(Reference 2)

Thus, lbm/ft^3 = lbf*s^2/ft/ft^3
}
\examples{
# Example 1

\donttest{
# See Source 1 and Source 2

library(iemisc)

try(kin_visc_water(mu = 34, rho = 0, rho_units = "kg/m^3", mu_units = "Pa*s or kg/m/s"))
}




# Example 2 (from the Reference)

install.load::load_package("iemisc", "units")

import::from(fpCompare, "\%==\%")


# For water at 68 F (20 C), mu = 2.09 * 10 ^ -8 slug/ft/s and rho = 1.937 slug/ft^3

kin_visc_water(mu = 2.09 * 10 ^ -8, rho = 1.937, rho_units =
"slug/ft^3", mu_units = "slug/ft/s")

# convert the units

rho <- set_units(1.937, slug/ft^3)

mu <- set_units(2.09 * 10 ^ -8, slug/ft/s)

mu1 <- set_units(mu, kg/m/s)

rho1 <- set_units(rho, "kg/m^3")

kin_visc_water(mu = mu1, rho = rho1, rho_units = "kg/m^3", mu_units =
"Pa*s or kg/m/s")

mu2 <- set_units(mu, lbf*s/ft^2)

rho2 <- set_units(rho, lb/ft^3)

kin_visc_water(mu = mu2, rho = rho2, rho_units = "lbm/ft^3", mu_units =
"lbf*s/ft^2")


# compare the results of part 1 and part 3 (they should be equivalent)

kin_visc_water(mu = 2.09 * 10 ^ -8, rho = 1.937, rho_units = "slug/ft^3",
mu_units = "slug/ft/s") \%==\% kin_visc_water(mu = mu2, rho = rho2, rho_units =
"lbm/ft^3", mu_units = "lbf*s/ft^2")





# Example 2 - Example from the hydraulics package

install.load::load_package("iemisc", "units")

import::from(fpCompare, "\%==\%")


nu <- hydraulics::kvisc(T = 55, units = "Eng", ret_units = TRUE); nu

nus <- hydraulics::dvisc(T = 55, units = "Eng", ret_units = TRUE) /
hydraulics::dens(T = 55, units = "Eng", ret_units = TRUE); nus


# compare the results of nu and nus (they should be equivalent)

drop_units(nu) \%==\% drop_units(nus)


nu2 <- dyn_visc_water(Temp = 55, units = "Eng", Eng_units = "lbf*s/ft^2") /
density_water(Temp = 55, units = "Eng", Eng_units = "slug/ft^3"); nu2

nus2 <- kin_visc_water(mu = dyn_visc_water(Temp = 55, units = "Eng", Eng_units =
"lbf*s/ft^2"), rho = density_water(Temp = 55, units = "Eng", Eng_units = "slug/ft^3"),
rho_units = "lbm/ft^3", mu_units = "lbf*s/ft^2"); nus2

# compare the results of nu2 and nus2 (they should be equivalent)

nu2 \%==\% nus2




}
\references{
\enumerate{
   \item Ven Te Chow, Ph.D., \emph{Open-Channel Hydraulics}, McGraw-Hill Classic Textbook Reissue, New York City, New York: McGraw-Hill Book Company, 1988, pages 7-8.
   \item Professor S.A. Kinnas, Commonly used units in CE319F (Elementary Fluid Mechanics), The University of Texas at Austin Department of Civil, Architectural and Environmental Engineering, \url{https://www.caee.utexas.edu/prof/kinnas/319LAB/notes13/units_ce319f_kinnas.pdf}.
}
}
\author{
Irucka Embry
}
