% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jackstraw_subspace.R
\name{jackstraw_subspace}
\alias{jackstraw_subspace}
\title{Jackstraw for the User-Defined Dimension Reduction Methods}
\usage{
jackstraw_subspace(dat, FUN, r = NULL, r1 = NULL, s = NULL, B = NULL,
  covariate = NULL, noise = NULL, verbose = TRUE, seed = NULL)
}
\arguments{
\item{dat}{a data matrix with \code{m} rows as variables and \code{n} columns as observations.}

\item{FUN}{optionally, provide a specfic function to estimate LVs. Must output \code{r} estimated LVs in a \code{n*r} matrix.}

\item{r}{a number of significant latent variables.}

\item{r1}{a numeric vector of latent variables of interest.}

\item{s}{a number of ``synthetic'' null variables. Out of \code{m} variables, \code{s} variables are independently permuted.}

\item{B}{a number of resampling iterations.}

\item{covariate}{a model matrix of covariates with \code{n} observations. Must include an intercept in the first column.}

\item{noise}{specify a parametric distribution to generate a noise term. If \code{NULL}, a non-parametric jackstraw test is performed.}

\item{verbose}{a logical specifying to print the computational progress.}

\item{seed}{a seed for the random number generator.}
}
\value{
\code{jackstraw_subspace} returns a list consisting of
\item{p.value}{\code{m} p-values of association tests between variables and their principal components}
\item{obs.stat}{\code{m} observed statistics}
\item{null.stat}{\code{s*B} null statistics}
}
\description{
Test association between the observed variables and their latent variables, captured by a user-defined dimension reduction method.
}
\details{
This function computes \code{m} p-values of linear association between \code{m} variables and their latent variables,
captured by a user-defined dimension reduction method.
Its resampling strategy accounts for the over-fitting characteristics due to direct computation of PCs from the observed data
and protects against an anti-conservative bias.

This function allows you to specify a parametric distribution of a noise term. It is an experimental feature. Then, a small number \code{s} of observed variables
are replaced by synthetic null variables generated from a specified distribution.
}
\examples{
set.seed(1234)
## simulate data from a latent variable model: Y = BL + E
B = c(rep(1,50),rep(-1,50), rep(0,900))
L = rnorm(20)
E = matrix(rnorm(1000*20), nrow=1000)
dat = B \%*\% t(L) + E
dat = t(scale(t(dat), center=TRUE, scale=TRUE))

## apply the jackstraw with the svd as a function
out = jackstraw_subspace(dat, FUN = function(x) svd(x)$v[,1,drop=FALSE], r=1, s=100, B=50)
}
\references{
Chung and Storey (2015) Statistical significance of variables driving systematic variation in high-dimensional data. Bioinformatics, 31(4): 545-554 \url{http://bioinformatics.oxfordjournals.org/content/31/4/545}

Chung (2018) Statistical significance for cluster membership. biorxiv, doi:10.1101/248633 \url{https://www.biorxiv.org/content/early/2018/01/16/248633}
}
\seealso{
\link{jackstraw_pca} \link{jackstraw}
}
\author{
Neo Christopher Chung \email{nchchung@gmail.com}
}
