% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/callbacks.R
\name{callback_lambda}
\alias{callback_lambda}
\title{Callback for creating simple, custom callbacks on-the-fly.}
\usage{
callback_lambda(
  on_epoch_begin = NULL,
  on_epoch_end = NULL,
  on_train_begin = NULL,
  on_train_end = NULL,
  on_train_batch_begin = NULL,
  on_train_batch_end = NULL,
  ...
)
}
\arguments{
\item{on_epoch_begin}{called at the beginning of every epoch.}

\item{on_epoch_end}{called at the end of every epoch.}

\item{on_train_begin}{called at the beginning of model training.}

\item{on_train_end}{called at the end of model training.}

\item{on_train_batch_begin}{called at the beginning of every train batch.}

\item{on_train_batch_end}{called at the end of every train batch.}

\item{...}{Any function in \code{\link[=Callback]{Callback()}} that you want to override by
passing \verb{function_name = function}. For example,
\code{callback_lambda(.., on_train_end = train_end_fn)}. The custom function
needs to have same arguments as the ones defined in \code{\link[=Callback]{Callback()}}.}
}
\value{
A \code{Callback} instance that can be passed to \code{\link[=fit.keras.src.models.model.Model]{fit.keras.src.models.model.Model()}}.
}
\description{
This callback is constructed with anonymous functions that will be called
at the appropriate time (during \verb{Model.\{fit | evaluate | predict\}}).
Note that the callbacks expects positional arguments, as:
\itemize{
\item \code{on_epoch_begin} and \code{on_epoch_end} expect two positional arguments:
\code{epoch}, \code{logs}
\item \code{on_train_begin} and \code{on_train_end} expect one positional argument:
\code{logs}
\item \code{on_train_batch_begin} and \code{on_train_batch_end} expect two positional
arguments: \code{batch}, \code{logs}
\item See \code{Callback} class definition for the full list of functions and their
expected arguments.
}
}
\section{Examples}{
\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Print the batch number at the beginning of every batch.
batch_print_callback <- callback_lambda(
  on_train_batch_begin = function(batch, logs) \{
    print(batch)
  \}
)

# Stream the epoch loss to a file in new-line delimited JSON format
# (one valid JSON object per line)
json_log <- file('loss_log.json', open = 'wt')
json_logging_callback <- callback_lambda(
  on_epoch_end = function(epoch, logs) \{
    jsonlite::write_json(
      list(epoch = epoch, loss = logs$loss),
      json_log,
      append = TRUE
    )
  \},
  on_train_end = function(logs) \{
    close(json_log)
  \}
)

# Terminate some processes after having finished model training.
processes <- ...
cleanup_callback <- callback_lambda(
  on_train_end = function(logs) \{
    for (p in processes) \{
      if (is_alive(p)) \{
        terminate(p)
      \}
    \}
  \}
)

model \%>\% fit(
  ...,
  callbacks = list(
    batch_print_callback,
    json_logging_callback,
    cleanup_callback
  )
)
}\if{html}{\out{</div>}}
}

\seealso{
\itemize{
\item \url{https://keras.io/api/callbacks/lambda_callback#lambdacallback-class}
}

Other callbacks: \cr
\code{\link{Callback}()} \cr
\code{\link{callback_backup_and_restore}()} \cr
\code{\link{callback_csv_logger}()} \cr
\code{\link{callback_early_stopping}()} \cr
\code{\link{callback_learning_rate_scheduler}()} \cr
\code{\link{callback_model_checkpoint}()} \cr
\code{\link{callback_reduce_lr_on_plateau}()} \cr
\code{\link{callback_remote_monitor}()} \cr
\code{\link{callback_swap_ema_weights}()} \cr
\code{\link{callback_tensorboard}()} \cr
\code{\link{callback_terminate_on_nan}()} \cr
}
\concept{callbacks}
