% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/layers-preprocessing.R
\name{layer_integer_lookup}
\alias{layer_integer_lookup}
\title{A preprocessing layer that maps integers to (possibly encoded) indices.}
\usage{
layer_integer_lookup(
  object,
  max_tokens = NULL,
  num_oov_indices = 1L,
  mask_token = NULL,
  oov_token = -1L,
  vocabulary = NULL,
  vocabulary_dtype = "int64",
  idf_weights = NULL,
  invert = FALSE,
  output_mode = "int",
  sparse = FALSE,
  pad_to_max_tokens = FALSE,
  name = NULL,
  ...
)
}
\arguments{
\item{object}{Object to compose the layer with. A tensor, array, or sequential model.}

\item{max_tokens}{Maximum size of the vocabulary for this layer. This should
only be specified when adapting the vocabulary or when setting
\code{pad_to_max_tokens=TRUE}. If NULL, there is no cap on the size of
the vocabulary. Note that this size includes the OOV
and mask tokens. Defaults to \code{NULL}.}

\item{num_oov_indices}{The number of out-of-vocabulary tokens to use.
If this value is more than 1, OOV inputs are modulated to
determine their OOV value.
If this value is 0, OOV inputs will cause an error when calling
the layer. Defaults to \code{1}.}

\item{mask_token}{An integer token that represents masked inputs. When
\code{output_mode} is \code{"int"}, the token is included in vocabulary
and mapped to index 0. In other output modes,
the token will not appear in the vocabulary and instances
of the mask token in the input will be dropped.
If set to NULL, no mask term will be added. Defaults to \code{NULL}.}

\item{oov_token}{Only used when \code{invert} is \code{TRUE}. The token to return
for OOV indices. Defaults to \code{-1}.}

\item{vocabulary}{Optional. Either an array of integers or a string path to a
text file. If passing an array, can pass a list, list,
1D NumPy array, or 1D tensor containing the integer vocabulary terms.
If passing a file path, the file should contain one line per term
in the vocabulary. If this argument is set,
there is no need to \code{adapt()} the layer.}

\item{vocabulary_dtype}{The dtype of the vocabulary terms, for example
\code{"int64"} or \code{"int32"}. Defaults to \code{"int64"}.}

\item{idf_weights}{Only valid when \code{output_mode} is \code{"tf_idf"}.
A list, list, 1D NumPy array, or 1D tensor or the same length
as the vocabulary, containing the floating point inverse document
frequency weights, which will be multiplied by per sample term
counts for the final TF-IDF weight.
If the \code{vocabulary} argument is set, and \code{output_mode} is
\code{"tf_idf"}, this argument must be supplied.}

\item{invert}{Only valid when \code{output_mode} is \code{"int"}.
If \code{TRUE}, this layer will map indices to vocabulary items
instead of mapping vocabulary items to indices.
Defaults to \code{FALSE}.}

\item{output_mode}{Specification for the output of the layer. Values can be
\code{"int"}, \code{"one_hot"}, \code{"multi_hot"}, \code{"count"}, or \code{"tf_idf"}
configuring the layer as follows:
\itemize{
\item \code{"int"}: Return the vocabulary indices of the input tokens.
\item \code{"one_hot"}: Encodes each individual element in the input into an
array the same size as the vocabulary,
containing a 1 at the element index. If the last dimension
is size 1, will encode on that dimension.
If the last dimension is not size 1, will append a new
dimension for the encoded output.
\item \code{"multi_hot"}: Encodes each sample in the input into a single
array the same size as the vocabulary,
containing a 1 for each vocabulary term present in the sample.
Treats the last dimension as the sample dimension,
if input shape is \verb{(..., sample_length)},
output shape will be \verb{(..., num_tokens)}.
\item \code{"count"}: As \code{"multi_hot"}, but the int array contains
a count of the number of times the token at that index
appeared in the sample.
\item \code{"tf_idf"}: As \code{"multi_hot"}, but the TF-IDF algorithm is
applied to find the value in each token slot.
For \code{"int"} output, any shape of input and output is supported.
For all other output modes, currently only output up to rank 2
is supported. Defaults to \code{"int"}.
}}

\item{sparse}{Boolean. Only applicable to \code{"multi_hot"}, \code{"count"}, and
\code{"tf_idf"} output modes. Only supported with TensorFlow
backend. If \code{TRUE}, returns a \code{SparseTensor}
instead of a dense \code{Tensor}. Defaults to \code{FALSE}.}

\item{pad_to_max_tokens}{Only applicable when \code{output_mode} is \code{"multi_hot"},
\code{"count"}, or \code{"tf_idf"}. If \code{TRUE}, the output will have
its feature axis padded to \code{max_tokens} even if the number
of unique tokens in the vocabulary is less than \code{max_tokens},
resulting in a tensor of shape \verb{(batch_size, max_tokens)}
regardless of vocabulary size. Defaults to \code{FALSE}.}

\item{name}{String, name for the object}

\item{...}{For forward/backward compatability.}
}
\value{
The return value depends on the value provided for the first argument.
If  \code{object} is:
\itemize{
\item a \code{keras_model_sequential()}, then the layer is added to the sequential model
(which is modified in place). To enable piping, the sequential model is also
returned, invisibly.
\item a \code{keras_input()}, then the output tensor from calling \code{layer(input)} is returned.
\item \code{NULL} or missing, then a \code{Layer} instance is returned.
}
}
\description{
This layer maps a set of arbitrary integer input tokens into indexed integer
output via a table-based vocabulary lookup. The layer's output indices will
be contiguously arranged up to the maximum vocab size, even if the input
tokens are non-continguous or unbounded. The layer supports multiple options
for encoding the output via \code{output_mode}, and has optional support for
out-of-vocabulary (OOV) tokens and masking.

The vocabulary for the layer must be either supplied on construction or
learned via \code{adapt()}. During \code{adapt()}, the layer will analyze a data set,
determine the frequency of individual integer tokens, and create a
vocabulary from them. If the vocabulary is capped in size, the most frequent
tokens will be used to create the vocabulary and all others will be treated
as OOV.

There are two possible output modes for the layer.  When \code{output_mode} is
\code{"int"}, input integers are converted to their index in the vocabulary (an
integer).  When \code{output_mode} is \code{"multi_hot"}, \code{"count"}, or \code{"tf_idf"},
input integers are encoded into an array where each dimension corresponds to
an element in the vocabulary.

The vocabulary can optionally contain a mask token as well as an OOV token
(which can optionally occupy multiple indices in the vocabulary, as set
by \code{num_oov_indices}).
The position of these tokens in the vocabulary is fixed. When \code{output_mode}
is \code{"int"}, the vocabulary will begin with the mask token at index 0,
followed by OOV indices, followed by the rest of the vocabulary. When
\code{output_mode} is \code{"multi_hot"}, \code{"count"}, or \code{"tf_idf"} the vocabulary will
begin with OOV indices and instances of the mask token will be dropped.

\strong{Note:} This layer uses TensorFlow internally. It cannot
be used as part of the compiled computation graph of a model with
any backend other than TensorFlow.
It can however be used with any backend when running eagerly.
It can also always be used as part of an input preprocessing pipeline
with any backend (outside the model itself), which is how we recommend
to use this layer.

\strong{Note:} This layer is safe to use inside a \code{tf.data} pipeline
(independently of which backend you're using).
}
\section{Examples}{
\strong{Creating a lookup layer with a known vocabulary}

This example creates a lookup layer with a pre-existing vocabulary.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(12, 36, 1138, 42) |> as.integer()
data <- op_array(rbind(c(12, 1138, 42),
                      c(42, 1000, 36)))  # Note OOV tokens
out <- data |> layer_integer_lookup(vocabulary = vocab)
out
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[1 3 4]
##  [4 0 2]], shape=(2, 3), dtype=int64)

}\if{html}{\out{</div>}}

\strong{Creating a lookup layer with an adapted vocabulary}

This example creates a lookup layer and generates the vocabulary by
analyzing the dataset.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data <- op_array(rbind(c(12, 1138, 42),
                      c(42, 1000, 36)))  # Note OOV tokens
layer <- layer_integer_lookup()
layer |> adapt(data)
layer |> get_vocabulary() |> str()
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## List of 6
##  $ : int -1
##  $ : num 42
##  $ : num 1138
##  $ : num 1000
##  $ : num 36
##  $ : num 12

}\if{html}{\out{</div>}}

Note that the OOV token -1 have been added to the vocabulary. The remaining
tokens are sorted by frequency (42, which has 2 occurrences, is first) then
by inverse sort order.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{layer(data)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[5 2 1]
##  [1 3 4]], shape=(2, 3), dtype=int64)

}\if{html}{\out{</div>}}

\strong{Lookups with multiple OOV indices}

This example demonstrates how to use a lookup layer with multiple OOV
indices.  When a layer is created with more than one OOV index, any OOV
tokens are hashed into the number of OOV buckets, distributing OOV tokens in
a deterministic fashion across the set.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(12, 36, 1138, 42) |> as.integer()
data <- op_array(rbind(c(12, 1138, 42),
                      c(37, 1000, 36)))  # Note OOV tokens
out <- data |>
  layer_integer_lookup(vocabulary = vocab,
                       num_oov_indices = 2)
out
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[2 4 5]
##  [1 0 3]], shape=(2, 3), dtype=int64)

}\if{html}{\out{</div>}}

Note that the output for OOV token 37 is 1, while the output for OOV token
1000 is 0. The in-vocab terms have their output index increased by 1 from
earlier examples (12 maps to 2, etc) in order to make space for the extra
OOV token.

\strong{One-hot output}

Configure the layer with \code{output_mode='one_hot'}. Note that the first
\code{num_oov_indices} dimensions in the ont_hot encoding represent OOV values.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(12, 36, 1138, 42) |> as.integer()
data <- op_array(c(12, 36, 1138, 42, 7), 'int32')  # Note OOV tokens
layer <- layer_integer_lookup(vocabulary = vocab,
                              output_mode = 'one_hot')
layer(data)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[0 1 0 0 0]
##  [0 0 1 0 0]
##  [0 0 0 1 0]
##  [0 0 0 0 1]
##  [1 0 0 0 0]], shape=(5, 5), dtype=int64)

}\if{html}{\out{</div>}}

\strong{Multi-hot output}

Configure the layer with \code{output_mode = 'multi_hot'}. Note that the first
\code{num_oov_indices} dimensions in the multi_hot encoding represent OOV tokens

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(12, 36, 1138, 42) |> as.integer()
data <- op_array(rbind(c(12, 1138, 42, 42),
                      c(42,    7, 36,  7)), "int64")  # Note OOV tokens
layer <- layer_integer_lookup(vocabulary = vocab,
                              output_mode = 'multi_hot')
layer(data)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[0 1 0 1 1]
##  [1 0 1 0 1]], shape=(2, 5), dtype=int64)

}\if{html}{\out{</div>}}

\strong{Token count output}

Configure the layer with \code{output_mode='count'}. As with multi_hot output,
the first \code{num_oov_indices} dimensions in the output represent OOV tokens.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(12, 36, 1138, 42) |> as.integer()
data <- rbind(c(12, 1138, 42, 42),
              c(42,    7, 36,  7)) |> op_array("int64")
layer <- layer_integer_lookup(vocabulary = vocab,
                              output_mode = 'count')
layer(data)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[0 1 0 1 2]
##  [2 0 1 0 1]], shape=(2, 5), dtype=int64)

}\if{html}{\out{</div>}}

\strong{TF-IDF output}

Configure the layer with \code{output_mode='tf_idf'}. As with multi_hot output,
the first \code{num_oov_indices} dimensions in the output represent OOV tokens.

Each token bin will output \code{token_count * idf_weight}, where the idf weights
are the inverse document frequency weights per token. These should be
provided along with the vocabulary. Note that the \code{idf_weight} for OOV
tokens will default to the average of all idf weights passed in.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(12, 36, 1138, 42) |> as.integer()
idf_weights <- c(0.25, 0.75, 0.6, 0.4)
data <- rbind(c(12, 1138, 42, 42),
              c(42,    7, 36,  7)) |> op_array("int64")
layer <- layer_integer_lookup(output_mode = 'tf_idf',
                              vocabulary = vocab,
                              idf_weights = idf_weights)
layer(data)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[0.   0.25 0.   0.6  0.8 ]
##  [1.   0.   0.75 0.   0.4 ]], shape=(2, 5), dtype=float32)

}\if{html}{\out{</div>}}

To specify the idf weights for oov tokens, you will need to pass the entire
vocabulary including the leading oov token.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(-1, 12, 36, 1138, 42) |> as.integer()
idf_weights <- c(0.9, 0.25, 0.75, 0.6, 0.4)
data <- rbind(c(12, 1138, 42, 42),
              c(42,    7, 36,  7)) |> op_array("int64")
layer <- layer_integer_lookup(output_mode = 'tf_idf',
                              vocabulary = vocab,
                              idf_weights = idf_weights)
layer(data)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[0.   0.25 0.   0.6  0.8 ]
##  [1.8  0.   0.75 0.   0.4 ]], shape=(2, 5), dtype=float32)

}\if{html}{\out{</div>}}

When adapting the layer in \code{"tf_idf"} mode, each input sample will
be considered a document, and IDF weight per token will be
calculated as:
\code{log(1 + num_documents / (1 + token_document_count))}.

\strong{Inverse lookup}

This example demonstrates how to map indices to tokens using this layer.
(You can also use \code{adapt()} with \code{inverse = TRUE}, but for simplicity we'll
pass the vocab in this example.)

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(12, 36, 1138, 42) |> as.integer()
data <- op_array(c(1, 3, 4,
                  4, 0, 2)) |> op_reshape(c(2,-1)) |> op_cast("int32")
layer <- layer_integer_lookup(vocabulary = vocab, invert = TRUE)
layer(data)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[  12 1138   42]
##  [  42   -1   36]], shape=(2, 3), dtype=int64)

}\if{html}{\out{</div>}}

Note that the first index correspond to the oov token by default.

\strong{Forward and inverse lookup pairs}

This example demonstrates how to use the vocabulary of a standard lookup
layer to create an inverse lookup layer.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vocab <- c(12, 36, 1138, 42) |> as.integer()
data <- op_array(rbind(c(12, 1138, 42), c(42, 1000, 36)), "int32")
layer <- layer_integer_lookup(vocabulary = vocab)
i_layer <- layer_integer_lookup(vocabulary = get_vocabulary(layer),
                                invert = TRUE)
int_data <- layer(data)
i_layer(int_data)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[  12 1138   42]
##  [  42   -1   36]], shape=(2, 3), dtype=int64)

}\if{html}{\out{</div>}}

In this example, the input token 1000 resulted in an output of -1, since
1000 was not in the vocabulary - it got represented as an OOV, and all OOV
tokens are returned as -1 in the inverse layer. Also, note that for the
inverse to work, you must have already set the forward layer vocabulary
either directly or via \code{adapt()} before calling \code{get_vocabulary()}.
}

\seealso{
\itemize{
\item \url{https://keras.io/api/layers/preprocessing_layers/categorical/integer_lookup#integerlookup-class}
}

Other categorical features preprocessing layers: \cr
\code{\link{layer_category_encoding}()} \cr
\code{\link{layer_hashed_crossing}()} \cr
\code{\link{layer_hashing}()} \cr
\code{\link{layer_string_lookup}()} \cr

Other preprocessing layers: \cr
\code{\link{layer_auto_contrast}()} \cr
\code{\link{layer_category_encoding}()} \cr
\code{\link{layer_center_crop}()} \cr
\code{\link{layer_discretization}()} \cr
\code{\link{layer_equalization}()} \cr
\code{\link{layer_feature_space}()} \cr
\code{\link{layer_hashed_crossing}()} \cr
\code{\link{layer_hashing}()} \cr
\code{\link{layer_max_num_bounding_boxes}()} \cr
\code{\link{layer_mel_spectrogram}()} \cr
\code{\link{layer_mix_up}()} \cr
\code{\link{layer_normalization}()} \cr
\code{\link{layer_rand_augment}()} \cr
\code{\link{layer_random_brightness}()} \cr
\code{\link{layer_random_color_degeneration}()} \cr
\code{\link{layer_random_color_jitter}()} \cr
\code{\link{layer_random_contrast}()} \cr
\code{\link{layer_random_crop}()} \cr
\code{\link{layer_random_flip}()} \cr
\code{\link{layer_random_grayscale}()} \cr
\code{\link{layer_random_hue}()} \cr
\code{\link{layer_random_posterization}()} \cr
\code{\link{layer_random_rotation}()} \cr
\code{\link{layer_random_saturation}()} \cr
\code{\link{layer_random_sharpness}()} \cr
\code{\link{layer_random_shear}()} \cr
\code{\link{layer_random_translation}()} \cr
\code{\link{layer_random_zoom}()} \cr
\code{\link{layer_rescaling}()} \cr
\code{\link{layer_resizing}()} \cr
\code{\link{layer_solarization}()} \cr
\code{\link{layer_stft_spectrogram}()} \cr
\code{\link{layer_string_lookup}()} \cr
\code{\link{layer_text_vectorization}()} \cr

Other layers: \cr
\code{\link{Layer}()} \cr
\code{\link{layer_activation}()} \cr
\code{\link{layer_activation_elu}()} \cr
\code{\link{layer_activation_leaky_relu}()} \cr
\code{\link{layer_activation_parametric_relu}()} \cr
\code{\link{layer_activation_relu}()} \cr
\code{\link{layer_activation_softmax}()} \cr
\code{\link{layer_activity_regularization}()} \cr
\code{\link{layer_add}()} \cr
\code{\link{layer_additive_attention}()} \cr
\code{\link{layer_alpha_dropout}()} \cr
\code{\link{layer_attention}()} \cr
\code{\link{layer_auto_contrast}()} \cr
\code{\link{layer_average}()} \cr
\code{\link{layer_average_pooling_1d}()} \cr
\code{\link{layer_average_pooling_2d}()} \cr
\code{\link{layer_average_pooling_3d}()} \cr
\code{\link{layer_batch_normalization}()} \cr
\code{\link{layer_bidirectional}()} \cr
\code{\link{layer_category_encoding}()} \cr
\code{\link{layer_center_crop}()} \cr
\code{\link{layer_concatenate}()} \cr
\code{\link{layer_conv_1d}()} \cr
\code{\link{layer_conv_1d_transpose}()} \cr
\code{\link{layer_conv_2d}()} \cr
\code{\link{layer_conv_2d_transpose}()} \cr
\code{\link{layer_conv_3d}()} \cr
\code{\link{layer_conv_3d_transpose}()} \cr
\code{\link{layer_conv_lstm_1d}()} \cr
\code{\link{layer_conv_lstm_2d}()} \cr
\code{\link{layer_conv_lstm_3d}()} \cr
\code{\link{layer_cropping_1d}()} \cr
\code{\link{layer_cropping_2d}()} \cr
\code{\link{layer_cropping_3d}()} \cr
\code{\link{layer_dense}()} \cr
\code{\link{layer_depthwise_conv_1d}()} \cr
\code{\link{layer_depthwise_conv_2d}()} \cr
\code{\link{layer_discretization}()} \cr
\code{\link{layer_dot}()} \cr
\code{\link{layer_dropout}()} \cr
\code{\link{layer_einsum_dense}()} \cr
\code{\link{layer_embedding}()} \cr
\code{\link{layer_equalization}()} \cr
\code{\link{layer_feature_space}()} \cr
\code{\link{layer_flatten}()} \cr
\code{\link{layer_flax_module_wrapper}()} \cr
\code{\link{layer_gaussian_dropout}()} \cr
\code{\link{layer_gaussian_noise}()} \cr
\code{\link{layer_global_average_pooling_1d}()} \cr
\code{\link{layer_global_average_pooling_2d}()} \cr
\code{\link{layer_global_average_pooling_3d}()} \cr
\code{\link{layer_global_max_pooling_1d}()} \cr
\code{\link{layer_global_max_pooling_2d}()} \cr
\code{\link{layer_global_max_pooling_3d}()} \cr
\code{\link{layer_group_normalization}()} \cr
\code{\link{layer_group_query_attention}()} \cr
\code{\link{layer_gru}()} \cr
\code{\link{layer_hashed_crossing}()} \cr
\code{\link{layer_hashing}()} \cr
\code{\link{layer_identity}()} \cr
\code{\link{layer_jax_model_wrapper}()} \cr
\code{\link{layer_lambda}()} \cr
\code{\link{layer_layer_normalization}()} \cr
\code{\link{layer_lstm}()} \cr
\code{\link{layer_masking}()} \cr
\code{\link{layer_max_num_bounding_boxes}()} \cr
\code{\link{layer_max_pooling_1d}()} \cr
\code{\link{layer_max_pooling_2d}()} \cr
\code{\link{layer_max_pooling_3d}()} \cr
\code{\link{layer_maximum}()} \cr
\code{\link{layer_mel_spectrogram}()} \cr
\code{\link{layer_minimum}()} \cr
\code{\link{layer_mix_up}()} \cr
\code{\link{layer_multi_head_attention}()} \cr
\code{\link{layer_multiply}()} \cr
\code{\link{layer_normalization}()} \cr
\code{\link{layer_permute}()} \cr
\code{\link{layer_rand_augment}()} \cr
\code{\link{layer_random_brightness}()} \cr
\code{\link{layer_random_color_degeneration}()} \cr
\code{\link{layer_random_color_jitter}()} \cr
\code{\link{layer_random_contrast}()} \cr
\code{\link{layer_random_crop}()} \cr
\code{\link{layer_random_flip}()} \cr
\code{\link{layer_random_grayscale}()} \cr
\code{\link{layer_random_hue}()} \cr
\code{\link{layer_random_posterization}()} \cr
\code{\link{layer_random_rotation}()} \cr
\code{\link{layer_random_saturation}()} \cr
\code{\link{layer_random_sharpness}()} \cr
\code{\link{layer_random_shear}()} \cr
\code{\link{layer_random_translation}()} \cr
\code{\link{layer_random_zoom}()} \cr
\code{\link{layer_repeat_vector}()} \cr
\code{\link{layer_rescaling}()} \cr
\code{\link{layer_reshape}()} \cr
\code{\link{layer_resizing}()} \cr
\code{\link{layer_rnn}()} \cr
\code{\link{layer_separable_conv_1d}()} \cr
\code{\link{layer_separable_conv_2d}()} \cr
\code{\link{layer_simple_rnn}()} \cr
\code{\link{layer_solarization}()} \cr
\code{\link{layer_spatial_dropout_1d}()} \cr
\code{\link{layer_spatial_dropout_2d}()} \cr
\code{\link{layer_spatial_dropout_3d}()} \cr
\code{\link{layer_spectral_normalization}()} \cr
\code{\link{layer_stft_spectrogram}()} \cr
\code{\link{layer_string_lookup}()} \cr
\code{\link{layer_subtract}()} \cr
\code{\link{layer_text_vectorization}()} \cr
\code{\link{layer_tfsm}()} \cr
\code{\link{layer_time_distributed}()} \cr
\code{\link{layer_torch_module_wrapper}()} \cr
\code{\link{layer_unit_normalization}()} \cr
\code{\link{layer_upsampling_1d}()} \cr
\code{\link{layer_upsampling_2d}()} \cr
\code{\link{layer_upsampling_3d}()} \cr
\code{\link{layer_zero_padding_1d}()} \cr
\code{\link{layer_zero_padding_2d}()} \cr
\code{\link{layer_zero_padding_3d}()} \cr
\code{\link{rnn_cell_gru}()} \cr
\code{\link{rnn_cell_lstm}()} \cr
\code{\link{rnn_cell_simple}()} \cr
\code{\link{rnn_cells_stack}()} \cr
}
\concept{categorical features preprocessing layers}
\concept{layers}
\concept{preprocessing layers}
