% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/varma.R
\name{search.varma}
\alias{search.varma}
\title{Search for Best VARMA Models}
\usage{
search.varma(
  y,
  x = NULL,
  numTargets = 1,
  ySizes = NULL,
  yPartitions = NULL,
  xGroups = NULL,
  maxParams = c(1, 0, 0, 0, 0, 0),
  seasonsCount = 0,
  maxHorizon = 0,
  newX = NULL,
  simUsePreviousEstim = TRUE,
  olsStdMultiplier = 2,
  lbfgsOptions = get.options.lbfgs(),
  metricOptions = get.options.metric(),
  modelCheckItems = get.items.modelcheck(),
  searchItems = get.items.search(),
  searchOptions = get.options.search()
)
}
\arguments{
\item{y}{A matrix of endogenous data with variables in the columns.}

\item{x}{A matrix of exogenous data with variables in the columns. It can be \code{NULL}.}

\item{numTargets}{An integer representing the number of target variables.
For example, if 2, the first two variables in the first columns of \code{y} will be targets.
Information is saved just for the target variables.
It must be positive and cannot be larger than the number of endogenous variables.}

\item{ySizes}{An integer vector specifying the number of endogenous variables in the regressions.
For example, \code{c(1,2)} means the model set contains all regressions with 1 and 2 endogenous variables.
If \code{NULL}, \code{c(1)} is used.}

\item{yPartitions}{A list of integer vectors that partition the indexes of the endogenous variables.
No regression is estimated with two variables in the same partition.
If \code{NULL}, each variable is placed in its own partition, and the size of the model set is maximized.}

\item{xGroups}{A list of integer vectors that determine different combinations of the indexes of the exogenous variables to be used as exogenous variables in the SUR regressions.}

\item{maxParams}{An integer vector that determines the maximum values for the parameters of the VARMA model: \code{(p,d,q,P,D,Q)}. If \code{NULL}, \code{c(2,0,0,0,0,0)} is used.}

\item{seasonsCount}{An integer value representing the number of observations per unit of time.}

\item{maxHorizon}{An integer value representing the maximum value for the prediction horizon if \code{type1} is \code{TRUE} in the \code{modelCheckItems} argument. Also, it is used as the maximum prediction horizon in checking predictions.}

\item{newX}{A matrix of new exogenous data for out-of-sample prediction. It must have the same number of columns as the \code{x} argument.}

\item{simUsePreviousEstim}{If \code{TRUE}, parameters are initialized only in the first step of the simulation. The initial values of the n-th simulation (with one more observation) are the estimations from the previous step.}

\item{olsStdMultiplier}{A number used as a multiplier for the standard deviation of OLS, used for restricting maximum likelihood estimation.}

\item{lbfgsOptions}{A list containing L-BFGS optimization options.
Use \link{get.options.lbfgs} function for initialization.}

\item{metricOptions}{A list of options for measuring performance.
Use \link{get.options.metric} function to get them.}

\item{modelCheckItems}{A list of options for excluding a subset of the model set.
See and use \link{get.items.modelcheck} function to get them.}

\item{searchItems}{A list of options for specifying the purpose of the search.
See and use \link{get.items.search} function to get them.}

\item{searchOptions}{A list of extra options for performing the search.
See and use \link{get.options.search} function to get them.}
}
\value{
A nested list with the following members:
\item{counts}{Information about the expected number of models, number of estimated models, failed estimations, and some details about the failures.}
\item{...}{Results reported separately for each metric, then for each target variable, then for each requested type of output. This part of the output is highly nested and items are reported based on the arguments of the search.}
\item{info}{General information about the search process, some arguments, elapsed time, etc.}

Note that the output does not contain any estimation results,
but minimum required data to estimate the models (Use \code{summary()} function to get the estimation).
}
\description{
Use this function to create a Vector Autoregressive Moving Average model set and search for the best models (and other information) based on in-sample and out-of-sample evaluation metrics.
}
\examples{
# We simulate some data for this example:

# sample data:
n = 50 # number of observations
num_x_r <- 3L # number of relevant explanatory variables
num_x_ir <-
  20 # (relatively large) number of irrelevant explanatory variables
set.seed(340)
sample <- sim.bin(num_x_r, n)
x_ir <- lapply(1:num_x_ir, function(x) rnorm(n))

# prepare data:
data <- data.frame(sample$y, sample$x, x_ir)
colnames(data) <- c("Y", colnames(sample$x), paste0("z", 1:num_x_ir))

# Use glm function to estimate and analyse:
model1 <- glm(Y ~ . - Y, data = data, family = binomial())
summary(model1)

# You can also use this package estimation function:
model2 <- estim.bin(sample$y, data[, 3:ncol(data), drop = FALSE])
# format and print coefficients:
coefs2 <- data.frame(model2$estimations[1:4])
colnames(coefs2) <- names(model2$estimations)[1:4]
print(coefs2)

# Alternatively, You can define a search process:
x_sizes = c(1:4) # assuming we know the number of relevant explanatory variables is less than 4
metric_options <-
  get.options.metric(typesIn = c("sic")) # We use SIC for searching
search_res <- search.bin(sample$y, data[, 3:ncol(data)],
                        xSizes = x_sizes, metricOptions = metric_options)
print(search_res$sic$target1$model$bests$best1$exoIndices) # print best model's explanatory indexes

# Use summary function to estimate the best model:
search_sum <-
  summary(search_res, y = sample$y, x = data[, 3:ncol(data)])
# format and print coefficients:
model3 <- search_sum$sic$target1$model$bests$best1
coefs3 <- data.frame(model3$estimations[1:4])
colnames(coefs3) <- names(model3$estimations)[1:4]
print(coefs3)

# Try a step-wise search (you can estimate larger models, faster):
x_sizes_steps = list(c(1, 2, 3), c(4), c(5))
counts_steps = c(NA, 10, 9)
search_items <- get.items.search(bestK = 10)
search_step_res <- search.bin.stepwise(
  sample$y,
  data[, 3:ncol(data)],
  xSizeSteps = x_sizes_steps,
  countSteps = counts_steps,
  metricOptions = metric_options,
  searchItems = search_items
)
print(search_step_res$sic$target1$model$bests$best1$exoIndices)
# Use summary like before.
}
\seealso{
\link{estim.varma}, \link{search.varma.stepwise}
}
