% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bin.R
\name{estim.bin}
\alias{estim.bin}
\title{Estimate a Discrete Choice Model}
\usage{
estim.bin(
  y,
  x,
  w = NULL,
  linkFunc = c("logit", "probit"),
  newX = NULL,
  pcaOptionsX = NULL,
  costMatrices = NULL,
  aucOptions = get.options.roc(),
  simFixSize = 0,
  simTrainFixSize = 0,
  simTrainRatio = 0.75,
  simSeed = 0,
  weightedEval = FALSE,
  printMsg = FALSE
)
}
\arguments{
\item{y}{A matrix of endogenous data with variable in the column.}

\item{x}{A matrix of exogenous data with variables in the columns.}

\item{w}{Weights of the observations in \code{y}.
\code{NULL} means equal weights for all observations.}

\item{linkFunc}{A character string that shows the probability assumption. It can be \code{logit} or \code{probit}.}

\item{newX}{A numeric matrix where for each row in it, probabilities are projected and reported. It can be \code{NULL}.}

\item{pcaOptionsX}{A list of options to use principal components of the \code{x}, instead of the actual values. Set \code{NULL} to disable. Use \code{\link[=get.options.pca]{get.options.pca()}} for initialization.}

\item{costMatrices}{A list of numeric matrices where each one determines how to score the calculated probabilities. See and use \link{search.bin} for more information and initialization.}

\item{aucOptions}{A list of options for AUC calculation. See and use \code{[get.options.roc()]} for more information and initialization.}

\item{simFixSize}{An integer that determines the number of out-of-sample simulations. Use zero to disable the simulation.}

\item{simTrainFixSize}{An integer representing the number of data points in the training sample in the out-of-sample simulation. If zero, \code{trainRatio} will be used.}

\item{simTrainRatio}{A number representing the size of the training sample relative to the available size, in the out-of-sample simulation. It is effective if \code{trainFixSize} is zero.}

\item{simSeed}{A seed for the random number generator. Use zero for a random value.}

\item{weightedEval}{If \code{TRUE}, weights will be used in evaluations.}

\item{printMsg}{Set to \code{TRUE} to enable printing some details.}
}
\value{
A nested list with the following items:
\item{counts}{Information about different aspects of the estimation such as the number of observation, number of exogenous variables, etc.}
\item{estimations}{Estimated coefficients, standard errors, z-statistics, p-values, etc.}
\item{metrics}{Value of different goodness of fit and out-of-sample performance metrics.}
\item{projections}{Information on the projected values, if \code{newX} is provided.}
\item{info}{Some other general information.}
}
\description{
Use this function to estimate a discrete choice model.
}
\details{
Binary regression is a statistical technique used to estimate the probability of one of two possible outcomes, represented by a binary dependent variable that takes on two values, such as 0 and 1.
This is achieved by modeling the relationship between one or more independent variables and the binary dependent variable.
The most commonly used binary regression models are the logit model, also known as logistic regression, and the probit model, also known as probit regression.
In general, a binary regression model can be written as \eqn{f(p) = \beta_0 + \beta_1x_1 + \ldots + \beta_kx_k}, where \eqn{p} is the probability that \eqn{y} is 1 and \eqn{f} is the link function.
For logistic regression, the logit function is used as the link function: \eqn{f(p) = \ln{\frac{p}{1-p}}}.
For probit regression, the probit function is used as the link function: \eqn{f(p) = \Phi^{-1}(p)}, where \eqn{\Phi^{-1}} is the inverse cumulative distribution function of the standard normal distribution.
The parameters of the binary regression model are estimated using maximum likelihood estimation.

Note that the focus in \code{ldt} is model uncertainty and the main purpose of exporting this method is to show the inner calculations of the search process in \link{search.bin} function.
}
\examples{
# Example 1 (simulation, small model):
sample <- sim.bin(3L, 100)
data <- data.frame(sample$y, sample$x)
#   Estimate using glm
fit <- glm(Y ~ X1 + X2, data = data, family = binomial())
#   Estimate using estim.bin in this package
fit1 <- estim.bin(sample$y, sample$x[,2:ncol(sample$x), drop=FALSE], linkFunc = "logit",
                 newX = sample$x[,2:3, drop=FALSE])

# Example 2 (simulation, large model with PCA analysis):
sample <- sim.bin(30L, 100, probit = TRUE)
data <- data.frame(sample$y, sample$x)
colnames(data) <- c(colnames(sample$y),colnames(sample$x))
pca_options <- get.options.pca(ignoreFirst = 1, exactCount = 3)
fit1 <- estim.bin(sample$y, sample$x[,2:ncol(sample$x), drop=FALSE], linkFunc = "probit",
                 pcaOptionsX = pca_options)

}
\seealso{
\link{search.bin}, \link{search.bin.stepwise}
}
