% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/communities.R
\name{find_partition_with_rep}
\alias{find_partition_with_rep}
\title{Finds the optimal partition using the Leiden algorithm with replicate starts}
\usage{
find_partition_with_rep(
  graph,
  edge_weights,
  resolution = 1,
  niter = 2,
  nrep = 10
)
}
\arguments{
\item{graph}{The igraph graph to define the partition on}

\item{edge_weights}{Vector of edge weights. In weighted graphs, a real number is assigned to each (directed or undirected) edge. For an unweighted graph, this is set to 1. Refer to igraph, weighted graphs.}

\item{resolution}{Numeric scalar, resolution parameter controlling communities detected (default=1.0) Higher resolutions lead to more communities, while lower resolutions lead to fewer communities.}

\item{niter}{Number of iterations that the algorithm should be run for (default=2)}

\item{nrep}{Number of replicate starts with random number being updated. (default=10) The result with the best quality will be returned.}
}
\value{
A vector of membership values
}
\description{
This function performs Leiden algorithm \code{nrep} times and returns the
result from the run with the maximum quality.

Since Leiden algorithm has stochastic process, repeating stochastically may
improve the result. However, users should be aware of whether there is indeed
a community structure with exploration, rather than blindly trusting the
returned result that comes with the highest quality value.

The random number generator (RNG) is not re-seeded at each new start of
community detection, in order to keep the independence of each replicate. To
get reproducible result, users can run \code{set.seed()} before calling these
functions.

\code{\link{find_partition}} only performs the community detection once and
the reproducibility can also be ensured with \code{set.seed()}.
}
\examples{
library(igraph)

# To run 10 replicates and get the partitioning with the highest quality
membership <- find_partition_with_rep(exampleGraph, E(exampleGraph)$weight, nrep = 10)

# To get reprodicible result for every function call, do `set.seed()` right before calling
set.seed(233)
res1 <- find_partition_with_rep(exampleGraph, E(exampleGraph)$weight, resolution = 2)
# Here, no seed was set...
res2 <- find_partition_with_rep(exampleGraph, E(exampleGraph)$weight, resolution = 2)
set.seed(233)
res3 <- find_partition_with_rep(exampleGraph, E(exampleGraph)$weight, resolution = 2)
identical(res1, res2) # FALSE (usually), as no seed as set
identical(res1, res3) # TRUE (always), as set.seed() was used directly before the function call

}
