\name{PieChart}
\alias{PieChart}
\alias{pc}

\title{Pie Chart}

\description{
Abbreviation: \code{pc}

Plots a pie chart of a categorical variable (\code{x}). The default chart is a doughnut or ring version of a pie chart, that is, a hole in the middle of the pie. Either directly enter the corresponding numerical value (\code{y}) or have the numerical variable be the tabulated counts for the frequency of occurrence for each value of the categorical variable. Also displays the frequency table for the variable with the corresponding chi-square inferential analysis. Real numbers can also be entered directly.
}

\usage{
PieChart(x, y=NULL, data=mydata,

         radius=1, hole=0.65, hole.fill=getOption("panel.fill"),

         fill=NULL, 
         color="lightgray",
         trans=getOption("trans.bar.fill"),

         density=NULL, angle=45,
         lty="solid", lwd=1, edges=200,

         clockwise=FALSE, init.angle=ifelse (clockwise, 90, 0), 

         values=getOption("values"),
         values.color=getOption("values.color"), 
	 values.cex=getOption("values.cex"),
         values.digits=getOption("values.digits"),
         values.pos=getOption("values.pos"),

         main=NULL, main.cex=1.2, labels.cex=0.9, cex,

         add=NULL, x1=NULL, y1=NULL, x2=NULL, y2=NULL,

         eval.df=NULL, quiet=getOption("quiet"),
         width=6.5, height=6, pdf.file=NULL,
         \dots)

pc(\ldots)
}

\arguments{
  \item{x}{For each level of this categorical variable, x, display the
        frequencies as slices of a pie.}
  \item{y}{Numeric variable that sets the area of each slice of the pie.
        If not specified,
        then its value is the frequency of each category of \code{x},
        automatically tabulated.}
  \item{data}{Optional data frame that contains the variable(s) of interest,
        default is \code{mydata}.}\cr

  \item{radius}{The pie is drawn in a box with sides that range from -1 to 1,
        so the maximum value of the radius without truncating the pie is 1.}
  \item{hole}{The proportion of the radius that defines the inner hole
        for what is called a doughnut or hole plot. To show the full pie,
        set to \code{FALSE} or the value of \code{0}.}
  \item{hole.fill}{Fill color of the hole, which by default is the same color as
       \code{panel.fill} as set by the color theme or individually with
       the \code{\link{style}} function.} \cr


  \item{fill}{Specified color of each slice. Default is the discrete 
        scale with, with fixed
        chroma (50) and luminance (75) for unbiased comparison 
        across colors, for all color themes except 
        \code{"gray"} and \code{"white"}, with default gray scale.
        Can explicitly choose \code{"grays"}
        or \code{"colors"} , or pre-specified R color schemes 
        \code{"rainbow"}, \code{"terrain"}, and \code{"heat"}. Can also
        provide pre-defined color ranges \code{"blues"}, \code{"reds"}
        and \code{"greens"}, as well as
        custom colors, such as generated by \code{\link{getColors}}.}
  \item{color}{Border color of sides and the pie, can be a vector
        to customize the color for each slice. Default is
        \code{bar.color} from the lessR \code{\link{style}} function.}
  \item{trans}{Transparency factor of the area of each slice.
        Default is
        \code{trans.bar.fill} from the lessR \code{\link{style}} function.}\cr
  

  \item{density}{Density of shading lines, in lines per inch. Default value is
       \code{NULL}, that is, no shading lines.}
  \item{angle}{Angle of shading lines in degrees.}

  \item{lty}{Type of line that borders each slice, such as "solid", the default.
        Can be a vector. Acceptable values are \code{"blank"}, \code{"solid"},
        \code{"dashed"}, \code{"dotted"}, \code{"dotdash"}, and \code{"longdash"}.}
  \item{lwd}{Width of line that borders each slice.}
  \item{edges}{Approximation of a circle with a polygon drawn with the 
       number of specified edges.}\cr



  \item{clockwise}{Default value of \code{FALSE} specifies to draw the
        slices counter-clockwise, otherwise clockwise.}
  \item{init.angle}{Starting angle (in degrees) for the slices. For counter-clockwise
       the default value is 0 (3 o'clock), otherwise 90 (12 o'clock).}\cr


  \item{values}{If not the default value of \code{"off"}, adds the
        numerical results to the plot according to \code{"\%"}, 
        \code{"prop"} or \code{"input"}, that is, percentages,
         proportions, or the value from which the slices are plotted,
         such as tabulated counts if \code{y} is not specified,
         or the value of \code{y} if the plotted values are provided. If
         any other \code{values} parameter is specified, default is set
         to \code{"\%"}.}
  \item{values.color}{Color of the plotted text. Could be a vector to specify
        a unique color for each value. If fewer colors are specified than
        the number of categories, the colors are recycled.}
  \item{values.cex}{Character expansion factor, the size, of the plotted text, 
        for which the default value is 0.95.}
  \item{values.digits}{Number of decimal digits for which to display the values.
        Default is 0, round to the nearest integer, for \code{"\%"}
        and 2 for \code{"prop"}.}
  \item{values.pos}{Position of the plotted text. Default is inside the pie,
        or, if \code{"label"}, as part of the label for each value
        outside of the pie.}\cr


  \item{main}{Title of graph. Set the color with \code{main.color} with
       the \code{\link{style}} function.}
  \item{main.cex}{Character expansion factor of title relative to 1.}\cr
  \item{labels.cex}{Character expansion factor of labels relative to 1.}
  \item{cex}{General character expansion factor for default values of 
        main.cex, labels.cex, and values.cex. Useful for adjustment of text
        for larger or smaller images.}


  \item{add}{\bold{Draw one or more objects}, text or a geometric figures,
       on the plot.
       Possible values are any text to be written, the first argument, which is
       \code{"text"}, or, to indicate a figure,
       \code{"rect"} (rectangle), \code{"line"}, \code{"arrow"},
       \code{"v.line"} (vertical line), and \code{"h.line"} (horizontal line).
       The value \code{"means"} is short-hand for vertical and horizontal lines
       at the respective means. Does not apply to Trellis graphics.
       Customize with parameters such as \code{fill} and \code{color}
       from the \code{\link{style}} function.}
  \item{x1}{First x coordinate to be considered for each object.
           All coordinates vary from -1 to 1.}
  \item{y1}{First y coordinate to be considered for each object.}
  \item{x2}{Second x coordinate to be considered for each object.
        Only used for \code{"rect"}, \code{"line"} and \code{arrow}.}
  \item{y2}{Second y coordinate to be considered for each object.
        Only used for \code{"rect"}, \code{"line"} and \code{arrow}.}\cr

  \item{eval.df}{Determines if to check for existing data frame and
        specified variables. By default is \code{TRUE} 
        unless the \code{shiny} package is loaded then set to \code{FALSE} so
        that Shiny will run. Needs to be set to \code{FALSE} if using
        the pipe \code{\%\>\%} notation.}
  \item{quiet}{If set to \code{TRUE}, no text output. Can change system default
       with \code{\link{style}} function.}
  \item{width}{Width of the plot window in inches, defaults to 4.5.}
  \item{height}{Height of the plot window in inches, defaults to 4.5.}
  \item{pdf.file}{Name of the pdf file to if graphics to be redirected to a
        pdf file.}\cr


  \item{...}{Other parameter values for graphics as defined processed 
      by \code{\link{pie}} and \code{\link{par}} for general graphics,
      which includes \code{radius} of the pie, and \code{color.main} for
      the title of the graph.}
}


\details{
OVERVIEW\cr
Plot a pie chart with default colors, presumably with a relatively small number of values for each variable.  By default, colors are selected for the slices, background and grid lines, all of which can be customized. The basic computations of the chart are provided with the standard R functions \code{\link{pie}} and \code{\link{chisq.test}} and the \code{lessR} function  \code{\link{chisq.test}}. A minor modification of the original \code{\link{pie}} code provides for the hole in the middle of the pie, the default doughnut or ring chart.

COLORS\cr
1. For gray scale, such as for the default \code{lightbronze} color theme as well as the \code{gray} and \code{white} color themes, colors are automatically generated from the lightest to the darkest displayed levels of gray. For other color themes a range of up to 8 hues are displayed for the slices based on the HCL color space with a chroma of 50 and a luminance of 70, so as to have the colors at equal perceived brightness. These color schemes can also be manually requested regardless of the current color theme. Three more built-in R color palettes are also available by setting \code{color} to one of \code{"rainbow"}, \code{"heat"} and \code{"terrain"}.  The most vivid of all the palettes is \code{"rainbow"}.\cr
2. The desired colors can be explicitly specified with the \code{fill} option, which overrides any other color options. When plotting one variable, include one color in this color list, the color used for all of the slices  If the list includes more than once color, the \code{c} function must be used to generate the list, as in\cr
\code{fill=c("coral3","seagreen3")}.

When plotting an ordered factor then neither of the two standard color palettes are used.  Instead, the resulting slice colors for each level of the ordered factor are also ordered in a progression of colors. The default progression is based on the first color of either the regular, vivid or gray color palettes, but this can be changed with the \code{fill} and \code{end.fill} options, or individually specify the color of each piece with the \code{fill} option. A specified palette can, for example, be from light to dark of the same hue, or from a light color of one hue to a dark color of another hue. Each color value can be specified with a color name, or with a specification with the \code{\link{rgb}} function.  See the examples below.  

Use the \code{\link{showColors}} function in this package to get, for each color:  name,  sample color swatch, and corresponding rgb specification.  For a very small number of levels, such as two, it is may be desirable to specify the low and high values to not be closer to each other than the default values. Or use \code{\link{getColors}} to get color palates of colors in the HCL color space to maintain equal perceived brightness across the colors.

To change the background color, set the \code{"panel.fill"} argument of the \code{\link{style}} function. The hole of the pie defaults to that color, which, of course, can also be specified to a different color. 

ANNOTATIONS\cr
Use the \code{add} and related parameters to annotate the plot with text and/or geometric figures. Each object is placed according from one to four corresponding coordinates, the required coordinates to plot that object, as shown in the following table. The values of the coordinates vary from -1 to 1.\cr

\tabular{lll}{
Value \tab Object \tab Required Coordinates\cr
----------- \tab ------------------- \tab ----------------\cr
text \tab text \tab x1, x2\cr
\code{"rect"} \tab rectangle \tab x1, y1, x2, y2\cr
\code{"line"} \tab line segment \tab x1, y1, x2, y2\cr
\code{"arrow"} \tab arrow \tab x1, y1, x2, y2\cr
----------- \tab ------------------- \tab ----------------\cr
}

The value of \code{add} specifies the object. For a single object, enter a single value. Then specify the value of the needed corresponding coordinates, as specified in the above table. For multiple placements of that object, specify vectors of corresponding coordinates. To annotate multiple objects, specify multiple values for \code{add} as a vector. Then list the corresponding coordinates, for up to each of four coordinates, in the order of the objects listed in \code{add}. See the examples for illustrations.

Can also specify vectors of different properties, such as \code{add.color}. That is, different objects can be different colors, different transparency levels, etc.

STATISTICS\cr
In addition to the pie chart, descriptive and inferential statistics are presented.  First, for integer variables such as counts, the frequency table with proportions is displayed. Second, the corresponding chi-square test is also displayed. For real valued variables read from a data frame, the summary statistics such as the mean are reported.

PDF OUTPUT\cr
Because \code{lessR} functions generate their own graphics calls, the standard graphic output functions such as \code{\link{pdf}} do not work with the \code{lessR} graphics functions.  Instead, to obtain pdf output, use the \code{pdf.file} option, perhaps with the optional \code{width} and \code{height} options. These files are written to the default working directory, which can be explicitly specified with the R \code{\link{setwd}} function.

ONLY VARIABLES ARE REFERENCED\cr
The referenced variable in a \code{lessR} function can only be a variable name. This referenced variable must exist in either the referenced data frame, \code{mydata} by default, or in the user's workspace, more formally called the global environment. That is, expressions cannot be directly evaluated. For example:

\code{    > PieChart(rnorm(10))   # does NOT work}

Instead, do the following:
\preformatted{    > Y <- rnorm(10)   # create vector Y in user workspace
    > PieChart(Y)     # directly reference Y}
}


\author{David W. Gerbing (Portland State University; \email{gerbing@pdx.edu})}

\seealso{
\code{\link{pie}}, \code{\link{chisq.test}}.
}


\examples{
# get the data from a file included with lessR
mydata <- rd("Employee", format="lessR")

# --------------------------------------------------------
# pie (doughnut) chart from the data for a single variable
# --------------------------------------------------------

# basic pie chart, actually a doughnut or ring chart
# with default hcl colors (except for themes "gray" and "white")
PieChart(Dept)
# short name
#pc(Dept)

# standard pie chart with no hole
pc(Dept, hole=0)

# specify a unique slice color for each of the two slices
# turn off borders
PieChart(Gender, fill=c("pink","lightblue"), lty="blank")

# use getColors function to create the pie slice colors
# here as a separate function call
# need to set the correct number of colors to span the full range
mycolors <- getColors(n=5, clr=getColors("aliceblue", end.clr="steelblue"))
PieChart(Dept, fill=mycolors)
# call getColors directly from PieChart
PieChart(Dept, fill=getColors(n=5, clr="aliceblue", end.clr="blue"))

# specify the colors from the R palette rainbow
PieChart(Dept, fill="rainbow")

# display the percentage inside each slice of the pie
# provide a unique color for each displayed value
PieChart(Dept, values="\%",
         values.color=c("yellow", "pink", "blue", "purple", "brown"))

# display the counts inside each slice of the pie
# reduce size of displayed counts to 0.75
PieChart(Dept, values="input", values.cex=0.75,
         values.color=getOption("window.fill"))

# add transparency and custom color for the displayed values
PieChart(Dept, trans=.6, values="\%", values.color=rgb(.3,.3,.3))


# ------------------------------
# pie chart directly from counts
# ------------------------------

# from vector
# pie chart of one variable with three levels
# enter counts as a vector with the combine function, c
# must supply the level names and variable name
# use abbreviation pc for PieChart
City <- c(206, 94, 382)
names(City) <- c("LA","Chicago","NY")
pc(City, main="Employees in Each City")

# counts from data frame
x <- c("ACCT", "ADMN", "FINC", "MKTG", "SALE")
y <- c(5, 6, 4, 6, 15)
mydata <- data.frame(x,y)
names(mydata) <- c("Dept", "Count")
PieChart(Dept, Count)

# real numbers from data frame 
Dept <- c("ACCT", "ADMN", "FINC", "MKTG", "SALE")
Salary <- c(86208.42, 29808.29, 42305.52, 75855.81, 65175.51)
mydata <- data.frame(x,y)
pc(Dept, Salary)
rm(Dept)
rm(Salary)

# -----------
# annotations
# -----------

mydata <- rd("Employee", format="lessR")

# Place a message in the center of the pie
# Use \n to indicate a new line
PieChart(Dept, add="Employees by\nDepartment", x1=0, y1=0)

# Use style to change some parameter values
style(add.trans=.8, add.fill="gold", add.color="gold4", add.lwd=0.5)
# Add a rectangle around the message centered at <0,0>
PieChart(Dept, add=c("rect", "Employees by\nDepartment"),
                     x1=c(-.4,0), y1=c(-.2, 0), x2=.4, y2=.2)

}

% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ pie chart }
\keyword{ color }

