testLingexpr <- function(d, ctx) {
    x <- d$x
    d$x <- NULL
    ling <- colnames(d)
    negs <- startsWith(ling, 'X.')
    ling[negs] <- substring(ling[negs], 3)
    hedges <- substr(ling, 1, 2)
    atomics <- substr(ling, 4, 5)
    empty <- atomics==''
    atomics[empty] <- hedges[empty]
    hedges[empty] <- '-'
    atomics[negs] <- paste0('neg.', atomics[negs])

    for (i in seq_len(ncol(d))) {
        fun <- lingexpr(ctx, atomic=atomics[i], hedge=hedges[i])
        res <- round(fun(x), 10)
        #cat(colnames(d)[i]); cat(':')
        expect_true(all(res==d[, i]))
    }
}


test_that('lingexpr ctx3', {
    d <- structure(list(ex.sm = c(1, 0.9573419744, 0.7656052715, 0.4449881174, 
        0.1901770105, 0.0420707222, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ex.bi = c(0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0.0252540428, 0.1520193674, 0.3854895105, 0.708762429, 0.9313249684, 
        1), si.sm = c(1, 0.9999600019, 0.9429557956, 0.7778252496, 0.5070595947, 
        0.2541304608, 0.0877023987, 0.0077754083, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), si.bi = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0.0021701389, 0.0647969149, 0.2139247626, 0.4495536821, 0.7318239796, 
        0.9185797936, 0.9972092678, 1), ve.sm = c(1, 1, 1, 0.9674969363, 
        0.8407352941, 0.6182322304, 0.3365502451, 0.1336520786, 0.0228129713, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0), ve.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0.011692237, 0.1041195324, 0.2886058871, 0.5622426471, 
        0.8038939951, 0.9498039216, 0.9999724265, 1, 1), sm = c(1, 1, 
        1, 1, 0.9992334588, 0.9687207762, 0.893703014, 0.7741801723, 
        0.610152251, 0.4269482967, 0.2754026957, 0.1569456539, 0.0715771712, 
        0.0192972477, 0.0001058834, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0), me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.003811802, 
        0.03623859, 0.1017539372, 0.2003578435, 0.332050309, 0.4968313337, 
        0.681104029, 0.8273299185, 0.9290507284, 0.9862664588, 1, 1, 
        1, 1, 1, 1, 1, 1, 0.9923689953, 0.9440542809, 0.8512344869, 0.7139096133, 
        0.5337581657, 0.3623594292, 0.2240492519, 0.1188276337, 0.0466945747, 
        0.0076500749, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), bi = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0128118901, 
        0.0584741018, 0.1372248726, 0.2490642026, 0.3939920918, 0.5720085401, 
        0.7449349944, 0.873358852, 0.9572776301, 0.9966913286, 1, 1, 
        1, 1), ml.sm = c(1, 1, 1, 1, 1, 1, 1, 0.9999949255, 0.9586349055, 
        0.8363671782, 0.6331917436, 0.3899306329, 0.2030414439, 0.0765950714, 
        0.0105915156, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
        ), ml.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0297398, 0.1199204824, 
        0.2705439815, 0.4816102972, 0.7241708423, 0.894983194, 0.9848878384, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9931598424, 0.9194367394, 
        0.7648059292, 0.5310766984, 0.3079218193, 0.1452097569, 0.0429405111, 
        0.001114082, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ml.bi = c(0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0046439425, 0.0585589349, 
        0.172916744, 0.3477173698, 0.5828477317, 0.8022047078, 0.9406539766, 
        0.9981955381, 1, 1, 1, 1, 1, 1, 1), ro.sm = c(1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 0.9957271055, 0.8906658984, 0.6448474471, 0.3233841384, 
        0.1073921471, 0.0075248823, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0), ro.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.033536821, 
        0.1798539764, 0.4422958583, 0.7600656969, 0.9495812506, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.970593492, 
        0.8092293872, 0.5087192019, 0.2230523747, 0.053510274, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0), ro.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0.0014863965, 0.078128716, 0.270895762, 
        0.5787928875, 0.8527627877, 0.9859754436, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1), qr.sm = c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9479152389, 
        0.7723251452, 0.4755209761, 0.2055921053, 0.0472040901, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), qr.me = c(0, 0, 0, 0, 0, 0, 
        0, 0, 0.0057108918, 0.0971743935, 0.3001787509, 0.6074826325, 
        0.8574332968, 0.9834496769, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9937808388, 0.8925513155, 
        0.6673875081, 0.3541645251, 0.1288519965, 0.0150803237, 0, 0, 
        0, 0, 0, 0, 0, 0), qr.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0.0289113898, 0.1649912338, 0.4126119335, 0.7223350123, 
        0.9227119629, 0.9991546294, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1), vr.sm = c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9993215643, 
        0.9701723968, 0.8980049773, 0.7828193058, 0.6246153824, 0.423393207, 
        0.1978223569, 0.0514140382, 7.60563e-05, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
        ), vr.me = c(0, 0, 0, 0.0092028086, 0.0985689252, 0.2830053787, 
        0.5090442674, 0.693059142, 0.8340557646, 0.9320341353, 0.986994254, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9928240875, 0.9464676192, 0.8570928989, 
        0.7246999267, 0.5492887025, 0.3308592263, 0.127850589, 0.019470405, 
        0, 0, 0), vr.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0335408148, 0.1609350662, 
        0.3779865817, 0.5878124075, 0.7546199813, 0.8784093032, 0.959180373, 
        0.996933191, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1), ty.me = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0.040469215, 0.3936011905, 0.9084472656, 0.9593098958, 0.5078241257, 
        0.08203125, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0), x = c(0, 0.01953125, 0.0390625, 0.05859375, 
        0.078125, 0.09765625, 0.1171875, 0.13671875, 0.15625, 0.17578125, 
        0.1953125, 0.21484375, 0.234375, 0.25390625, 0.2734375, 0.29296875, 
        0.3125, 0.33203125, 0.3515625, 0.37109375, 0.390625, 0.41015625, 
        0.4296875, 0.44921875, 0.46875, 0.48828125, 0.5078125, 0.52734375, 
        0.546875, 0.56640625, 0.5859375, 0.60546875, 0.625, 0.64453125, 
        0.6640625, 0.68359375, 0.703125, 0.72265625, 0.7421875, 0.76171875, 
        0.78125, 0.80078125, 0.8203125, 0.83984375, 0.859375, 0.87890625, 
        0.8984375, 0.91796875, 0.9375, 0.95703125, 0.9765625, 0.99609375
        )), .Names = c("ex.sm", "ex.bi", "si.sm", "si.bi", "ve.sm", "ve.bi", 
        "sm", "me", "bi", "ml.sm", "ml.me", "ml.bi", "ro.sm", "ro.me", 
        "ro.bi", "qr.sm", "qr.me", "qr.bi", "vr.sm", "vr.me", "vr.bi", 
        "ty.me", "x"), row.names = c(1L, 21L, 41L, 61L, 81L, 101L, 121L, 
        141L, 161L, 181L, 201L, 221L, 241L, 261L, 281L, 301L, 321L, 341L, 
        361L, 381L, 401L, 421L, 441L, 461L, 481L, 501L, 521L, 541L, 561L, 
        581L, 601L, 621L, 641L, 661L, 681L, 701L, 721L, 741L, 761L, 781L, 
        801L, 821L, 841L, 861L, 881L, 901L, 921L, 941L, 961L, 981L, 1001L, 
        1021L), class = "data.frame")

    testLingexpr(d, ctx3())
})


test_that('lingexpr ctx5', {
    d <- structure(list(ex.sm = c(1, 0.9573419744, 0.7656052715, 0.4449881174, 
        0.1901770105, 0.0420707222, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ex.bi = c(0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0.0252540428, 0.1520193674, 0.3854895105, 0.708762429, 0.9313249684, 
        1), si.sm = c(1, 0.9999600019, 0.9429557956, 0.7778252496, 0.5070595947, 
        0.2541304608, 0.0877023987, 0.0077754083, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), si.bi = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0.0021701389, 0.0647969149, 0.2139247626, 0.4495536821, 0.7318239796, 
        0.9185797936, 0.9972092678, 1), ve.sm = c(1, 1, 1, 0.9674969363, 
        0.8407352941, 0.6182322304, 0.3365502451, 0.1336520786, 0.0228129713, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0), ve.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0.011692237, 0.1041195324, 0.2886058871, 0.5622426471, 
        0.8038939951, 0.9498039216, 0.9999724265, 1, 1), lm = c(0, 0, 
        0, 0, 0, 0, 0.003811802, 0.1017539372, 0.332050309, 0.681104029, 
        0.9290507284, 1, 1, 1, 1, 0.9923689953, 0.8512344869, 0.5337581657, 
        0.2240492519, 0.0466945747, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
        ), um = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0271061477, 0.1779899775, 
        0.461228044, 0.801645147, 0.9783837191, 1, 1, 1, 1, 0.9572776301, 
        0.7449349944, 0.3939920918, 0.1372248726, 0.0128118901, 0, 0, 
        0, 0, 0, 0), sm = c(1, 1, 1, 1, 0.9992334588, 0.9687207762, 0.893703014, 
        0.7741801723, 0.610152251, 0.4269482967, 0.2754026957, 0.1569456539, 
        0.0715771712, 0.0192972477, 0.0001058834, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0), me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0.003811802, 0.03623859, 0.1017539372, 0.2003578435, 
        0.332050309, 0.4968313337, 0.681104029, 0.8273299185, 0.9290507284, 
        0.9862664588, 1, 1, 1, 1, 1, 1, 1, 1, 0.9923689953, 0.9440542809, 
        0.8512344869, 0.7139096133, 0.5337581657, 0.3623594292, 0.2240492519, 
        0.1188276337, 0.0466945747, 0.0076500749, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0), bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0.0128118901, 0.0584741018, 0.1372248726, 0.2490642026, 
        0.3939920918, 0.5720085401, 0.7449349944, 0.873358852, 0.9572776301, 
        0.9966913286, 1, 1, 1, 1), ml.sm = c(1, 1, 1, 1, 1, 1, 1, 0.9999949255, 
        0.9586349055, 0.8363671782, 0.6331917436, 0.3899306329, 0.2030414439, 
        0.0765950714, 0.0105915156, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0), ml.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0.0297398, 0.1199204824, 0.2705439815, 0.4816102972, 0.7241708423, 
        0.894983194, 0.9848878384, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 0.9931598424, 0.9194367394, 0.7648059292, 0.5310766984, 
        0.3079218193, 0.1452097569, 0.0429405111, 0.001114082, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0), ml.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0.0046439425, 0.0585589349, 0.172916744, 
        0.3477173698, 0.5828477317, 0.8022047078, 0.9406539766, 0.9981955381, 
        1, 1, 1, 1, 1, 1, 1), ro.sm = c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        0.9957271055, 0.8906658984, 0.6448474471, 0.3233841384, 0.1073921471, 
        0.0075248823, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), 
        ro.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.033536821, 0.1798539764, 
        0.4422958583, 0.7600656969, 0.9495812506, 1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.970593492, 
        0.8092293872, 0.5087192019, 0.2230523747, 0.053510274, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ro.bi = c(0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0014863965, 0.078128716, 
        0.270895762, 0.5787928875, 0.8527627877, 0.9859754436, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1), qr.sm = c(1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 0.9479152389, 0.7723251452, 0.4755209761, 
        0.2055921053, 0.0472040901, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0), qr.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0.0057108918, 
        0.0971743935, 0.3001787509, 0.6074826325, 0.8574332968, 0.9834496769, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 0.9937808388, 0.8925513155, 0.6673875081, 
        0.3541645251, 0.1288519965, 0.0150803237, 0, 0, 0, 0, 0, 
        0, 0, 0), qr.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0.0289113898, 0.1649912338, 0.4126119335, 0.7223350123, 
        0.9227119629, 0.9991546294, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1), vr.sm = c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 0.9993215643, 0.9701723968, 0.8980049773, 0.7828193058, 
        0.6246153824, 0.423393207, 0.1978223569, 0.0514140382, 7.60563e-05, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0), vr.me = c(0, 0, 0, 0.0092028086, 
        0.0985689252, 0.2830053787, 0.5090442674, 0.693059142, 0.8340557646, 
        0.9320341353, 0.986994254, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 0.9928240875, 0.9464676192, 0.8570928989, 0.7246999267, 
        0.5492887025, 0.3308592263, 0.127850589, 0.019470405, 0, 
        0, 0), vr.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0335408148, 
        0.1609350662, 0.3779865817, 0.5878124075, 0.7546199813, 0.8784093032, 
        0.959180373, 0.996933191, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1, 1), ty.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.040469215, 0.3936011905, 
        0.9084472656, 0.9593098958, 0.5078241257, 0.08203125, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0), x = c(0, 0.01953125, 0.0390625, 0.05859375, 0.078125, 
        0.09765625, 0.1171875, 0.13671875, 0.15625, 0.17578125, 0.1953125, 
        0.21484375, 0.234375, 0.25390625, 0.2734375, 0.29296875, 
        0.3125, 0.33203125, 0.3515625, 0.37109375, 0.390625, 0.41015625, 
        0.4296875, 0.44921875, 0.46875, 0.48828125, 0.5078125, 0.52734375, 
        0.546875, 0.56640625, 0.5859375, 0.60546875, 0.625, 0.64453125, 
        0.6640625, 0.68359375, 0.703125, 0.72265625, 0.7421875, 0.76171875, 
        0.78125, 0.80078125, 0.8203125, 0.83984375, 0.859375, 0.87890625, 
        0.8984375, 0.91796875, 0.9375, 0.95703125, 0.9765625, 0.99609375
        )), .Names = c("ex.sm", "ex.bi", "si.sm", "si.bi", "ve.sm", 
        "ve.bi", "lm", "um", "sm", "me", "bi", "ml.sm", "ml.me", "ml.bi", 
        "ro.sm", "ro.me", "ro.bi", "qr.sm", "qr.me", "qr.bi", "vr.sm", 
        "vr.me", "vr.bi", "ty.me", "x"), row.names = c(1L, 21L, 41L, 
        61L, 81L, 101L, 121L, 141L, 161L, 181L, 201L, 221L, 241L, 261L, 
        281L, 301L, 321L, 341L, 361L, 381L, 401L, 421L, 441L, 461L, 481L, 
        501L, 521L, 541L, 561L, 581L, 601L, 621L, 641L, 661L, 681L, 701L, 
        721L, 741L, 761L, 781L, 801L, 821L, 841L, 861L, 881L, 901L, 921L, 
        941L, 961L, 981L, 1001L, 1021L), class = "data.frame")
    testLingexpr(d, ctx5())
})


test_that('lingexpr ctx3bilat', {
    d <- structure(list(ex.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.9771938131, 0.508754917, 
        0.0631555944, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0), ex.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.118129917, 
        0.6457544192, 0.9984019886), X.ex.sm = c(0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0252540428, 
        0.3854895105, 0.9313249684, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.ex.bi = c(1, 
        0.7656052715, 0.1901770105, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), si.sm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 1, 0.819501466, 0.2977962018, 0.0168407207, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), 
        si.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0453514739, 0.3955078125, 
        0.889878738, 1), X.si.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0021701389, 0.2139247626, 
        0.7318239796, 0.9972092678, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.si.bi = c(1, 
        0.9429557956, 0.5070595947, 0.0877023987, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0), ve.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0.9813602941, 
        0.6703921569, 0.1668669872, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ve.bi = c(0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0.004253865, 0.2443438914, 0.7632230392, 0.9975980392, 
        1), X.ve.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0.1041195324, 0.5622426471, 0.9498039216, 
        1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0), X.ve.bi = c(1, 1, 0.8407352941, 
        0.3365502451, 0.0228129713, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), sm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 1, 1, 0.9783837191, 0.801645147, 0.461228044, 
        0.1779899775, 0.0271061477, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0), me = c(0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0.0271061477, 0.1779899775, 0.461228044, 0.801645147, 
        0.9783837191, 1, 1, 1, 1, 0.9572776301, 0.7449349944, 0.3939920918, 
        0.1372248726, 0.0128118901, 0, 0, 0, 0, 0, 0), bi = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0.0466945747, 0.2240492519, 0.5337581657, 
        0.8512344869, 0.9923689953, 1, 1), X.sm = c(0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0128118901, 0.1372248726, 
        0.3939920918, 0.7449349944, 0.9572776301, 1, 1, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0), X.me = c(0, 0, 0, 0, 0, 0, 0.003811802, 0.1017539372, 
        0.332050309, 0.681104029, 0.9290507284, 1, 1, 1, 1, 0.9923689953, 
        0.8512344869, 0.5337581657, 0.2240492519, 0.0466945747, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.bi = c(1, 1, 0.9992334588, 
        0.893703014, 0.610152251, 0.2754026957, 0.0715771712, 0.0001058834, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0), ml.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 
        1, 0.8672933402, 0.4345616087, 0.0970489206, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ml.me = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0970489206, 0.4345616087, 
        0.8672933402, 1, 1, 1, 1, 1, 1, 1, 0.9981955381, 0.8022047078, 
        0.3477173698, 0.0585589349, 0, 0, 0, 0, 0, 0), ml.bi = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0.001114082, 0.1452097569, 0.5310766984, 0.9194367394, 
        1, 1, 1, 1), X.ml.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0.0585589349, 0.3477173698, 0.8022047078, 
        0.9981955381, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.ml.me = c(0, 
        0, 0, 0, 0, 0, 0.0297398, 0.2705439815, 0.7241708423, 0.9848878384, 
        1, 1, 1, 1, 1, 1, 1, 0.9194367394, 0.5310766984, 0.1452097569, 
        0.001114082, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.ml.bi = c(1, 
        1, 1, 1, 0.9586349055, 0.6331917436, 0.2030414439, 0.0105915156, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0), ro.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 
        1, 1, 0.9229387194, 0.3805175038, 0.0182083571, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ro.me = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0182083571, 0.3805175038, 
        0.9229387194, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.8527627877, 
        0.270895762, 0.0014863965, 0, 0, 0, 0, 0), ro.bi = c(0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0.053510274, 0.5087192019, 0.970593492, 1, 1, 
        1, 1, 1), X.ro.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0.0014863965, 0.270895762, 0.8527627877, 
        1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.ro.me = c(0, 0, 0, 
        0, 0, 0, 0.1798539764, 0.7600656969, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1, 0.970593492, 0.5087192019, 0.053510274, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0), X.ro.bi = c(1, 1, 1, 1, 1, 0.9957271055, 
        0.6448474471, 0.1073921471, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), qr.sm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0.9681611436, 0.5426203856, 
        0.0699584247, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0), qr.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0699584247, 
        0.5426203856, 0.9681611436, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 0.9227119629, 0.4126119335, 0.0289113898, 0, 0, 0, 
        0), qr.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0.1288519965, 0.6673875081, 0.9937808388, 
        1, 1, 1, 1, 1, 1), X.qr.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0.0289113898, 0.4126119335, 0.9227119629, 
        1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.qr.me = c(0, 0, 
        0, 0, 0.0057108918, 0.3001787509, 0.8574332968, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9937808388, 0.6673875081, 0.1288519965, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.qr.bi = c(1, 1, 1, 1, 
        1, 1, 1, 0.7723251452, 0.2055921053, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), vr.sm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0.9158799213, 0.6596976273, 
        0.2385124611, 0.0027380257, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0), vr.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0027380257, 
        0.2385124611, 0.6596976273, 0.9158799213, 1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.996933191, 0.8784093032, 
        0.5878124075, 0.1609350662, 0, 0), vr.bi = c(0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.019470405, 
        0.3308592263, 0.7246999267, 0.9464676192, 1, 1, 1, 1, 1, 
        1, 1, 1), X.vr.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0.1609350662, 0.5878124075, 0.8784093032, 0.996933191, 
        1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.vr.me = c(0, 
        0, 0.0985689252, 0.5090442674, 0.8340557646, 0.986994254, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9464676192, 
        0.7246999267, 0.3308592263, 0.019470405, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0), X.vr.bi = c(1, 1, 1, 1, 1, 1, 1, 1, 0.9701723968, 
        0.7828193058, 0.423393207, 0.0514140382, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ty.me = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0134393601, 
        0.8372395833, 0.6337890625, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0), X.ty.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.3936011905, 
        0.9593098958, 0.08203125, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0), x = c(-1, -0.9609375, -0.921875, -0.8828125, 
        -0.84375, -0.8046875, -0.765625, -0.7265625, -0.6875, -0.6484375, 
        -0.609375, -0.5703125, -0.53125, -0.4921875, -0.453125, -0.4140625, 
        -0.375, -0.3359375, -0.296875, -0.2578125, -0.21875, -0.1796875, 
        -0.140625, -0.1015625, -0.0625, -0.0234375, 0.015625, 0.0546875, 
        0.09375, 0.1328125, 0.171875, 0.2109375, 0.25, 0.2890625, 
        0.328125, 0.3671875, 0.40625, 0.4453125, 0.484375, 0.5234375, 
        0.5625, 0.6015625, 0.640625, 0.6796875, 0.71875, 0.7578125, 
        0.796875, 0.8359375, 0.875, 0.9140625, 0.953125, 0.9921875
        )), .Names = c("ex.sm", "ex.bi", "X.ex.sm", "X.ex.bi", "si.sm", 
        "si.bi", "X.si.sm", "X.si.bi", "ve.sm", "ve.bi", "X.ve.sm", "X.ve.bi", 
        "sm", "me", "bi", "X.sm", "X.me", "X.bi", "ml.sm", "ml.me", "ml.bi", 
        "X.ml.sm", "X.ml.me", "X.ml.bi", "ro.sm", "ro.me", "ro.bi", "X.ro.sm", 
        "X.ro.me", "X.ro.bi", "qr.sm", "qr.me", "qr.bi", "X.qr.sm", "X.qr.me", 
        "X.qr.bi", "vr.sm", "vr.me", "vr.bi", "X.vr.sm", "X.vr.me", "X.vr.bi", 
        "ty.me", "X.ty.me", "x"), row.names = c(1L, 21L, 41L, 61L, 81L, 
        101L, 121L, 141L, 161L, 181L, 201L, 221L, 241L, 261L, 281L, 301L, 
        321L, 341L, 361L, 381L, 401L, 421L, 441L, 461L, 481L, 501L, 521L, 
        541L, 561L, 581L, 601L, 621L, 641L, 661L, 681L, 701L, 721L, 741L, 
        761L, 781L, 801L, 821L, 841L, 861L, 881L, 901L, 921L, 941L, 961L, 
        981L, 1001L, 1021L), class = "data.frame")
    testLingexpr(d, ctx3bilat())
})


test_that('lingexpr ctx5bilat', {
    d <- structure(list(ex.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.9771938131, 0.508754917, 
        0.0631555944, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0), ex.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.118129917, 
        0.6457544192, 0.9984019886), X.ex.sm = c(0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0252540428, 
        0.3854895105, 0.9313249684, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.ex.bi = c(1, 
        0.7656052715, 0.1901770105, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), si.sm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 1, 0.819501466, 0.2977962018, 0.0168407207, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), 
        si.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0453514739, 0.3955078125, 
        0.889878738, 1), X.si.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0021701389, 0.2139247626, 
        0.7318239796, 0.9972092678, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.si.bi = c(1, 
        0.9429557956, 0.5070595947, 0.0877023987, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0), ve.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0.9813602941, 
        0.6703921569, 0.1668669872, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ve.bi = c(0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0.004253865, 0.2443438914, 0.7632230392, 0.9975980392, 
        1), X.ve.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0.1041195324, 0.5622426471, 0.9498039216, 
        1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0), X.ve.bi = c(1, 1, 0.8407352941, 
        0.3365502451, 0.0228129713, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), lm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0715771712, 0.610152251, 0.9992334588, 
        1, 0.9992334588, 0.610152251, 0.0715771712, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), um = c(0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.1372248726, 
        0.7449349944, 1, 1, 0.9783837191, 0.461228044, 0.0271061477, 
        0, 0, 0), sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0.9783837191, 0.801645147, 
        0.461228044, 0.1779899775, 0.0271061477, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), me = c(0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0271061477, 0.1779899775, 0.461228044, 
        0.801645147, 0.9783837191, 1, 1, 1, 1, 0.9572776301, 0.7449349944, 
        0.3939920918, 0.1372248726, 0.0128118901, 0, 0, 0, 0, 0, 
        0), bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0466945747, 0.2240492519, 
        0.5337581657, 0.8512344869, 0.9923689953, 1, 1), X.lm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0271061477, 
        0.461228044, 0.9783837191, 1, 1, 0.7449349944, 0.1372248726, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.um = c(0, 0, 0, 0.003811802, 
        0.332050309, 0.9290507284, 1, 1, 0.8512344869, 0.2240492519, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0), X.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0.0128118901, 0.1372248726, 0.3939920918, 
        0.7449349944, 0.9572776301, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.me = c(0, 
        0, 0, 0, 0, 0, 0.003811802, 0.1017539372, 0.332050309, 0.681104029, 
        0.9290507284, 1, 1, 1, 1, 0.9923689953, 0.8512344869, 0.5337581657, 
        0.2240492519, 0.0466945747, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0), X.bi = c(1, 1, 0.9992334588, 0.893703014, 0.610152251, 
        0.2754026957, 0.0715771712, 0.0001058834, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0), ml.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0.8672933402, 
        0.4345616087, 0.0970489206, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0), ml.me = c(0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0.0970489206, 0.4345616087, 0.8672933402, 
        1, 1, 1, 1, 1, 1, 1, 0.9981955381, 0.8022047078, 0.3477173698, 
        0.0585589349, 0, 0, 0, 0, 0, 0), ml.bi = c(0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0.001114082, 0.1452097569, 0.5310766984, 0.9194367394, 
        1, 1, 1, 1), X.ml.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0.0585589349, 0.3477173698, 0.8022047078, 
        0.9981955381, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.ml.me = c(0, 
        0, 0, 0, 0, 0, 0.0297398, 0.2705439815, 0.7241708423, 0.9848878384, 
        1, 1, 1, 1, 1, 1, 1, 0.9194367394, 0.5310766984, 0.1452097569, 
        0.001114082, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.ml.bi = c(1, 
        1, 1, 1, 0.9586349055, 0.6331917436, 0.2030414439, 0.0105915156, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0), ro.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 
        1, 1, 0.9229387194, 0.3805175038, 0.0182083571, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ro.me = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0182083571, 0.3805175038, 
        0.9229387194, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.8527627877, 
        0.270895762, 0.0014863965, 0, 0, 0, 0, 0), ro.bi = c(0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0.053510274, 0.5087192019, 0.970593492, 1, 1, 
        1, 1, 1), X.ro.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0.0014863965, 0.270895762, 0.8527627877, 
        1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.ro.me = c(0, 0, 0, 
        0, 0, 0, 0.1798539764, 0.7600656969, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 1, 0.970593492, 0.5087192019, 0.053510274, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0), X.ro.bi = c(1, 1, 1, 1, 1, 0.9957271055, 
        0.6448474471, 0.1073921471, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), qr.sm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0.9681611436, 0.5426203856, 
        0.0699584247, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0), qr.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0699584247, 
        0.5426203856, 0.9681611436, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
        1, 1, 0.9227119629, 0.4126119335, 0.0289113898, 0, 0, 0, 
        0), qr.bi = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0.1288519965, 0.6673875081, 0.9937808388, 
        1, 1, 1, 1, 1, 1), X.qr.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0.0289113898, 0.4126119335, 0.9227119629, 
        1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.qr.me = c(0, 0, 
        0, 0, 0.0057108918, 0.3001787509, 0.8574332968, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9937808388, 0.6673875081, 0.1288519965, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.qr.bi = c(1, 1, 1, 1, 
        1, 1, 1, 0.7723251452, 0.2055921053, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), vr.sm = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0.9158799213, 0.6596976273, 
        0.2385124611, 0.0027380257, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0), vr.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0027380257, 
        0.2385124611, 0.6596976273, 0.9158799213, 1, 1, 1, 1, 1, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.996933191, 0.8784093032, 
        0.5878124075, 0.1609350662, 0, 0), vr.bi = c(0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.019470405, 
        0.3308592263, 0.7246999267, 0.9464676192, 1, 1, 1, 1, 1, 
        1, 1, 1), X.vr.sm = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0.1609350662, 0.5878124075, 0.8784093032, 0.996933191, 
        1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), X.vr.me = c(0, 
        0, 0.0985689252, 0.5090442674, 0.8340557646, 0.986994254, 
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9464676192, 
        0.7246999267, 0.3308592263, 0.019470405, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0), X.vr.bi = c(1, 1, 1, 1, 1, 1, 1, 1, 0.9701723968, 
        0.7828193058, 0.423393207, 0.0514140382, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), ty.me = c(0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.0134393601, 
        0.8372395833, 0.6337890625, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0), X.ty.me = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.3936011905, 
        0.9593098958, 0.08203125, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        0, 0, 0, 0, 0, 0, 0), x = c(-1, -0.9609375, -0.921875, -0.8828125, 
        -0.84375, -0.8046875, -0.765625, -0.7265625, -0.6875, -0.6484375, 
        -0.609375, -0.5703125, -0.53125, -0.4921875, -0.453125, -0.4140625, 
        -0.375, -0.3359375, -0.296875, -0.2578125, -0.21875, -0.1796875, 
        -0.140625, -0.1015625, -0.0625, -0.0234375, 0.015625, 0.0546875, 
        0.09375, 0.1328125, 0.171875, 0.2109375, 0.25, 0.2890625, 
        0.328125, 0.3671875, 0.40625, 0.4453125, 0.484375, 0.5234375, 
        0.5625, 0.6015625, 0.640625, 0.6796875, 0.71875, 0.7578125, 
        0.796875, 0.8359375, 0.875, 0.9140625, 0.953125, 0.9921875
        )), .Names = c("ex.sm", "ex.bi", "X.ex.sm", "X.ex.bi", "si.sm", 
        "si.bi", "X.si.sm", "X.si.bi", "ve.sm", "ve.bi", "X.ve.sm", "X.ve.bi", 
        "lm", "um", "sm", "me", "bi", "X.lm", "X.um", "X.sm", "X.me", 
        "X.bi", "ml.sm", "ml.me", "ml.bi", "X.ml.sm", "X.ml.me", "X.ml.bi", 
        "ro.sm", "ro.me", "ro.bi", "X.ro.sm", "X.ro.me", "X.ro.bi", "qr.sm", 
        "qr.me", "qr.bi", "X.qr.sm", "X.qr.me", "X.qr.bi", "vr.sm", "vr.me", 
        "vr.bi", "X.vr.sm", "X.vr.me", "X.vr.bi", "ty.me", "X.ty.me", 
        "x"), row.names = c(1L, 21L, 41L, 61L, 81L, 101L, 121L, 141L, 
        161L, 181L, 201L, 221L, 241L, 261L, 281L, 301L, 321L, 341L, 361L, 
        381L, 401L, 421L, 441L, 461L, 481L, 501L, 521L, 541L, 561L, 581L, 
        601L, 621L, 641L, 661L, 681L, 701L, 721L, 741L, 761L, 781L, 801L, 
        821L, 841L, 861L, 881L, 901L, 921L, 941L, 961L, 981L, 1001L, 
        1021L), class = "data.frame")
    testLingexpr(d, ctx5bilat())
})
