% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lasnormalize.r
\docType{methods}
\name{lasnormalize}
\alias{lasnormalize}
\alias{lasunnormalize}
\alias{-,LAS,RasterLayer-method}
\alias{-,LAS,function-method}
\title{Remove the topography from a point cloud}
\usage{
lasnormalize(las, algorithm, na.rm = FALSE)

lasunnormalize(las)

\S4method{-}{LAS,RasterLayer}(e1, e2)

\S4method{-}{LAS,`function`}(e1, e2)
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{algorithm}{a spatial interpolation function. \code{lidR} have \link{tin},
\link{kriging}, \link{knnidw} or a \link[raster:raster]{RasterLayer} representing a digital terrain
model (can be computed with \link{grid_terrain})}

\item{na.rm}{logical. When using a \code{RasterLayer} as DTM, by defaut the function fails if a point
fall in an empty pixel because a Z elevation cannot be NA. If \code{na.rm = TRUE} points with an
elvation of NA are filtered. Becareful this creates a copy of the point cloud.}

\item{e1}{a LAS object}

\item{e2}{\link[raster:raster]{RasterLayer} representing a digital terrain model (can be
computed with \link{grid_terrain}) or a spatial interpolation function. \code{lidR} has \link{tin},
\link{kriging}, and \link{knnidw}.}
}
\value{
If the input is a \code{LAS} object, return a \code{LAS} object. If the input is a
\code{LAScatalog}, returns a \code{LAScatalog}.
}
\description{
Subtract digital terrain model (DTM) from LiDAR point cloud to create a dataset normalized with
the ground at 0. The DTM can originate from an external file or can be computed by the user. It can
also be computed on-the-fly. In this case the algorithm does not use rasterized data and each point
is interpolated. There is no inaccuracy due to the discretization of the terrain and the resolution
of the terrain is virtually infinite.\cr\cr
How well the edges of the dataset are interpolated depends on the interpolation method used.
Thus, a buffer around the region of interest is always recommended to avoid edge effects.\cr\cr
The attribute Z of the returned LAS object is the normalized elevation. A new attribute 'Zref' records
the former elevation values, which enables the use of \code{lasunormalize} to restore original point elevations.\cr\cr
}
\section{Working with a \code{LAScatalog}}{

This section appears in each function that supports a LAScatalog as input.\cr

In \code{lidR} when the input of a function is a \link[lidR:LAScatalog-class]{LAScatalog} the
function uses the LAScatalog processing engine. The user can modify the engine options using
the \link[lidR:catalog_options_tools]{available options}. A careful reading of the
\link[lidR:LAScatalog-class]{engine documentation} is recommended before processing \code{LAScatalogs}. Each
\code{lidR} function should come with a section that documents the supported engine options.\cr

The \code{LAScatalog} engine supports \code{.lax} files that \emph{significantly} improve the computation
speed of spatial queries using a spatial index. Users should really take advantage a \code{.lax} files,
but this is not mandatory.\cr
}

\section{Supported processing options}{

Supported processing options for a \code{LAScatalog} (in bold). For more details see the
\link[lidR:LAScatalog-class]{LAScatalog engine documentation}:
\itemize{
\item \strong{chunk size}: How much data is loaded at once.
\item \strong{chunk buffer*}: Mandatory to get a continuous output without edge effects. The buffer
is always removed once processed and will never be returned either in R or in files.
\item \strong{chunk alignment}: Align the processed chunks.
\item \strong{cores}: How many cores are used. More cores means more data is loaded at once.
\item \strong{progress}: Displays a progression estimation.
\item \strong{output_files*}: Mandatory because the output is likely to be too big to be returned
 in R and needs to be written in las/laz files. Supported templates are \code{\{XLEFT\}}, \code{\{XRIGHT\}},
\code{\{YBOTTOM\}}, \code{\{YTOP\}}, \code{\{XCENTER\}}, \code{\{YCENTER\}} \code{\{ID\}} and, if
chunk size is equal to 0 (processing by file), \code{\{ORIGINALFILENAME\}}.
\item \strong{laz_compression}: write \code{las} or \code{laz} files
\item select: The function will write files equivalent to the original ones. Thus \code{select = "*"}
and cannot be changed.
\item \strong{filter}: Read only points of interest.
}
}

\examples{
LASfile <- system.file("extdata", "Topography.laz", package="lidR")
las <- readLAS(LASfile)

plot(las)

# First option: use a RasterLayer as DTM
# =======================================================

dtm <- grid_terrain(las, 1, kriging(k = 10L))
las <- lasnormalize(las, dtm)

plot(dtm)
plot(las)

# restore original elevations
las <- lasunnormalize(las)
plot(las)

# operator - can be used. This is equivalent to the previous
las <- las - dtm
plot(las)

# restore original elevations
las <- lasunnormalize(las)

# Second option: interpolate each point (no discretization)
# =========================================================

las <- lasnormalize(las, tin())
plot(las)

# operator - can be used. This is equivalent to the previous
las <- lasunnormalize(las)
las <- las - tin()

\dontrun{
# All the following syntaxes are correct
las <- lasnormalize(las, knnidw())
las <- lasnormalize(las, knnidw(k = 8, p = 2))
las <- las - knnidw()
las <- las - knnidw(k = 8)
las <- lasnormalize(las, kriging())
las <- las - kriging(k = 8)
}

}
\seealso{
\link[raster:raster]{raster}
\link[lidR:grid_terrain]{grid_terrain}
}
