% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_stats.R
\name{getRhythmStats}
\alias{getRhythmStats}
\title{Compute rhythm statistics from fitted models}
\usage{
getRhythmStats(
  fit,
  fitType = c("posterior_mean", "posterior_samples", "raw"),
  features = NULL,
  dopar = TRUE,
  rms = FALSE
)
}
\arguments{
\item{fit}{A \code{limorhyde2} object.}

\item{fitType}{String indicating which fitted models to use to compute the
rhythmic statistics. A typical analysis using \code{limorhyde2} will be based on
'posterior_mean', the default.}

\item{features}{Vector of names, row numbers, or logical values for
subsetting the features. \code{NULL} indicates all features.}

\item{dopar}{Logical indicating whether to run calculations in parallel if
a parallel backend is already set up, e.g., using
\code{\link[doParallel:registerDoParallel]{doParallel::registerDoParallel()}}. Recommended to minimize runtime.}

\item{rms}{Logical indicating whether to calculate \code{rms_amp}.}
}
\value{
A \code{data.table} containing the following rhythm statistics:
\itemize{
\item \code{peak_phase}: time between 0 and \code{fit$period} at which the peak or maximum
value occurs
\item \code{peak_value}
\item \code{trough_phase}: time between 0 and \code{fit$period} at which the trough or
minimum value occurs
\item \code{trough_value}
\item \code{peak_trough_amp}: \code{peak_value - trough_value}
\item \code{rms_amp}: root mean square difference between fitted curve and mean value
between time 0 and \code{fit$period} (only calculated if \code{rms} is \code{TRUE})
\item \code{mesor}: mean value between time 0 and \code{fit$period}
}

The rows of the \code{data.table} depend on the \code{fit} object and \code{fitType}:
\itemize{
\item \code{fit} contains data from one condition and \code{fitType} is posterior_mean' or
'raw': one row per feature.
\item \code{fit} contains data from one condition and \code{fitType} is
'posterior_samples': one row per feature per posterior sample.
\item \code{fit} contains data from multiple conditions and \code{fitType} is
'posterior_mean' or 'raw': one row per feature per condition.
\item \code{fit} contains data from multiple conditions and \code{fitType} is
'posterior_samples': one row per feature per condition per posterior
sample.
}
}
\description{
This function uses \code{\link[stats:optim]{stats::optim()}} to compute various properties of
fitted curves with respect to time, potentially in each condition and for
each posterior sample, and adjusting for any covariates.
}
\examples{
library('data.table')

# rhythmicity in one condition
y = GSE54650$y
metadata = GSE54650$metadata

fit = getModelFit(y, metadata)
fit = getPosteriorFit(fit)
rhyStats = getRhythmStats(fit, features = c('13170', '13869'))

# rhythmicity and differential rhythmicity in multiple conditions
y = GSE34018$y
metadata = GSE34018$metadata

fit = getModelFit(y, metadata, nKnots = 3L, condColname = 'cond')
fit = getPosteriorFit(fit)
rhyStats = getRhythmStats(fit, features = c('13170', '12686'))
diffRhyStats = getDiffRhythmStats(fit, rhyStats)
}
\seealso{
\code{\link[=getModelFit]{getModelFit()}}, \code{\link[=getPosteriorFit]{getPosteriorFit()}},  \code{\link[=getPosteriorSamples]{getPosteriorSamples()}},
\code{\link[=getDiffRhythmStats]{getDiffRhythmStats()}}, \code{\link[=getStatsIntervals]{getStatsIntervals()}}
}
