% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lineqDGP.R
\name{simulate.lineqDGP}
\alias{simulate.lineqDGP}
\title{Simulation Method for the \code{"lineqDGP"} S3 Class}
\usage{
\method{simulate}{lineqDGP}(object, nsim = 1, seed = NULL, xtest, ...)
}
\arguments{
\item{object}{an object with class \code{"lineqDGP"}.}

\item{nsim}{the number of simulations.}

\item{seed}{see \code{\link{simulate}}.}

\item{xtest}{a vector (or matrix) with the test input design.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
An object with the simulations of \code{"lineqDGP"} models.
\item{x}{A vector (or matrix) with the training input design.}
\item{y}{The training output vector at \code{x}.}
\item{xtest}{A vector (or matrix) with the test input design.}
\item{Phi.test}{A matrix corresponding to the hat basis functions evaluated at \code{xtest}.
The basis functions are indexed by rows.}
\item{xi.sim}{Posterior sample-path of the finite-dimensional Gaussian vector.}
\item{ysim}{Posterior sample-path of the observed GP. Note: \code{ysim = Phi.test \%*\% xi.sim}.}
}
\description{
Simulation method for the \code{"lineqDGP"} S3 class.
}
\details{
The posterior sample-path of the finite-dimensional GP with inequality constraints is computed
according to (Maatouk and Bay, 2017).
}
\examples{
# creating the model
sigfun <- function(x) return(1/(1+exp(-7*(x-0.5))))
x <- seq(0, 1, length = 5)
y <- sigfun(x)
model <- create(class = "lineqDGP", x, y, constrType = "monotonicity")

# updating and expanding the model
model$localParam$m <- 30
model$kernParam$par <- c(1, 0.2)
model <- augment(model)

# sampling from the model
xtest <- seq(0, 1, length = 100)
ytest <- sigfun(xtest)
sim.model <- simulate(model, nsim = 50, seed = 1, xtest = xtest)
mu <- apply(sim.model$ysim, 1, mean)
qtls <- apply(sim.model$ysim, 1, quantile, probs = c(0.05, 0.95))
matplot(xtest, t(qtls), type = "l", lty = 1, col = "gray90",
        main = "Constrained Kriging model")
polygon(c(xtest, rev(xtest)), c(qtls[2,], rev(qtls[1,])), col = 'gray90', border = NA)
lines(xtest, ytest, lty = 2)
lines(xtest, mu, type = "l", col = "darkgreen")
points(x, y, pch = 20)
legend("right", c("ytrain","ytest","mean","confidence"), lty = c(NaN,2,1,NaN),
       pch = c(20,NaN,NaN,15), col = c("black","black","darkgreen","gray80"))

}
\references{
Maatouk, H. and Bay, X. (2017),
"Gaussian process emulators for computer experiments with inequality constraints".
\emph{Mathematical Geosciences},
49(5):557-582.
\href{https://link.springer.com/article/10.1007/s11004-017-9673-2}{[link]}
}
\seealso{
\code{\link{create.lineqDGP}}, \code{\link{augment.lineqDGP}}, \code{\link{predict.lineqDGP}}
}
\author{
A. F. Lopez-Lopera.
}
