% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/round.r
\name{round_date}
\alias{round_date}
\alias{floor_date}
\alias{ceiling_date}
\title{Round, floor and ceiling methods for date-time objects}
\usage{
round_date(x, unit = "second",
  week_start = getOption("lubridate.week.start", 7))

floor_date(x, unit = "seconds",
  week_start = getOption("lubridate.week.start", 7))

ceiling_date(x, unit = "seconds", change_on_boundary = NULL,
  week_start = getOption("lubridate.week.start", 7))
}
\arguments{
\item{x}{a vector of date-time objects}

\item{unit}{a character string specifying a time unit or a multiple of a unit
to be rounded to. Valid base units are \code{second}, \code{minute}, \code{hour}, \code{day},
\code{week}, \code{month}, \code{bimonth}, \code{quarter}, \code{season}, \code{halfyear} and
\code{year}. Arbitrary unique English abbreviations as in the \code{\link[=period]{period()}}
constructor are allowed. Rounding to multiple of units (except weeks) is
supported.}

\item{week_start}{when unit is \code{weeks} specify the reference day; 7 being Sunday.}

\item{change_on_boundary}{If NULL (the default) don't change instants on the
boundary (\code{ceiling_date(ymd_hms('2000-01-01 00:00:00'))} is \code{2000-01-01 00:00:00}), but round up \code{Date} objects to the next boundary
(\code{ceiling_date(ymd("2000-01-01"), "month")} is \code{"2000-02-01"}). When
\code{TRUE}, instants on the boundary are rounded up to the next boundary. When
\code{FALSE}, date-time on the boundary are never rounded up (this was the
default for \pkg{lubridate} prior to \code{v1.6.0}. See section \code{Rounding Up Date Objects} below for more details.}
}
\description{
Rounding to the nearest unit or multiple of a unit are supported. All
meaningfull specifications in English language are supported - secs, min,
mins, 2 minutes, 3 years etc.

Rounding to fractional seconds is supported. Please note that rounding to
fractions smaller than 1s can lead to large precision errors due to the
floating point representation of the POSIXct objects. See examples.

\code{round_date()} takes a date-time object and rounds it to the nearest value
of the specified time unit. For rounding date-times which is exactly halfway
between two consecutive units, the convention is to round up. Note that this
is in line with the behavior of R's \code{\link[base:round.POSIXt]{base::round.POSIXt()}} function
but does not follow the convention of the base \code{\link[base:round]{base::round()}} function
which "rounds to the even digit" per IEC 60559.

\code{floor_date()} takes a date-time object and rounds it down to the nearest
boundary of the specified time unit.

\code{ceiling_date()} takes a date-time object and rounds it up to the nearest
boundary of the specified time unit.
}
\details{
In \pkg{lubridate}, rounding of a date-time objects tries to
preserve the class of the input object whenever possible. This is done by
first rounding to an instant and then converting to the original class by
usual R conventions.
}
\section{Rounding Up Date Objects}{


By default rounding up \code{Date} objects follows 3 steps:
\enumerate{
\item Convert to an instant representing lower bound of the Date:
\code{2000-01-01} --> \code{2000-01-01 00:00:00}
\item Round up to the \strong{next} closest rounding unit boundary. For example,
if the rounding unit is \code{month} then next closest boundary of \code{2000-01-01}
is \code{2000-02-01 00:00:00}.

The motivation for this is that the "partial" \code{2000-01-01} is conceptually
an interval (\code{2000-01-01 00:00:00} -- \code{2000-01-02 00:00:00}) and the day
hasn't started clocking yet at the exact boundary \code{00:00:00}. Thus, it
seems wrong to round up a day to its lower boundary.

The behavior on the boundary can be changed by setting
\code{change_on_boundary} to a non-\code{NULL} value.
\item If rounding unit is smaller than a day, return the instant from step 2
(\code{POSIXct}), otherwise convert to and return a \code{Date} object.
}
}

\examples{

## print fractional seconds
options(digits.secs=6)

x <- as.POSIXct("2009-08-03 12:01:59.23")
round_date(x, ".5s")
round_date(x, "sec")
round_date(x, "second")
round_date(x, "minute")
round_date(x, "5 mins")
round_date(x, "hour")
round_date(x, "2 hours")
round_date(x, "day")
round_date(x, "week")
round_date(x, "month")
round_date(x, "bimonth")
round_date(x, "quarter") == round_date(x, "3 months")
round_date(x, "halfyear")
round_date(x, "year")

x <- as.POSIXct("2009-08-03 12:01:59.23")
floor_date(x, ".1s")
floor_date(x, "second")
floor_date(x, "minute")
floor_date(x, "hour")
floor_date(x, "day")
floor_date(x, "week")
floor_date(x, "month")
floor_date(x, "bimonth")
floor_date(x, "quarter")
floor_date(x, "season")
floor_date(x, "halfyear")
floor_date(x, "year")

x <- as.POSIXct("2009-08-03 12:01:59.23")
ceiling_date(x, ".1 sec") # imprecise representation at 0.1 sec !!!
ceiling_date(x, "second")
ceiling_date(x, "minute")
ceiling_date(x, "5 mins")
ceiling_date(x, "hour")
ceiling_date(x, "day")
ceiling_date(x, "week")
ceiling_date(x, "month")
ceiling_date(x, "bimonth") == ceiling_date(x, "2 months")
ceiling_date(x, "quarter")
ceiling_date(x, "season")
ceiling_date(x, "halfyear")
ceiling_date(x, "year")

## POSIXct precision is pretty much limited to seconds:
as.POSIXct("2009-08-03 12:01:59.3") ## -> "2009-08-03 12:01:59.2 CEST"
ceiling_date(x, ".1 sec") ## -> "2009-08-03 12:01:59.2 CEST"

x <- ymd("2000-01-01")
ceiling_date(x, "month")
ceiling_date(x, "month", change_on_boundary = TRUE)
}
\seealso{
\code{\link[base:round]{base::round()}}
}
\keyword{chron}
\keyword{manip}
