% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cliBrtSunDurFrc.R
\name{cliBrtSunDurFrcPoints}
\alias{cliBrtSunDurFrcPoints}
\title{Estimator for Fraction of Bright Sunshine Duration}
\usage{
cliBrtSunDurFrcPoints(
  temp,
  prec,
  lat,
  lon,
  elv,
  year = 2000,
  aprchSIM = c("Solar123", "SPLASH")
)
}
\arguments{
\item{temp}{'numeric' R object with one-year time series of monthly mean air temperature (in °C)}

\item{prec}{'numeric' R object with one-year time series of monthly precipitation sum (in mm)}

\item{lat}{'numeric' vector with the latitude coordinates (in decimal degrees)}

\item{lon}{'numeric' vector with the longitude coordinates (in decimal degrees)}

\item{elv}{'numeric' vector with the elevation values (in meters above sea level)}

\item{year}{'numeric' vector with values of the year (using astronomical year numbering)}

\item{aprchSIM}{'character' vector of length 1 that indicates the formula used to estimate the value of solar
irradiance/irradiation for a specific day. Valid values are as follows: \cr
(a) \code{'Solar123'} -
in this approach, first, the mean hourly solar irradiance under cloudless-sky conditions is calculated as
proposed by Yin (1997b), with a minor modification, using the daytime means of optical air mass and
cosine zenith; the former is computed as recommended by Yin (1997b), while the latter is estimated by using
Eq 5 of Yin (1997a); however, in contrast to the original approach, where the solar constant was fixed at
\eqn{4.9212 MJ m^{-2} hr^{-1}}{4.9212 MJ m^{-2} hr^{-1}}, according to Yin (1999), its value is corrected by
calendar day for the variable ellipticity of the Earth's orbit, by using the scheme of Brock (1981); in the
calculations, the values of solar declination and daylength are derived by using the approach of Brock (1981);
\cr
(b) \code{'SPLASH'} -
in this approach, first, under varying orbital parameters, the daily solar radiation at the top of the
atmosphere is calculated (\eqn{H_{0}}{H_{0}}, Eq 7 in Davis et al. (2017)), and then this value is
multiplied by the atmospheric transmittivity to obtain the value of daily surface radiation; in this case as
well, cloudless conditions are assumed, i.e., the transmission coefficient is taken into account with an
universal value of 0.75, however, its value is modified as a function of elevation, by using the scheme of
Allen (1996); the daylength is calculated via Eq 1.6.11 in Duffie and Beckman (1991), using the sunset hour
angle (\eqn{h_{s}}{h_{s}}, Eq 8. in Davis et al. (2017)); finally, the mean hourly surface radiation is
derived as the quotient of the daily surface radiation and the daylength.}
}
\value{
A 12-column matrix with monthly averages of relative sunshine duration.
}
\description{
Estimates monthly averages for daily fraction of bright sunshine duration, for a given geographical
    location (latitude, longitude, and elevation) and year, by using the monthly time series of temperature and
    precipitation.
}
\details{
To estimate the monthly averages of relative sunlight duration, the approach presented by Yin (1999) is
    implemented here. Many variables in this estimation scheme can be easily and unambiguously determined, but the
    approach uses two important quantities, the calculation method of which can be chosen here depending on the
    purpose of the investigations. One of them is the estimated value of the mean hourly solar irradiance under
    cloudless-sky conditions. This quantity can be estimated in this implementation of the approach with the
    original method (\code{aprchSIM = 'Solar123'}) or with the solar radiation model used in the SPLASH algorithm,
    considering the variability of orbital parameters of the Earth over time (\code{aprchSIM = 'SPLASH'}). The
    latter is recommended for paleo-climatological and paleo-environmental studies. These solar radiation models
    is also applied to calculate the daylength, whose monthly averages are used to estimate monthly averages of
    daily potential evapotranspiration (Eqs. A10 and A11 in Yin (1998)). \cr
    The procedure proposed by Yin (1999) requires the calculation of several regional factors (see Eq 3.3 in Yin
    (1999)). Each regional factor is activated as a function of latitude and longitude. However, it is important
    to note that in this implementation, these factors are activated with the current configuration of continents
    and islands. Continents and regions are classified using the high-resolution world map of the
    \code{\link[rworldmap]{rworldmap-package}}. In checking whether or not a given geographic location can be
    defined as an island, the high-resolution world map of the \code{\link[rnaturalearth]{rnaturalearth}}
    is applied.
}
\note{
As with any function with a point mode, a set of basic input data is defined here. In this case, they are as
   follows: \code{'temp'} (one-year time series of monthly mean air temperature), and \code{'prec'} (one-year time
   series of monthly precipitation sum). The objects \code{'temp'} and \code{'prec'} must be either 12-length
   vectors or 12-column matrices. The first dimensions of these matrices have to be the same length. The function
   automatically converts vectors into single-row matrices during the error handling, and then uses these
   matrices. The first dimensions of these matrices determines the number of rows in the result matrix. In the
   case of arguments that do not affect the course of the calculation procedure or the structure of the return
   object, scalar values (i.e., 'numeric' vector of length 1) may also be allowed. In this case, they are as
   follows: \code{'lat'} (latitude coordinates in decimal degrees), \code{'lon'} (longitude coordinates in decimal
   degrees), \code{'elv'} (elevation in meters above sea level), and \code{'year'} (year using astronomical year
   numbering). These scalars are converted to vectors by the function during the error handling, and these vectors
   are applied in the further calculations. If these data are stored in vectors of length at least 2, their length
   must be the same size of first dimension of the matrices containing the basic data.
}
\examples{
library (graphics)

# Loading mandatory data for the Example 'Points'
data(inp_exPoints)

# Measured and estimated one-year time series of the monthly mean relative sunshine duration,
# at a grid cell near Szeged, Hungary (46.3N, 20.2E), in the year 2010
with(inp_exPoints, {
bsdf01 <- matrix(nrow = 0, ncol = 12, dimnames = list(NULL, month.abb))
bsdf01 <- rbind(bsdf01, "Measured" = bsdf["2010", ])
bsdf01 <- rbind(bsdf01, "Solar123" = cliBrtSunDurFrcPoints(temp["2010", ], prec["2010", ],
    lat, lon, elv, year = 2010))
bsdf01 <- rbind(bsdf01, "SPLASH" = cliBrtSunDurFrcPoints(temp["2010", ], prec["2010", ],
    lat, lon, elv, year = 2010, aprchSIM = "SPLASH"))
cols <- c("black", "green", "blue")
matplot(t(bsdf01), type = "l", lwd = 2, col = cols, xaxt = "n", xlab = "Month",
    ylab = "Average relative sunshine duration (unitless)")
axis(1, at = seq(1, ncol(bsdf01)), labels = colnames(bsdf01))
legend(1, 0.7, legend = rownames(bsdf01), col = cols, lty = 1 : 2, lwd = 2, xpd = TRUE)
})

\donttest{
# Relative root mean square error between measured and estimated values for the 'bsdf',
# at a grid cell near Szeged, Hungary (46.3N, 20.2E), in the period 1981-2010
with(inp_exPoints, {
years <- seq(1981, 2010)
bsdf02 <- cliBrtSunDurFrcPoints(temp, prec, lat, lon, elv, year = years)
rrmse <- function(pre, obs) { (sqrt(mean((pre - obs) ^ 2.)) / mean(obs)) * 100. }
rrmse_bsdf <- sapply(1 : 12, function(i) { rrmse(bsdf02[, i], bsdf[, i])  })
cols <- c("black", "green")
plot(rrmse_bsdf, type = "l", lwd = 2, col = cols, xaxt = "n", xlab = "Month",
    ylab = "Relative root mean square error (\%)")
axis(1, at = 1 : 12, labels = month.abb)
})
}

}
\references{
\cite{Allen RG (1996) Assessing integrity of weather data for reference evapotranspiration estimation. J Irrig
    Drain Eng 122(2):97–106. \doi{10.1061/(ASCE)0733-9437(1996)122:2(97)}}

\cite{Berger A, Loutre MF (1991) Insolation values for the climate of the last 10 million years. Quat Sci Rev
    10(4):297-317. \doi{10.1016/0277-3791(91)90033-Q}}

\cite{Brock TD (1981) Calculating solar radiation for ecological studies. Ecol Model 14(1–2):1-19.
    \doi{10.1016/0304-3800(81)90011-9}}

\cite{Davis TW, Prentice IC, Stocker BD, Thomas RT, Whitley RJ, Wang H, Evans BJ, Gallego-Sala AV, Sykes MT,
    Cramer W (2017) Simple process-led algorithms for simulating habitats (SPLASH v.1.0): robust indices of
    radiation, evapotranspiration and plant-available moisture. Geosci Model Dev 10(2):689–708.
    \doi{10.5194/gmd-10-689-2017}}

\cite{Duffie JA, Beckman WA (1991) Solar Engineering of Thermal Processes. Second Edition. Wiley-Interscience,
    New York, NY}

\cite{Yin X (1997a) Calculating daytime mean relative air mass. Agric For Meteorol 87(2-3):85-90.
    \doi{10.1016/S0168-1923(97)00029-4}}

\cite{Yin X (1997b) Optical Air Mass: Daily Integration and its Applications. Meteorol Atmos Phys 63(3-4):227-233.
    \doi{10.1007/BF01027387}}

\cite{Yin X (1998) The Albedo of Vegetated Land Surfaces: Systems Analysis and Mathematical Modeling. Theor Appl
    Climatol 60(1–4):121–140. \doi{10.1007/s007040050038}}

\cite{Yin X (1999) Bright Sunshine Duration in Relation to Precipitation, Air Temperature and Geographic Location.
    Theor Appl Climatol 64(1–2):61–68. \doi{10.1007/s007040050111}}
}
