% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/execute.multi.analysis.R
\name{execute.multi.analysis}
\alias{execute.multi.analysis}
\title{Performs Multiple Analyses on Distance Data}
\usage{
execute.multi.analysis(
  species.code,
  unidentified.sightings = NULL,
  species.presence = NULL,
  covariate.uncertainty = NULL,
  models.by.species.code,
  ddf.model.objects,
  ddf.model.options = list(criterion = "AIC", species.field.name = "species"),
  region.table,
  sample.table,
  obs.table,
  dht.options = list(convert.units = 1),
  bootstrap,
  bootstrap.options = list(resample = "samples", n = 1, quantile.type = 7),
  silent = FALSE
)
}
\arguments{
\item{species.code}{vector of all the species codes to be included in
the analysis}

\item{unidentified.sightings}{a list with an element for each
unidentified code which contains a vector of corresponding identified
species codes or NULL if not required}

\item{species.presence}{must be specified if species.code.definitions is
specified. A list with an element for each strata which contains the vector
of species codes present in that strata}

\item{covariate.uncertainty}{a dataframe detailing the variables to be
resampled - variable.layer, variable.name, cor.factor.layer,
cor.factor.name , uncertainty.layer, uncertainty.name,
uncertainty.measure, sampling.distribution. or NULL if not required}

\item{models.by.species.code}{a list of character vectors of model names
with the elements named by species code}

\item{ddf.model.objects}{a list of all the ddf models named in models.by.species.code}

\item{ddf.model.options}{a list of options 1) selection.criterion either "AIC",
"AICc" or "BIC" 2) species.field.name describing the field name in the ddf
dataset containing species codes.}

\item{region.table}{dataframe of region records - Region.Label and Area}

\item{sample.table}{dataframe of sample records - Region.Label,
Sample.Label, Effort}

\item{obs.table}{dataframe of observation records with fields object,
Region.Label, and Sample.Label which give links to sample.table,
region.table and the data records used in \code{model}}

\item{dht.options}{list containing option for dht: convert.units indicated
if the distance measurement units are different from shapefile and transect
coordinate units.}

\item{bootstrap}{if TRUE resamples data to obtain variance estimate}

\item{bootstrap.options}{a list of options that can be set 1) n: number of
repetitions 2) resample: how to resample data ("samples", "observations")}

\item{silent}{boolean used to suppress progress counter output}
}
\value{
object of class "ma" which consists of a list of objects of class
  "ma.element". Each "ma.element" consists of the following elements:
  \item{individuals}{Summary, N (abundance) and D (density) tables}
  \item{clusters}{Summary, N (abundance) and D (density) tables}
  \item{Expected.S}{Expected cluster size table}
  \item{ddf}{Model details including a summary of convergence and selection
    as well as parameter estimates for selected models.}
}
\description{
Analyses are performed for multiple species contained within the same
dataset. Individual detection function analyses of each species must have
already been completed using the \code{ddf} function in the \code{mrds}
library. This function may then perform additional tasks such as assessing
variance via a non-parametric bootstrap, including covariate variability via
a parametric bootstrap, including model uncertainty and dealing with species
codes which relate to unidentified sightings.
}
\details{
This is a new package with limited testing on real data, please drop
me a line if you plan on using it (lhm[at]st-and.ac.uk).

The model fitting code in this function obtains its data and the model
descriptions from the ddf objects passed in via the \code{ddf.models} argument.

If you wish to include model uncertainty then each model which you wish to
be included in the analyses must have already been run and should be
provided in the \code{ddf.models} argument. The \code{model.names} argument
tells this function which \code{"ddf"} objects are
associated with which species code in the dataset. This object must be
constructed as a list of vectors. Each element in the list must be named
corresponding to one of the species codes in the dataset and contain a
character vector of object names.

For the majority of analyses the variance will be estimated using a
non-parametric bootstrap, indicated by the \code{bootstrap} argument. You
may select options for the bootstrap using the \code{bootstrap.options}
argument. This is a list with elements specifying the number of repetitions
and whether to resample samples within strata (\code{$resample = "samples"})
or observations withing strata (\code{$resample = "observations"}). In
addition, the \code{bootstrap.covariates} is a boolean argument specifying
whether or not a parametric bootstrap should be performed on any of the
covariates. The details of which variables should be resampled and from
which distributions should be entered in the \code{covariate.uncertainty}
dataframe. This dataframe should contain 7 columns with the following names:
\code{variable.layer}, \code{variable.name},
\code{cor.factor.layer}, \code{cor.factor.name}, \code{uncertainty.layer},
\code{uncertainty.name}, \code{uncertainty.measure} and
\code{sampling.distribution}. [Currently this is only implemented for the
observation layer]. The \code{variable.name} and
\code{uncertainty.name} should be the names of the variable in the dataset
giving the covariate to be resampled and the variable containing the
uncertainty respectively. The \code{cor.factor.layer} specifies the data
layer which contains the correction factor variable, although alternatively
"numeric" can be entered. The \code{cor.factor.name} specifies the name of
the correction factor variable or the correction factor value if "numeric"
was specified for the correction factor layer.
The \code{uncertainty.name} should specify what
values the uncertainty variable contains and should be one of \code{"sd"},
\code{"var"} or \code{"CV"}. The \code{sampling.distribution} should specify
one of the following distributions to parametrically resample from
\code{"Normal"}, \code{"Normal.Absolute"}, \code{"Lognormal.BC"},
\code{"Poisson"} or \code{"TruncPoissonBC"}. The remaining column in this
dataset, \code{variable.correction.factos}, allows the user to specify a
value by which the variable should be scaled. If this is not required this
should be set to 1.

If there are unidentified sightings in the dataset then the
\code{unidentified.sightings} argument should be \code{true} and a
\code{species.code.definitions} list should be provided. This list must
contain one element for every unidentified species code which should be
named according to this code. Each element will contain a vector of
identified species codes corresponding to those species which the
unidentified code could have potentially been. This function uses this
information to prorate the abundance estimated from the unidentified species
codes to the relevant abundances from the identified codes. The prorating is
done individually for each strata. The function can be forced not to prorate
to any given species in any selected strata using the \code{species.presence}
argument. This is a list containing one element for each strata, each must be
named using the appropriate strata name. Each element should contain a vector
of identified species codes corresponding to which species are present in
each strata.
}
\examples{

#Load the example data
data("mads.data")
ddf.data <- mads.data$dist.data
region.table <- mads.data$region.table
sample.table <- mads.data$sample.table
obs.table <- mads.data$obs.table

# Fit candidate detection function models using ddf in mrds
# Fit a half normal model
df.all.hn <- ddf(dsmodel = ~mcds(key = "hn", formula = ~ 1),
                 method='ds', data=ddf.data, meta.data=list(width=1))
summary(df.all.hn)
plot(df.all.hn)
# Fit a hazard rate model
df.all.hr <- ddf(dsmodel = ~mcds(key = "hn", formula = ~ 1),
                 method='ds', data=ddf.data, meta.data=list(width=1))
summary(df.all.hr)
plot(df.all.hr)

# Set up mads data:

# A vector of the species names
species.codes <- c("CD", "WSD", "Unid")

# A list defining which species the unidentified categories could be
unid.defs <- list("Unid" = c("CD", "WSD"))

# Specify which models are to be tried for each species code
mod.uncert <- list("CD" = c("df.all.hn", "df.all.hr"),
                   "WSD" = c("df.all.hn", "df.all.hr"),
                   "Unid" = c("df.all.hn", "df.all.hr"))
                   
# Provide the models in a named list and the selection criteria
models <- list("df.all.hn" = df.all.hn,
               "df.all.hr" = df.all.hr)
model.opts <- list(criterion = "AIC")

\donttest{
# Bootstrap options
bootstrap.opts <- list(resample = 'samples', n=999)

#Warning this will take some time to run!
results<- execute.multi.analysis( species.code = species.codes,
                                  unidentified.sightings = unid.defs,
                                  models.by.species.code = mod.uncert, 
                                  ddf.model.objects = models, 
                                  ddf.model.options = model.opts, 
                                  region.table = region.table, 
                                  sample.table = sample.table, 
                                  obs.table = obs.table, 
                                  bootstrap = TRUE, 
                                  bootstrap.option = bootstrap.opts)
}

#Short example to run as per CRAN requirements - 
#   warning only 1 repetition, results not interpretable!
bootstrap.opts <- list(resample = 'samples', n=1)

results<- execute.multi.analysis( species.code = species.codes,
                                  unidentified.sightings = unid.defs,
                                  models.by.species.code = mod.uncert, 
                                  ddf.model.objects = models, 
                                  ddf.model.options = model.opts, 
                                  region.table = region.table, 
                                  sample.table = sample.table, 
                                  obs.table = obs.table, 
                                  bootstrap = TRUE, 
                                  bootstrap.option = bootstrap.opts)

#These are simulated data and true abundances are: 
#  CD (common dolphins) = 3000 
#  WSD (white sided dolphins) =  1500
              
summary(results)                                   

}
\references{
Marques, F.F.C. and S.T. Buckland. 2004. Covariate models for the detection
    function. In: Advanced Distance Sampling, eds. S.T. Buckland,
    D.R.Anderson, K.P. Burnham, J.L. Laake, D.L. Borchers, and L. Thomas.
    Oxford University Press.
  Gerodette, T. and Forcada, J. 2005 Non-recovery of two spotted and spinner
    dolphin populations in the eastern tropical Pacific Ocean. Marine Ecology
    Progress Series, 291:1-21.
}
\author{
Laura Marshall
}
\keyword{Model}
\keyword{Statistical}
