% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/innerEstim.R
\name{GSCA}
\alias{GSCA}
\title{Generalized structured component analysis (GSCA) weights}
\description{
When used with \code{\link{weightFun.pls}}, \code{\link{innerEstim.gsca}} and 
\code{\link{outerEstim.gsca}} implement the generalized structured component analysis
indicator weighting system. Using \code{\link{weightFun.optim}} with the
\code{\link{optimCrit.gsca}} optimization criterion provides an anternative
approach to calculate GSCA weights by direct numerical minimization of the
GSCA criterion function.
}
\details{
The two step GSCA weight algorithm is designed to minimize.

\code{SS(ZV-ZWA)}

The parameter matrix \code{A} contains all model parameters including
\code{inner}, reflective \code{inner}, and \code{formative}. The weight
matrices \code{V} and \code{W}, which can contain duplicate elements,
contain the indicator weights.

The first step of GSCA estimation method is calculation of regressions
coefficients \code{A} given weights \code{W} and \code{V}. The function
\code{\link{innerEstim.gsca}} update the part of \code{A} corresponding to 
regressions between the composites, corresponding to \code{E} matrix in 
matrixpls. The regressions between composites and indicators are estimated
in \code{\link{outerEstim.gsca}}.

This algorithm for estimating the relationships between the composites
is therefore identical to the PLS path weighting scheme with
the exception that correlations are not used for inverse relationships and
there is no falling back to identity scheme for composites that are not
connected to other composites
The second step of GSCA is calculating a new set of weights conditional on
the regression coeffcients \code{A} to minimize the sum of error terms in
the regressions. This step is implemented in \code{\link{outerEstim.gsca}} after
updating the regresions between indicators and composites.
The implementation of GSCA in \pkg{matrixpls} differs from the Hwang & Takane (2004)
version in that during the first step, only regressions between composites are
estimated. The regressions between composites and indicators are estimated by
the second stage 
the indicators and compositess. Since these covariances need to be calculated in the second step, it is more
efficient to not calculate them during the first step.

A part of the code for \code{\link{outerEstim.gsca}} is adopted from the \pkg{ASGCA} package, licenced
under GPL3.
}
\examples{
if(!require(ASGSCA)){
    print("This example requires the ASGSCA package from Bioconductor")
} else{
# Run the example from ASGSCA package using GSCA estimation

data(GenPhen)
W0 <- matrix(c(rep(1,2),rep(0,8),rep(1,2),rep(0,8),rep(1,3),rep(0,7),rep(1,2)),
             nrow=8,ncol=4)
B0 <- matrix(c(rep(0,8),rep(1,2),rep(0,3),1,rep(0,2)),nrow=4,ncol=4)

# Set seed becayse ASGSCA uses random numbers as starting values 
set.seed(1)

GSCA.res <-GSCA(GenPhen,W0, B0,estim=TRUE,path.test=FALSE, 
                 latent.names=c("Gene1","Gene2",
                                "Clinical pathway 1",
                                "Clinical pathway 2"))

# Setup matrixpls to estimate the same model. Note that ASGSCA places dependent
# variables on columns but matrixpls uses rows for dependent variables

inner <- t(B0)
formative <- t(W0)
reflective <- matrix(0,8,4)

colnames(formative) <- rownames(reflective) <- names(GenPhen)

colnames(inner) <- rownames(inner) <- 
  rownames(formative) <- colnames(reflective) <-
  c("Gene1","Gene2","Clinical pathway 1","Clinical pathway 2")

model <- list(inner = inner, 
              reflective = reflective,
              formative = formative)

# Estimate using alternating least squares

matrixpls.res1 <- matrixpls(cov(GenPhen),  model,
                            outerEstim = outerEstim.gsca,
                            innerEstim = innerEstim.gsca)

# Estimate using direct minimization of the estimation criterion
# Set the convergence criterion to be slightly stricter than normally
# to get indentical results

matrixpls.res2 <- matrixpls(cov(GenPhen),  model,
                            weightFun = weightFun.optim,
                            optimCrit = optimCrit.gsca,
                            control = list(reltol = 1e-12))

# Compare the weights

do.call(cbind,lapply(list(ASGSCA =GSCA.res[["Weight"]],
                          matrixpls_als = t(attr(matrixpls.res1,"W")),
                          matrixpls_optim =t(attr(matrixpls.res2,"W"))),
                     function(W) W[W!=0]))


# Check the criterion function values

optimCrit.gsca(matrixpls.res1)
optimCrit.gsca(matrixpls.res2)

}
}
\author{
Mikko Rönkkö, Hela Romdhani, Stepan Grinek, Heungsun Hwang, Aurelie Labbe.
}
\references{
Hwang, H., & Takane, Y. (2004). Generalized structured component analysis. Psychometrika, 69(1), 81–99. doi:10.1007/BF02295841

Hela Romdhani, Stepan Grinek, Heungsun Hwang and Aurelie Labbe. (2014). ASGSCA: Association Studies for multiple SNPs and multiple traits using
Generalized Structured Equation Models. R package version 1.4.0.
}

