% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sjl.R
\name{sjl_sc}
\alias{sjl_sc}
\alias{sjl_sc_rel}
\title{Compute Jankowski's MCTQ sleep-corrected social jetlag}
\usage{
sjl_sc(so_w, se_w, so_f, se_f, abs = TRUE, method = "shorter")

sjl_sc_rel(so_w, se_w, so_f, se_f, method = "shorter")
}
\arguments{
\item{so_w}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time
of sleep onset on workdays} from a standard, micro, or shift version of
the MCTQ questionnaire. You can use \code{\link[=so]{so()}} to compute it for
the standard or shift version.}

\item{se_w}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time
of sleep end on workdays} from a standard, micro, or shift version of the
MCTQ questionnaire.}

\item{so_f}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time
of sleep onset on work-free days} from a standard, micro, or shift version
of the MCTQ questionnaire. You can use \code{\link[=so]{so()}} to compute it
for the standard or shift version.}

\item{se_f}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time
of sleep end on work-free days} from a standard, micro, or shift version
of the MCTQ questionnaire.}

\item{abs}{(optional) a \code{\link[base:logical]{logical}} object indicating if the
function must return an absolute value (default: \code{TRUE}).}

\item{method}{(optional) a string indicating which method the function must
use to compute the social jetlag. See the Methods section to learn
more (default: \code{"shorter"}).}
}
\value{
\itemize{
\item If \code{abs = TRUE}, a \code{\link[lubridate:duration]{Duration}} object corresponding
to the absolute sleep-corrected social jetlag.
\item If \code{abs = FALSE}, a \code{\link[lubridate:duration]{Duration}} object
corresponding to the relative sleep-corrected social jetlag.
}

The output may also vary depending on the \code{method} used.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{sjl_sc()} computes the \strong{Jankowski's (2017) sleep-corrected social jetlag}
for standard, micro, and shift versions of the Munich ChronoType
Questionnaire (MCTQ).

\code{sjl_sc_rel()} is just a wrapper for \code{sjl_sc()} with \code{abs = FALSE}.

Please note that the Jankowski (2017) did not proposed a "relative"
sleep-corrected social jetlag, but the user may consider using it.
}
\details{
\strong{Standard MCTQ} functions were created following the guidelines in
Roenneberg, Wirz-Justice, & Merrow (2003), Roenneberg, Allebrandt, Merrow, &
Vetter (2012), and from The Worldwide Experimental Platform (theWeP, n.d.).

\strong{\eqn{\mu}MCTQ} functions were created following the guidelines in Ghotbi
et al. (2020), in addition to the guidelines used for the standard MCTQ.

\strong{MCTQ\eqn{^{Shift}}{ Shift}} functions were created following the
guidelines in Juda, Vetter, & Roenneberg (2013), in addition to the
guidelines used for the standard MCTQ.

See the References section to learn more.
\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the
\link[lubridate:lubridate-package]{lubridate} and \link[hms:hms-package]{hms}
packages. Please refer to those package documentations to learn more about
them.
}

\subsection{Rounding and fractional time}{

Some operations may produce an output with fractional time (e.g.,
\code{"19538.3828571429s (~5.43 hours)"}, \code{01:15:44.505}). If you want, you
can round it with \code{\link[=round_time]{round_time()}}.

Our recommendation is to avoid rounding, but, if you do, make sure that you
only round your values after all computations are done. That way you avoid
\href{https://en.wikipedia.org/wiki/Round-off_error}{round-off errors}.
}
}
\section{Guidelines}{


In an article published in 2017, Konrad S. Jankowski argued that the original
formula for computing the social jetlag (\eqn{SJL}) captures not only the
misalignment between social and biological time, but also the sleep debt
resulting from sleep deprivation during workdays. Jankowski then proposed the
following guideline for a sleep-corrected social jetlag
(\eqn{SJL_{sc}}{SJL_sc}) computation.
\subsection{Notes}{
\itemize{
\item The Jankowski's alternative is disputed. We recommend seeing
Roenneberg, Pilz, Zerbini, & Winnebeck (2019) discussion about it
(see item 3.4.2).
\item For MCTQ\eqn{^{Shift}}{ Shift}, the computation below must be applied to
each shift section of the questionnaire.
\item Due to time arithmetic issues, \code{sjl_sc()} does a slightly different
computation by default than those proposed by the author mentioned above.
See \code{vignette("sjl-computation", package = "mctq")} for more details.
\item If you are visualizing this documentation in plain text, you may have some
trouble understanding the equations. You can see this documentation on the
package \href{https://docs.ropensci.org/mctq/reference/}{website}.
}
}

\subsection{For standard and micro versions of the MCTQ}{

\strong{\deqn{\textrm{If } SD_W > SD_F \; \& \; SE_W \leq SE_F \; , \; |
SE_F - SE_W |}{If SD_W > SD_F & SE_W <= SE_F, | SE_F - SE_W |}}
\strong{\deqn{\textrm{Else } \; , \; | SO_F - SO_W |}{Else, | SO_F - SO_W |}}

Where:
\itemize{
\item \eqn{SO_W} = Local time of sleep onset on workdays.
\item \eqn{SE_W} = Local time of sleep end on workdays.
\item \eqn{SO_F} = Local time of sleep onset on work-free days.
\item \eqn{SE_F} = Local time of sleep end on work-free days.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days.
}

\subsection{For the shift version of the MCTQ}{

\strong{\deqn{\textrm{If } SD_W^{M/E/N} > SD_F^{M/E/N} \; \& \; SE_W^{M/E/N}
\leq SE_F^{M/E/N} \; , \; | SE_F^{M/E/N} - SE_W^{M/E/N} |}{
If SD_W_M/E/N > SD_F_M/E/N & SE_W_M/E/N <= SE_F_M/E/N, | SE_F_M/E/N -
SE_W_M/E/N |}}
\strong{\deqn{\textrm{Else } \; , \; | SO_F^{M/E/N} - SO_W^{M/E/N} |}{
Else, | SO_F_M/E/N - SO_W_M/E/N |}}

Where:
\itemize{
\item \eqn{SO_W^{M/E/N}} = Local time of sleep onset between two days in a
particular shift.
\item \eqn{SE_W^{M/E/N}} = Local time of sleep end between two days in a
particular shift.
\item \eqn{SO_F^{M/E/N}} = Local time of sleep onset between two free days after
a particular shift.
\item \eqn{SE_F^{M/E/N}} = Local time of sleep end between two free days after a
particular shift.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days, \eqn{M} =
Morning shift; \eqn{E} = Evening shift; \eqn{N} = Night shift.
}
}

\section{Methods for computing the sleep-corrected social jetlag}{


There are different approaches to compute the sleep-corrected social jetlag
(\eqn{SJL_{sc}}{SJL_sc}). By default, \code{sjl_sc()} uses an approach that we
call "the shorter interval approach" (\code{"shorter"}).

The topics below provide a simple explanation of each method supported by
\code{sjl_sc()}. To get a detail understating of this methods, see
\code{vignette("sjl-computation", package = "mctq")}.
\itemize{
\item \code{"difference"}
}

By using \code{method = "difference"}, \code{sjl_sc()} will do the exact computation
proposed by Jankowski, i.e., \eqn{SJL_{sc}}{SJL_sc} will be computed as the
linear difference between \eqn{SO_f}/\eqn{SE_f} and \eqn{SO_W}/\eqn{SE_W}
(see the
Guidelines section).

\strong{We do not recommend using this method}, as it has many limitations.
\itemize{
\item \code{"shorter"}
}

This is the default method for \code{sjl_sc()}. It's based on the shorter interval
between \eqn{SO_f}/\eqn{SE_f} and \eqn{SO_W}/\eqn{SE_W}, solving most of the
issues relating to \eqn{SJL_{sc}}{SJL_sc} computation.
\itemize{
\item \code{"longer"}
}

The \code{"longer"} method uses the same logic of the \code{"shorter"} method, but,
instead of using the shorter interval between \eqn{SO_f}/\eqn{SE_f} and
\eqn{SO_W}/\eqn{SE_W}, it uses the longer interval between the two,
considering a two-day window.

This method may help in special contexts, like when dealing with
shift-workers that have a greater than 12 hours distance between their
sleep hours.
}

\examples{
## Scalar example

so_w <- hms::parse_hm("02:00")
se_w <- hms::parse_hm("10:00")
so_f <- hms::parse_hm("01:00")
se_f <- hms::parse_hm("08:00")

sjl_sc(so_w, se_w, so_f, se_f)
#> [1] "3600s (~1 hours)" # Expected
sjl_sc(so_w, se_w, so_f, se_f, abs = FALSE)
#> [1] "-3600s (~-1 hours)" # Expected (negative sjl_sc)
sjl_sc_rel(so_w, se_w, so_f, se_f) # Wrapper function
#> [1] "-3600s (~-1 hours)" # Expected (negative sjl_sc)
sjl(msl(so_w, sdu(so_w, se_w)), msl(so_f, sdu(so_f, se_f)))
#> [1] "5400s (~1.5 hours)" # Expected

so_w <- hms::parse_hm("22:00")
se_w <- hms::parse_hm("06:00")
so_f <- hms::parse_hm("01:00")
se_f <- hms::parse_hm("06:00") # sd_w > sd_f & se_w <= se_f

sjl_sc(so_w, se_w, so_f, se_f) # sjl_sc = | se_f - se_w |
#> [1] "0s" # Expected
sjl_sc(so_w, se_w, so_f, se_f, abs = FALSE)
#> [1] "0s" # Expected
sjl_sc_rel(so_w, se_w, so_f, se_f) # Wrapper function
#> [1] "0s" # Expected
sjl(msl(so_w, sdu(so_w, se_w)), msl(so_f, sdu(so_f, se_f)))
#> [1] "5400s (~1.5 hours)" # Expected

so_f <- hms::as_hms(NA)

sjl_sc(so_w, se_w, so_f, se_f)
#> [1] NA # Expected

## Vector example

so_w <- c(hms::parse_hm("00:00"), hms::parse_hm("01:00"))
se_w <- c(hms::parse_hm("08:00"), hms::parse_hm("07:00"))
so_f <- c(hms::parse_hm("01:00"), hms::parse_hm("01:00"))
se_f <- c(hms::parse_hm("09:00"), hms::parse_hm("09:00"))

sjl_sc(so_w, se_w, so_f, se_f)
#> [1] "3600s (~1 hours)" "0s" # Expected
sjl_sc(so_w, se_w, so_f, se_f, abs = FALSE)
#> [1] "3600s (~1 hours)" "0s" # Expected
sjl_sc_rel(so_w, se_w, so_f, se_f) # Wrapper function
#> [1] "3600s (~1 hours)" "0s" # Expected
sjl(msl(so_w, sdu(so_w, se_w)), msl(so_f, sdu(so_f, se_f)))
#> [1] "3600s (~1 hours)" "3600s (~1 hours)" # Expected

## See other examples in '?sjl()'
}
\references{
Ghotbi, N., Pilz, L. K., Winnebeck, E. C., Vetter, C., Zerbini, G., Lenssen,
D., Frighetto, G., Salamanca, M., Costa, R., Montagnese, S., & Roenneberg, T.
(2020). The \eqn{\mu}MCTQ: an ultra-short version of the Munich ChronoType
Questionnaire. \emph{Journal of Biological Rhythms}, \emph{35}(1), 98-110.
\doi{10.1177/0748730419886986}.

Jankowski K. S. (2017). Social jet lag: sleep-corrected formula.
\emph{Chronobiology International}, \emph{34}(4), 531-535.
\doi{10.1080/07420528.2017.1299162}.

Juda, M., Vetter, C., & Roenneberg, T. (2013). The Munich ChronoType
Questionnaire for shift-workers (MCTQ\eqn{^{Shift}}{ Shift}). \emph{Journal of
Biological Rhythms}, \emph{28}(2), 130-140. \doi{10.1177/0748730412475041}.

Roenneberg T., Allebrandt K. V., Merrow M., & Vetter C. (2012). Social jetlag
and obesity. \emph{Current Biology}, \emph{22}(10), 939-43.
\doi{10.1016/j.cub.2012.03.038}.

Roenneberg, T., Pilz, L. K., Zerbini, G., & Winnebeck, E. C. (2019).
Chronotype and social jetlag: a (self-) critical review. \emph{Biology}, \emph{8}(3),
54. \doi{10.3390/biology8030054}.

Roenneberg, T., Wirz-Justice, A., & Merrow, M. (2003). Life between clocks:
daily temporal patterns of human chronotypes. \emph{Journal of Biological
Rhythms}, \emph{18}(1), 80-90. \doi{10.1177/0748730402239679}.

The Worldwide Experimental Platform (n.d.). MCTQ.
\url{https://www.thewep.org/documentations/mctq/}
}
\seealso{
Other MCTQ functions: 
\code{\link{fd}()},
\code{\link{gu}()},
\code{\link{le_week}()},
\code{\link{msf_sc}()},
\code{\link{msl}()},
\code{\link{napd}()},
\code{\link{sd24}()},
\code{\link{sd_overall}()},
\code{\link{sd_week}()},
\code{\link{sdu}()},
\code{\link{sjl_weighted}()},
\code{\link{sjl}()},
\code{\link{so}()},
\code{\link{tbt}()}
}
\concept{MCTQ functions}
