% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RidgePEER_cv.R
\name{RidgePEER.cv}
\alias{RidgePEER.cv}
\title{Graph-constrained estimation with regulariazation parameter selection via Cross-Validation}
\usage{
RidgePEER.cv(Q, y, Z, X = NULL, Q.normalized = FALSE, r.const = 0.001,
  lambda.Q.grid = exp(seq(-6, 6, length.out = 10)),
  lambda.2.grid = exp(seq(-6, 6, length.out = 10)), K = 10,
  verbose = FALSE)
}
\arguments{
\item{Q}{graph-originated penalty matrix \eqn{(p \times p)}, typically: a graph Laplacian matrix}

\item{y}{response values matrix \eqn{(n \times 1)}}

\item{Z}{design matrix \eqn{(n \times p)} modeled as random effects variables (to be penalized in regression modeling)}

\item{X}{design matrix \eqn{(n \times k)} modeled as fixed effects variables (not to be penalized in regression modeling)}

\item{Q.normalized}{logical whether or not to use normalized version of a graph-originated penalty matrix}

\item{r.const}{scalar value of regularization parameter for a Ridge term by which matrix \code{Q} is corrected 
(used in situations when current value of \code{lambda.2} Cross-Validation parameter grid is less than it)}

\item{lambda.Q.grid}{vector of \code{lambda.Q} values to iterate over in Cross-Validation procedure}

\item{lambda.2.grid}{vector of \code{lambda.2} values to iterate over in Cross-Validation procedure}

\item{K}{number of Cross-Validation procedure folds}

\item{verbose}{logical whether or not message out information from function execution}
}
\value{
\item{b.est}{vector with estimated values of \eqn{b} coefficients}
\item{beta.est}{vector with estimated values of \eqn{\beta} coefficients}
\item{ERR.mat.cv}{matrix of MSE errors obtained in Cross-Validation for each parameter grids combination considered}
\item{lambda.Q.cv}{value of \eqn{\lambda_Q} regularization parameter selected via Cross-Validation}
\item{lambda.R.cv}{\code{lambda.Q.cv} * \code{lambda.2.cv} value}
\item{lambda.2.cv}{value of \eqn{\lambda_2} regularization parameter selected via Cross-Validation}
}
\description{
Perform penalized estimation where penalty term is a linear combination of 
graph-originated penalty and Ridge-originated penalty terms. Corresponding regularization
parameter values \eqn{\lambda} (\eqn{\lambda_Q, \lambda_R}) are selected via K-fold Cross-Validation
procedure. 

Assumes MSE of response variable values prediction to be minimized in the Cross-Validation procedure.
}
\examples{
# Example 1. 

set.seed(1234)
# graph Adjacency matrix
n <- 200 
p1 <- 10
p2 <- 90
p <- p1 + p2
A <- matrix(rep(0, p*p), nrow = p, ncol = p)
A[1:p1, 1:p1] <- 1
A[(p1+1):p, (p1+1):p] <- 1
# graph Laplacian matrix
L <- Adj2Lap(A)
L.norm <- L2L.normalized(L)
# Z design matrix 
Z <- matrix(rnorm(n*p), nrow = n, ncol = p)
# True b coefficients 
b.true<- c(rep(1, p1), rep(0, p2))
beta.true <- runif(3)
intercept <- 0
eta <- intercept + Z \%*\% b.true 
R2 <- 0.5 # assumed variance explained 
sd.eps <- sqrt(var(eta) * (1 - R2) / R2)
error <- rnorm(n, sd = sd.eps)
# Y observed 
Y <- eta + error

# model estimation
RidgePEER.cv.fit <- RidgePEER.cv(Q = L.norm, y = Y, Z = Z, X = NULL,
    lambda.Q.grid = exp(seq(-6, 6, length.out = 2)),
    lambda.2.grid = exp(seq(-6, 6, length.out = 2)))

# b coefficient estimates
RidgePEER.cv.fit$b.est
# regularization parameters values
RidgePEER.cv.fit$lambda.Q
RidgePEER.cv.fit$lambda.2 
RidgePEER.cv.fit$lambda.R # lambda.R = lambda.Q * lambda.2


# Example 2.: model with non-penalized covariates 

# X design matrix (covariates which are not to be penalized)
X <- matrix(rnorm(n*3), nrow = n, ncol = 3)
beta.true <- runif(3)
intercept <- 0
eta <- intercept + Z \%*\% b.true + X \%*\% beta.true
R2 <- 0.5 # assumed variance explained 
sd.eps <- sqrt(var(eta) * (1 - R2) / R2)
error <- rnorm(n, sd = sd.eps)
# Y observed 
Y <- eta + error

# model estimation
RidgePEER.cv.fit <- RidgePEER.cv(Q = L.norm, y = Y, Z = Z, X = X,
    lambda.Q.grid = exp(seq(-6, 6, length.out = 2)),
    lambda.2.grid = exp(seq(-6, 6, length.out = 2)))

# b coefficient estimates
RidgePEER.cv.fit$b.est
# beta coefficient estimates
RidgePEER.cv.fit$beta.est
# regularization parameters values
RidgePEER.cv.fit$lambda.Q
RidgePEER.cv.fit$lambda.2 
RidgePEER.cv.fit$lambda.R # lambda.R = lambda.Q * lambda.2


}

