% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crPEER.R
\name{crPEER}
\alias{crPEER}
\title{Graph-constrained regression with additional Ridge term to handle the non-invertibility of 
a graph-originated penalty matrix}
\usage{
crPEER(Q, y, Z, X = NULL, lambda.2 = 0.001, Q.normalized = FALSE,
  compute.Grace = FALSE)
}
\arguments{
\item{Q}{graph-originated penalty matrix \eqn{(p \times p)}, typically: a graph Laplacian matrix}

\item{y}{response values matrix \eqn{(n \times 1)}}

\item{Z}{design matrix \eqn{(n \times p)} modeled as random effects variables (to be penalized in regression modeling)}

\item{X}{design matrix \eqn{(n \times k)} modeled as fixed effects variables (not to be penalized in regression modeling)}

\item{lambda.2}{scalar value of regularization parameter for additional Ridge term by which matrix \code{Q} is corrected}

\item{Q.normalized}{logical whether or not to use normalized version of a graph-originated penalty matrix}

\item{compute.Grace}{logical whether or not to compute \href{https://arxiv.org/abs/1506.08339}{Grace test} 
(a significance test for graph-constrained estimation) results}
}
\value{
\item{b.est}{vector with estimated values of \eqn{b} coefficients}
\item{beta.est}{vector with estimated values of \eqn{\beta} coefficients}
\item{lambda.Q}{value of \eqn{\lambda_Q} regularization parameter}
\item{lambda.R}{\code{lambda.Q} * \code{lambda.2} value}
\item{lambda.2}{assumed "Ridge correction" \code{lambda.2} fixed regularization parameter value}
\item{grace.test.res}{output from the Grace significance test for graph-constrained estimation computation}
}
\description{
Graph-constrained regression with additional Ridge term to handle the non-invertibility of 
a graph-originated penalty matrix
}
\examples{
# Example 1. 

set.seed(1234)
# graph Adjacency matrix
n <- 200 
p1 <- 10
p2 <- 90
p <- p1 + p2
A <- matrix(rep(0, p*p), nrow = p, ncol = p)
A[1:p1, 1:p1] <- 1
A[(p1+1):p, (p1+1):p] <- 1
# graph Laplacian matrix
L <- Adj2Lap(A)
L.norm <- L2L.normalized(L)
# Z design matrix 
Z <- matrix(rnorm(n*p), nrow = n, ncol = p)
# True b coefficients 
b.true<- c(rep(1, p1), rep(0, p2))
beta.true <- runif(3)
intercept <- 0
eta <- intercept + Z \%*\% b.true 
R2 <- 0.5 # assumed variance explained 
sd.eps <- sqrt(var(eta) * (1 - R2) / R2)
error <- rnorm(n, sd = sd.eps)
# Y observed 
Y <- eta + error

# model estimation
crPEER.fit <- crPEER(Q = L.norm, y = Y, Z = Z, X = NULL)

# b coefficient estimates
crPEER.fit$b.est
# regularization parameters values
crPEER.fit$lambda.Q
crPEER.fit$lambda.2 # assumed to be a fixed constant 
crPEER.fit$lambda.R # lambda.R = lambda.Q * lambda.2


# Example 2.: model with non-penalized covariates

# X design matrix (covariates which are not to be penalized)
X <- matrix(rnorm(n*3), nrow = n, ncol = 3)
beta.true <- runif(3)
intercept <- 0
eta <- intercept + Z \%*\% b.true + X \%*\% beta.true
R2 <- 0.5 # assumed variance explained 
sd.eps <- sqrt(var(eta) * (1 - R2) / R2)
error <- rnorm(n, sd = sd.eps)
# Y observed 
Y <- eta + error

crPEER.fit <- crPEER(Q = L.norm, y = Y, Z = Z, X = X)

# b coefficient estimates
crPEER.fit$b.est
# beta coefficient estimates
crPEER.fit$beta.est
# regularization parameters values
crPEER.fit$lambda.Q
crPEER.fit$lambda.2 # assumed to be a fixed constant 
crPEER.fit$lambda.R # lambda.R = lambda.Q * lambda.2

}
\references{
Brumback, B. A., Ruppert, D., Wand, M. P., Comment on 'Variable selection and function estimation in
additive nonparametric regression using a data-based prior'. Journal of the American Statistical
Association (1999): 94, 794–797.

Karas, M., Brzyski, D., Randolph, T., Harezlak, D. Brain connectivity-informed regularization methods for regression. Paper in progress, to be submited as an invited paper on CCNS for a special issue of  Statistics in Biosciences by Nov 30, 2016 (reference will be updated).

Li, C., Li, H., Network-constrained regularization and variable selection for analysis of genomic data. 
Bioinformatics (2008): 24(9), 1175-1182.

Li, C., Li, H., Variable selection and regression analysis for graph-structured covariates with an application to genomics. 
The Annals of Applied Statistics (2010): 4(3), 1498–1516.

Randolph, T., Harezlak, J., Feng, Z., Structured penalties for functional linear models—partially empirical 
eigenvectors for regression. The Electronic Journal of Statistics (2012): 6, 323-353.
}

