% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mies_methods.R
\name{mies_select_from_archive}
\alias{mies_select_from_archive}
\title{Select Individuals from an OptimInstance}
\usage{
mies_select_from_archive(
  inst,
  n_select,
  rows,
  selector = SelectorBest$new()$prime(inst$search_space),
  group_size = 1,
  get_indivs = TRUE
)
}
\arguments{
\item{inst}{(\code{OptimInstance})\cr
Optimization instance to evaluate.}

\item{n_select}{(\code{integer(1)})\cr
Number of individuals to select.}

\item{rows}{optional (\code{integer})\cr
Indices of rows within \code{inst} to consider. If this is not given, then the entire archive is used.}

\item{selector}{(\code{\link{Selector}})\cr
\code{\link{Selector}} operator that selects individuals depending on configuration values
and objective results. When \code{selector$operate()} is called, then objectives that
are being minimized are multiplied with -1 (through \code{\link[=mies_get_fitnesses]{mies_get_fitnesses()}}), since \code{\link{Selector}}s always try to maximize fitness.
Defaults to \code{\link{SelectorBest}}.\cr
The \code{\link{Selector}} must be primed on a superset of \code{inst$search_space}; this \emph{includes} the "budget" component
when performing multi-fidelity optimization. All components on which \code{selector} is primed on must occur in the archive.\cr
The given \code{\link{Selector}} \emph{may} return duplicates.}

\item{group_size}{(\code{integer})\cr
Sampling group size hint, indicating that the caller would prefer there to not be any duplicates within this group size.
The \code{\link{Selector}} may or may not ignore this value, however.
This may possibly happen because of certain configuration parameters, or because the input size is too small.\cr
Must either be a scalar value or sum up to \code{n_select}. Must be non-negative. A scalar value of 0 is interpreted the same as 1.\cr
Default is 1.}

\item{get_indivs}{(\code{logical(1)})\cr
Whether to return configuration values from within the archive (\code{TRUE}) or just the indices within
the archive (\code{FALSE}). Default is \code{TRUE}.}
}
\value{
\code{integer} | \code{\link[data.table:data.table]{data.table}}: Selected individuals, either index into \code{inst} or subset of archive table,
depending on \code{get_indivs}.
}
\description{
Apply a \code{\link{Selector}} operator to a subset of configurations inside
an \code{OptimInstance}
and return the index within the archive (when \code{get_indivs} \code{FALSE}) or the configurations themselves
(when \code{get_indivs} is \code{TRUE}).

It is not strictly necessary for the selector to select unique individuals / individuals without replacement.

Individuals are selected independently of whether they are "alive" or not. To select only from alive individuals,
set \code{rows} to \code{inst$archive$data[, which(is.na(eol))]}.
}
\examples{
set.seed(1)
library("bbotk")
lgr::threshold("warn")

# Define the objective to optimize
objective <- ObjectiveRFun$new(
  fun = function(xs) {
    z <- exp(-xs$x^2 - xs$y^2) + 2 * exp(-(2 - xs$x)^2 - (2 - xs$y)^2)
    list(Obj = z)
  },
  domain = ps(x = p_dbl(-2, 4), y = p_dbl(-2, 4)),
  codomain = ps(Obj = p_dbl(tags = "maximize"))
)

# Get a new OptimInstance
oi <- OptimInstanceSingleCrit$new(objective,
  terminator = trm("evals", n_evals = 100)
)

s = sel("best")
s$prime(oi$search_space)

mies_init_population(inst = oi, mu = 6)

oi$archive

# Default: get individuals
mies_select_from_archive(oi, n_select = 2, rows = 1:6, selector = s)

# Alternatively: get rows within archive
mies_select_from_archive(oi, n_select = 2, rows = 1:6, selector = s,
  get_indivs = FALSE)

# Rows gotten from archive are relative from *all* rows, not from archive[rows]:
mies_select_from_archive(oi, n_select = 2, rows = 3:6, selector = s,
  get_indivs = FALSE)

##
# When using additional components: mies_select_from_archive learns about
# additional components from primed selector.

# Get a new OptimInstance
oi <- OptimInstanceSingleCrit$new(objective,
  terminator = trm("evals", n_evals = 100)
)

mies_init_population(inst = oi, mu = 6,
  additional_component_sampler = Sampler1DRfun$new(
    param = ps(additional = p_dbl(-1, 1)), rfun = function(n) -1
  )
)

oi$archive

# Wrong: using selector primed only on search space. The resulting
# individuals do not have the additional component.
mies_select_from_archive(oi, n_select = 2, rows = 1:6, selector = s)

# Correct: selector must be primed on search space + additional component
mies_prime_operators(oi$search_space, selectors = list(s),
  additional_components = ps(additional = p_dbl(-1, 1)))

mies_select_from_archive(oi, n_select = 2, rows = 1:6, selector = s)
}
\seealso{
Other mies building blocks: 
\code{\link{mies_evaluate_offspring}()},
\code{\link{mies_generate_offspring}()},
\code{\link{mies_get_fitnesses}()},
\code{\link{mies_init_population}()},
\code{\link{mies_step_fidelity}()},
\code{\link{mies_survival_comma}()},
\code{\link{mies_survival_plus}()}
}
\concept{mies building blocks}
