#' @title Terminator that Counts OptimizerMies Generations
#'
#' @name mlr_terminators_gens
#'
#' @description
#' [`Terminator`][bbotk::Terminator] that terminates after a given number of generations have passed in [`OptimizerMies`].
#'
#' If [`OptimizerMies`] is started on an archive that already has evaluated configurations, these evaluations count as
#' generation 0. If an initial, randomly sampled generation is generated by [`OptimizerMies`], it has generation number 1.
#' Setting `generation` to 1 therefore terminates after the evaluation of the initial sample, *unless* no initial sample is
#' generated by [`OptimizerMies`] and instead found in the archive. `generation` set to 0 avoids any evaluation within [`OptimizerMies`]
#' (but is ignored if no `dob` column is in the archive).
#'
#' When doing multi-fidelity optimization, and fidelity of a configuration is increased because of a step in the fidelity schedule,
#' or because they were sampled new and survived, then this fidelity refinement happens as part of an already started generation. This means
#' termination at this fidelity refinement step is avoided.
#'
#' @section Dictionary:
#' This [`Terminator`][bbotk::Terminator] can be created with the short access form [`trm()`][bbotk::trm] ([`trms()`][bbotk::trm] to get a list),
#' or through the [dictionary][mlr3misc::Dictionary] [`mlr_terminators`][bbotk::mlr_terminators] in the following way:
#'
#' ```
#' # preferred
#' trm("gens")
#' trms("gens")  # takes vector IDs, returns list of Terminators
#'
#' # long form
#' mlr_terminators$get("gens")
#' ```
#'
#' @section Configuration Parameters:
#' * `generations` :: `integer(1)`\cr
#'   Number of generations to evaluate, after which to stop. Not initialized and should be set to the desired value during construction.
#'
#' @examples
#' library("bbotk")
#' trm("gens", generations = 10)
#' @export
TerminatorGenerations = R6Class("TerminatorGenerations", inherit = Terminator,
  public = list(
    #' @description
    #' Initialize the `TerminatorGenerations` object.
    initialize = function() {
      param_set = ps(generations = p_int(0, special_vals = list(Inf), tags = "required"))
      super$initialize(id = "gens", param_set = param_set, properties = c("single-crit", "multi-crit"), unit = "generations")
    },

    #' @description
    #' Is `TRUE` if when the termination criterion is matched, `FALSE` otherwise.
    #' @param archive [`Archive`][bbotk::Archive]
    #'   Archive to check.
    #' @return `logical(1)`: Whether to terminate.
    is_terminated = function(archive) {
      assert_r6(archive, "Archive")
      max(archive$data$dob, 0) >= self$param_set$get_values()$generations
    }
  ),

  private = list(
    .status = function(archive) {
      c(
        max_steps = self$param_set$get_values()$generations,
        current_steps = max(archive$data$dob, 0)
      )
    }
  )
)

#' @title Get the Numger of Generations that a Terminator Allows
#'
#' @description
#' Get the number of generations of a [`TerminatorGenerations`]. When the [`TerminatorGenerations`]
#' is wrapped in a [`TerminatorCombo`][bbotk::TerminatorCombo], then the minimum number of generations
#' allowed by it are retrieved. This is the minimum of all `terminator_get_generations` if `$any` is set
#' to `TRUE`, and the maximum if `$any` is set to `FALSE`.
#'
#' The number of generations allowed by other [`Terminator`][bbotk::Terminator]s is infinity.
#'
#' @param x ([`Terminator`][bbotk::Terminator])\cr
#'   [`Terminator`][bbotk::Terminator] to query.
#' @return `numeric(1)`: The theoretical maximum number of generations allowed by the [`Terminator`][bbotk::Terminator].
#'
#' @export
terminator_get_generations = function(x) {
  UseMethod("terminator_get_generations")
}

#' @export
terminator_get_generations.default = function(x) {
  stop("Invalid terminator given.")
}

#' @export
terminator_get_generations.Terminator = function(x) Inf  # normie terminator not limiting generations in any way

#' @export
terminator_get_generations.TerminatorGenerations = function(x) x$param_set$values$generations

#' @export
terminator_get_generations.TerminatorCombo = function(x) {
  if (x$param_set$values$any) {
    # Terminate on "any" condition being true --> minimum of generations of child objects
    min(sapply(x$terminators, terminator_get_generations))
  } else {
    max(sapply(x$terminators, terminator_get_generations))
  }
}
