% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mies_methods.R
\name{mies_aggregate_single_generation}
\alias{mies_aggregate_single_generation}
\title{Aggregate a Value for a given Generation}
\usage{
mies_aggregate_single_generation(
  archive,
  fitness_aggregator,
  generation = NA,
  include_previous_generations = FALSE
)
}
\arguments{
\item{archive}{(\code{\link[bbotk:Archive]{Archive}})\cr
The archive over which to aggregate.}

\item{fitness_aggregator}{(\code{function})\cr
Aggregation function, called with information about alive individuals of each generation. See details.}

\item{generation}{(\code{numeric(1)})\cr
Generation for which to aggregate the value.
If \code{include_previous_generations} is \code{FALSE}, then an individual is considered to be alive at generation \code{i} if its \code{dob} is smaller or equal to \code{i}, and
if its \code{eol} is either \code{NA} or greater than \code{i}. If \code{include_previous_generations} is \code{TRUE}, then all individuals with \code{dob} smaller or equal to \code{i} are
considered.
If this is \code{NA}, the currently alive (\code{include_previous_generations} \code{FALSE}) or all (\code{include_previous_generations} \code{TRUE}) individuals are aggregated.
If multiple individuals considered "alive" with the same \code{x_id} are found, then only the last individual is used.
This excludes previous individuals that were re-evaluated with a different fidelity.}

\item{include_previous_generations}{(\code{logical(1)})\cr
Aggregate all individuals that were alive at \code{generation} or at any point before that.
Duplicates with the same \code{x_id} are removed, meaning that if an individual was re-evaluated with different fidelity, only the last
re-evaluation is counted.
However, note that individuals from different generations may still have been evaluated with different fidelity, so if
Default \code{FALSE}.}
}
\value{
The value returned by \code{fitness_aggregator} when applied to individuals alive at generation \code{generation}. If no
individuals of the requested generation are present, \code{fitness_aggregator} is not called
and \code{mies_aggregate_single_generation()} returns \code{NULL} instead.
}
\description{
Applies a \code{fitness_aggregator} function to the values that were alive in the archive at a given generation.
The function is supplied with the fitness values, and optionally other data, of all individuals that are alive at that point.
}
\details{
The \code{fitness_aggregator} function may have any of the following arguments, which will be given the following information when
\code{fitness_aggregator} is called:
\itemize{
\item \code{fitnesses} :: \code{matrix}\cr
Will contain fitnesses for each alive individual. This value has one column when doing single-crit optimization and one column for
each "criterion" when doing multi-crit optimization.
Fitnesses are always being maximized, so if an objective is being minimized, the \code{fitness_aggregator} function is given the objective values * -1.
\item \code{objectives_unscaled} :: \code{matrix}\cr
The objective values as given to \code{fitnesses}, but not multiplied by -1 if they are being minimized. It is recommended that
the \code{codomain} argument is queried for \code{"maximize"} or \code{"minimize"} tags when \code{objectives_unscaled} is used.
\item \code{budget} :: \code{scalar}\cr
If multi-fidelity evaluation is being performed, then this is the "budget" value of each individual. Otherwise it is a vector containing the value
1 for each individual.
\item \code{xdt} :: \code{data.table}\cr
The configurations that were evaluated for the alive individuals. Rows are in the same order as the values given to \code{fitnesses}
or \code{objectives_unscaled}.
\item \code{search_space} :: \code{\link[paradox:ParamSet]{ParamSet}}\cr
The search space of the \code{\link[bbotk:Archive]{Archive}} under evaluation.
\item \code{codomain} :: \code{\link[paradox:ParamSet]{ParamSet}}\cr
The codomain of the \code{\link[bbotk:Archive]{Archive}} under evaluation.
This is particularly useful when using \code{objectives_unscaled} to determine minimization or maximization.
}

Not all of these arguments need to present, but at least one of \code{fitnesses}, \code{objectives_unscaled}, or \code{xdt} must be.

\code{fitness_aggregator} will never be called for an empty generation.
}
\examples{
library("bbotk")
lgr::threshold("warn")

objective <- ObjectiveRFun$new(
  fun = function(xs) {
    list(y1 = xs$x1, y2 = xs$x2)
  },
  domain = ps(x1 = p_dbl(0, 1), x2 = p_dbl(-1, 0)),
  codomain = ps(y1 = p_dbl(0, 1, tags = "maximize"),
    y2 = p_dbl(-1, 0, tags = "minimize"))
)

oi <- OptimInstanceMultiCrit$new(objective, terminator = trm("none"))

try(mies_aggregate_single_generation(oi$archive, identity), silent = TRUE)

mies_aggregate_single_generation(oi$archive, function(fitnesses) fitnesses)


mies_init_population(oi, 2, budget_id = "x1", fidelity = .5)

oi$archive$data

mies_aggregate_single_generation(oi$archive, function(fitnesses) fitnesses)

# Notice how fitnesses are positive, since x2 is scaled with -1.
# To get the original objective-values, use objectives_unscaled:
mies_aggregate_single_generation(oi$archive,
  function(objectives_unscaled) objectives_unscaled)

# When `...` is used, all information is passed:
mies_aggregate_single_generation(oi$archive, function(...) names(list(...)))

# Generation 10 is not present, but individuals with eol `NA` are still
# considered alive:
mies_aggregate_single_generation(oi$archive, function(fitnesses) fitnesses,
  generation = 10)

# Re-evaluating points with higher "fidelity" (x1)
mies_step_fidelity(oi, budget_id = "x1", fidelity = 0.7)

oi$archive$data
# Lower-fidelity values are considered dead now, even for generation 1:
mies_aggregate_single_generation(oi$archive, function(fitnesses) fitnesses,
  generation = 1)

# This adds two new alive individuals at generation 2.
# Also the individuals from gen 1 are reevaluated with fidelity 0.8
mies_evaluate_offspring(oi, offspring = data.frame(x2 = c(-0.1, -0.2)),
  budget_id = "x1", fidelity = 0.9, reevaluate_fidelity = 0.8)

oi$archive$data

mies_aggregate_single_generation(oi$archive, function(budget, ...) budget)

mies_aggregate_single_generation(oi$archive, function(fitnesses) fitnesses,
  generation = 1)

mies_aggregate_single_generation(oi$archive, function(fitnesses) fitnesses,
  generation = 2)

# No individuals were killed, but some were fidelity-reevaluated.
# These are not present with include_previous_generations:
mies_aggregate_single_generation(oi$archive, function(fitnesses) fitnesses,
  generation = 2, include_previous_generations = TRUE)

# Typical use-case: get dominated hypervolume
mies_aggregate_single_generation(oi$archive, function(fitnesses) domhv(fitnesses))

# Get generation-wise mean fitness values
mies_aggregate_single_generation(oi$archive, function(fitnesses) {
  apply(fitnesses, 2, mean)
})
}
\seealso{
Other aggregation methods: 
\code{\link{mies_aggregate_generations}()},
\code{\link{mies_get_generation_results}()}
}
\concept{aggregation methods}
