% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measure_centrality.R
\name{centrality}
\alias{centrality}
\alias{node_degree}
\alias{node_closeness}
\alias{node_betweenness}
\alias{node_eigenvector}
\alias{node_reach}
\alias{node_power}
\title{Measures of node and tie centrality}
\usage{
node_degree(
  object,
  normalized = TRUE,
  alpha = 0,
  direction = c("all", "out", "in")
)

node_closeness(object, normalized = TRUE, direction = "out", cutoff = NULL)

node_betweenness(object, normalized = TRUE, cutoff = NULL)

node_eigenvector(object, normalized = TRUE, scale = FALSE)

node_reach(object, normalized = TRUE, k = 2)

node_power(object, normalized = TRUE, scale = FALSE, exponent = 1)
}
\arguments{
\item{object}{An object of a migraph-consistent class:
\itemize{
\item matrix (adjacency or incidence) from \code{{base}} R
\item edgelist, a data frame from \code{{base}} R or tibble from \code{{tibble}}
\item igraph, from the \code{{igraph}} package
\item network, from the \code{{network}} package
\item tbl_graph, from the \code{{tidygraph}} package
}}

\item{normalized}{Logical scalar, whether the centrality scores are normalized.
Different denominators are used depending on whether the object is one-mode or two-mode,
the type of centrality, and other arguments.}

\item{alpha}{Numeric scalar, the positive tuning parameter introduced in
Opsahl et al (2010) for trading off between degree and strength centrality measures.
By default, \code{alpha = 0}, which ignores tie weights and the measure is solely based
upon degree (the number of ties).
\code{alpha = 1} ignores the number of ties and provides the sum of the tie weights
as strength centrality.
Values between 0 and 1 reflect different trade-offs in the relative contributions of
degree and strength to the final outcome, with 0.5 as the middle ground.
Values above 1 penalise for the number of ties.
Of two nodes with the same sum of tie weights, the node with fewer ties will obtain
the higher score.
This argument is ignored except in the case of a weighted network.}

\item{direction}{Character string, “out” bases the measure on outgoing ties,
“in” on incoming ties, and "all" on either/the sum of the two.
For two-mode networks, "all" uses as numerator the sum of differences
between the maximum centrality score for the mode
against all other centrality scores in the network,
whereas "in" uses as numerator the sum of differences
between the maximum centrality score for the mode
against only the centrality scores of the other nodes in that mode.}

\item{cutoff}{Maximum path length to use during calculations.}

\item{scale}{Logical scalar, whether to rescale the vector so the maximum score is 1.}

\item{k}{Integer of steps out to calculate reach}

\item{exponent}{Decay rate for the Bonacich power centrality score.}
}
\value{
A single centralization score if the object was one-mode,
and two centralization scores if the object was two-mode.

Depending on how and what kind of an object is passed to the function,
the function will return a \code{tidygraph} object where the nodes have been updated

A numeric vector giving the betweenness centrality measure of each node.

A numeric vector giving the eigenvector centrality measure of each node.

A numeric vector giving each node's power centrality measure.
}
\description{
These functions calculate common centrality measures for one- and two-mode networks.
All measures attempt to use as much information as they are offered,
including whether the networks are directed, weighted, or multimodal.
If this would produce unintended results,
first transform the salient properties using e.g. \code{\link[=to_undirected]{to_undirected()}} functions.
All centrality and centralization measures return normalized measures by default,
including for two-mode networks.
}
\section{Functions}{
\itemize{
\item \code{node_degree()}: Calculates the degree centrality of nodes in an unweighted network,
or weighted degree/strength of nodes in a weighted network.

\item \code{node_closeness()}: Calculate the closeness centrality of nodes in a network

\item \code{node_betweenness()}: Calculate the betweenness centralities of nodes in a network

\item \code{node_eigenvector()}: Calculate the eigenvector centrality of nodes in a network

\item \code{node_reach()}: Calculate nodes' reach centrality
or how many nodes they can reach within \emph{k} steps

\item \code{node_power()}: Calculate the power centrality of nodes in a network

}}
\examples{
node_degree(mpn_elite_mex)
node_degree(ison_southern_women)
node_closeness(mpn_elite_mex)
node_closeness(ison_southern_women)
node_betweenness(mpn_elite_mex)
node_betweenness(ison_southern_women)
node_eigenvector(mpn_elite_mex)
node_eigenvector(ison_southern_women)
node_reach(ison_adolescents)
node_power(ison_southern_women, exponent = 0.5)
}
\references{
Faust, Katherine. 1997.
"Centrality in affiliation networks."
\emph{Social Networks} 19(2): 157-191.
\doi{10.1016/S0378-8733(96)00300-0}.

Borgatti, Stephen P., and Martin G. Everett. 1997.
"Network analysis of 2-mode data."
\emph{Social Networks} 19(3): 243-270.
\doi{10.1016/S0378-8733(96)00301-2}.

Borgatti, Stephen P., and Daniel S. Halgin. 2011.
"Analyzing affiliation networks."
In \emph{The SAGE Handbook of Social Network Analysis},
edited by John Scott and Peter J. Carrington, 417–33.
London, UK: Sage.
\doi{10.4135/9781446294413.n28}.

Opsahl, Tore, Filip Agneessens, and John Skvoretz. 2010.
"Node centrality in weighted networks: Generalizing degree and shortest paths."
\emph{Social Networks} 32, 245-251.
\doi{10.1016/j.socnet.2010.03.006}

Bonacich, Phillip. 1991.
“Simultaneous Group and Individual Centralities.”
\emph{Social Networks} 13(2):155–68.
\doi{10.1016/0378-8733(91)90018-O}.

Bonacich, Phillip. 1987.
“Power and Centrality: A Family of Measures.”
\emph{The American Journal of Sociology}
92(5): 1170–82.
\doi{10.1086/228631}.
}
\seealso{
\code{\link[=to_undirected]{to_undirected()}} for removing edge directions
and \code{\link[=to_unweighted]{to_unweighted()}} for removing weights from a graph.

Other measures: 
\code{\link{centralisation}},
\code{\link{closure}},
\code{\link{cohesion}()},
\code{\link{diversity}},
\code{\link{features}},
\code{\link{holes}},
\code{\link{tie_centrality}}
}
\concept{measures}
