% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/selection.R
\name{selection}
\alias{selection}
\title{Full Bayesian Models to handle missingness in Economic Evaluations (Selection Models)}
\usage{
selection(data, model.eff, model.cost, model.me = me ~ 1, model.mc = mc
  ~ 1, dist_e, dist_c, type, prob = c(0.05, 0.95), n.chains = 2,
  n.iter = 20000, n.burnin = floor(n.iter/2), inits = NULL,
  n.thin = 1, save_model = FALSE, prior = "default", ...)
}
\arguments{
\item{data}{A data frame in which to find the variables supplied in \code{model.eff}, \code{model.cost} (model formulas for effects and costs) 
and \code{model.me}, \code{model.mc} (model formulas for the missing effect and cost models). Among these,
effectiveness, cost and treatment indicator (only two arms) variables must always be provided and named 'e', 'c' and 't', respectively.}

\item{model.eff}{A formula expression in conventional \code{R} linear modelling syntax. The response must be a health economic
effectiveness outcome ('e') whose name must correspond to that used in \code{data}. Any covariates in the model must be provided on the right-hand side of the formula.
If there are no covariates, \code{1} should be specified on the right hand side of the formula.
By default, covariates are placed on the "location" parameter of the distribution through a linear model.}

\item{model.cost}{A formula expression in conventional \code{R} linear modelling syntax. The response must be a health economic
cost outcome ('c') whose name must correspond to that used in \code{data}. 
and any covariates are given on the right-hand side. Any covariates in the model must be provided on the right-hand side of the formula.
If there are no covariates, \code{1} should be specified on the right hand side of the formula.
By default, covariates are placed on the "location" parameter of the distribution through a linear model. A joint bivariate distribution for effects and costs can be specified by
including 'e' on the right-hand side of the formula for the costs model.}

\item{model.me}{A formula expression in conventional \code{R} linear modelling syntax. The response must be indicated with the 
term 'me'(missing effects) and any covariates must be provided on the right-hand side of the formula. 
If there are no covariates, \code{1} should be specified on the right hand side of the formula. By default, covariates are placed on the "probability" parameter for the missing effects through a logistic-linear model.}

\item{model.mc}{A formula expression in conventional \code{R} linear modelling syntax. The response must be indicated with the 
term 'mc'(missing costs) and any covariates must be provided on the right-hand side of the formula. 
If there are no covariates, \code{1} should be specified on the right hand side of the formula. By default, covariates are placed on the "probability" parameter for the missing costs through a logistic-linear model.}

\item{dist_e}{Distribution assumed for the effects. Current available choices are: Normal ('norm') or Beta ('beta').}

\item{dist_c}{Distribution assumed for the costs. Current available choices are: Normal ('norm'), Gamma ('gamma') or LogNormal ('lnorm').}

\item{type}{Type of missingness mechanism assumed. Choices are Missing At Random (MAR) and Missing Not At Random (MNAR).}

\item{prob}{A numeric vector of probabilities within the range (0,1), representing the upper and lower
CI sample quantiles to be calculated and returned for the imputed values.}

\item{n.chains}{Number of chains.}

\item{n.iter}{Number of iterations.}

\item{n.burnin}{Number of warmup iterations.}

\item{inits}{A list with elements equal to the number of chains selected; each element of the list is itself a list of starting values for the
\code{JAGS} model, or a function creating (possibly random) initial values. If \code{inits} is \code{NULL}, \code{JAGS}
will generate initial values for all the model parameters.}

\item{n.thin}{Thinning interval.}

\item{save_model}{Logical. If \code{save_model} is \code{TRUE} a \code{txt} file containing the model code is printed
in the current working directory.}

\item{prior}{A list containing the hyperprior values provided by the user. Each element of this list must be a vector of length two
containing the user-provided hyperprior values and must be named with the name of the corresponding parameter. For example, the hyperprior
values for the standard deviation effect parameters can be provided using the list \code{prior = list('sigma.prior.e' = c(0, 100))}.
For more information about how to provide prior hypervalues for different types of parameters and models see details. 
If \code{prior} is set to 'default', the default values will be used.}

\item{...}{Additional arguments that can be provided by the user. Examples are \code{center = TRUE} to center all the covariates in the model 
or the additional arguments that can be provided to the function \code{\link[BCEA]{bcea}} to summarise the health economic evaluation results.}
}
\value{
An object of the class 'missingHE' containing the following elements
\describe{
  \item{data_set}{A list containing the original data set provided in \code{data} (see Arguments), the number of observed and missing individuals 
  , the total number of individuals by treatment arm and the indicator vectors for the missing values}
  \item{model_output}{A list containing the output of a \code{JAGS} model generated from the functions \code{\link[R2jags]{jags}}, and 
  the posterior samples for the main parameters of the model and the imputed values}
  \item{cea}{A list containing the output of the economic evaluation performed using the function \code{\link[BCEA]{bcea}}}
  \item{type}{A character variable that indicate which type of missingness mechanism has been used to run the model, 
  either \code{MAR} or \code{MNAR} (see details)}
}
}
\description{
Full Bayesian cost-effectiveness models to handle missing data in the outcomes under different missing data 
mechanism assumptions, using alternative parametric distributions for the effect and cost variables and 
using a selection model approach to identify the model. The analysis is performed using the \code{BUGS} language, 
which is implemented in the software \code{JAGS} using the function \code{\link[R2jags]{jags}} The output is stored in an object of class 'missingHE'.
}
\details{
Depending on the distributions specified for the outcome variables in the arguments \code{dist_e} and
\code{dist_c} and the type of missingness mechanism specified in the argument \code{type}, different selection models
are built and run in the background by the function \code{selection}. These models consist in logistic regressions that are used to estimate
the probability of missingness in one or both the outcomes. 
A simple example can be used to show how selection models are specified. 
Consider a data set comprising a response variable \eqn{y} and a set of centered covariate \eqn{X_j}. For each subject in the trial \eqn{i = 1, ..., n}
we define an indicator variable \eqn{m_i} taking value \code{1} if the \eqn{i}-th individual is associated with a missing value and \code{0} otherwise.
This is modelled as:
\deqn{m_i ~ Bernoulli(\pi_i)}
\deqn{logit(\pi_i) = \gamma_0 + \sum\gamma_j X_j + \delta(y)}
where
\itemize{
\item \eqn{\pi_i} is the individual probability of a missing value in \eqn{y}
\item \eqn{\gamma_0} represents the marginal probability of a missing value in \eqn{y} on the logit scale.
\item \eqn{\gamma_j} represents the impact on the probability of a missing value in \eqn{y} of the centered covariates \eqn{X_j}.
\item \eqn{\delta} represents the impact on the probability of a missing value in \eqn{y} of the missing value itself.
}
When \eqn{\delta = 0} the model assumes a 'MAR' mechanism, while when \eqn{\delta != 0} the mechanism is 'MNAR'. For the parameters indexing the missingness model, 
the default prior distributions assumed are the following:
\itemize{
\item \eqn{\gamma_0 ~ Logisitc(0, 1)}
\item \eqn{\gamma_j ~ Normal(0, 0.01)}
\item \eqn{\delta ~ Normal(0, 1)}
}
When user-defined hyperprior values are supplied via the argument \code{prior} in the function \code{selection}, the elements of this list (see Arguments)
must be vectors of length two containing the user-provided hyperprior values and must take specific names according to the parameters they are associated with. 
Specifically, the names for the parameters indexing the model which are accepted by \strong{missingHE} are the following:
\itemize{
\item location parameters \eqn{\alpha_0} and \eqn{\beta_0}: "mean.prior.e"(effects) and/or "mean.prior.c"(costs)
\item auxiliary parameters \eqn{\sigma}: "sigma.prior.e"(effects) and/or "sigma.prior.c"(costs)
\item covariate parameters \eqn{\alpha_j} and \eqn{\beta_j}: "alpha.prior"(effects) and/or "beta.prior"(costs)
\item marginal probability of missing values \eqn{\gamma_0}: "p.prior.e"(effects) and/or "p.prior.c"(costs)
\item covariate parameters in the missingness model \eqn{\gamma_j} (if covariate data provided): "gamma.prior.e"(effects) and/or "gamma.prior.c"(costs)
\item mnar parameter \eqn{\delta}: "delta.prior.e"(effects) and/or "delta.prior.c"(costs)
} 
For simplicity, here we have assumed that the set of covariates \eqn{X_j} used in the models for the effects/costs and in the 
model of the missing effect/cost values is the same. However, it is possible to specify different sets of covariates for each model
using the arguments in the function \code{selection} (see Arguments).
}
\examples{
# Quck example to run using subset of MenSS dataset
MenSS.subset <- MenSS[1:80, ]

# Run the model using the selection function assuming a SCAR mechanism
# Use only 100 iterations to run a quick check
model.selection <- selection(data = MenSS.subset, model.eff = e ~ 1,model.cost = c ~ 1,
   model.me = me ~ 1, model.mc = mc ~ 1, dist_e = "norm", dist_c = "norm",
   type = "MAR", n.chains = 2, n.iter = 100)

# Print the results of the JAGS model
print(model.selection)
#

# Use dic information criterion to assess model fit
pic.dic <- pic(model.selection, criterion = "dic", module = "total")
pic.dic
#

\dontshow{
# Use waic information criterion to assess model fit
pic.waic <- pic(model.selection, criterion = "waic", module = "total")
pic.waic
}

# Assess model convergence using graphical tools
# Produce histograms of the posterior samples for the mean effects
diag.hist <- diagnostic(model.selection, type = "histogram", param = "mu.e")
#

# Compare observed effect data with imputations from the model
# using plots (posteiror means and credible intervals)
p1 <- plot(model.selection, class = "scatter", outcome = "effects")
#

# Summarise the CEA information from the model
summary(model.selection)

\donttest{
# Further examples which take longer to run
model.selection <- selection(data = MenSS, model.eff = e ~ u.0,model.cost = c ~ e,
   model.se = me ~ u.0, model.mc = mc ~ 1, dist_e = "norm", dist_c = "norm",
   type = "MAR", n.chains = 2, n.iter = 1000)
#
# Print results for all imputed values
print(model.selection, value.mis = TRUE)

# Use looic to assess model fit
pic.looic<-pic(model.selection, criterion = "looic", module = "total")
pic.looic

# Show density plots for all parameters
diag.hist <- diagnostic(model.selection, type = "denplot", param = "all")

# Plots of imputations for all data
p1 <- plot(model.selection, class = "scatter", outcome = "all")

# Summarise the CEA results
summary(model.selection)

}
#
#
}
\references{
Daniels, MJ. Hogan, JW. \emph{Missing Data in Longitudinal Studies: strategies for Bayesian modelling and sensitivity analysis}, CRC/Chapman Hall.

Baio, G.(2012). \emph{Bayesian Methods in Health Economics}. CRC/Chapman Hall, London.

Gelman, A. Carlin, JB., Stern, HS. Rubin, DB.(2003). \emph{Bayesian Data Analysis, 2nd edition}, CRC Press.

Plummer, M. \emph{JAGS: A program for analysis of Bayesian graphical models using Gibbs sampling.} (2003).
}
\seealso{
\code{\link[R2jags]{jags}}, \code{\link[BCEA]{bcea}}
}
\author{
Andrea Gabrio
}
\keyword{CEA}
\keyword{JAGS}
\keyword{Models}
\keyword{Selection}
\keyword{data}
\keyword{missing}
