% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cohens.d.R
\name{cohens.d}
\alias{cohens.d}
\alias{cohens.d.default}
\alias{cohens.d.formula}
\title{Cohen's d}
\usage{
cohens.d(x, ...)

\method{cohens.d}{default}(x, y = NULL, mu = 0, paired = FALSE, weighted = TRUE, cor = TRUE,
         ref = NULL, correct = FALSE, alternative = c("two.sided", "less", "greater"),
         conf.level = 0.95, group = NULL, split = NULL, sort.var = FALSE,
         digits = 2, as.na = NULL, write = NULL, append = TRUE,
         check = TRUE, output = TRUE, ...)

\method{cohens.d}{formula}(formula, data, weighted = TRUE, cor = TRUE, ref = NULL,
         correct = FALSE, alternative = c("two.sided", "less", "greater"),
         conf.level = 0.95, group = NULL, split = NULL, sort.var = FALSE,
         na.omit = FALSE, digits = 2, as.na = NULL, write = NULL, append = TRUE,
         check = TRUE, output = TRUE, ...)
}
\arguments{
\item{x}{a numeric vector or data frame.}

\item{...}{further arguments to be passed to or from methods.}

\item{y}{a numeric vector.}

\item{mu}{a numeric value indicating the reference mean.}

\item{paired}{logical: if \code{TRUE}, Cohen's d for a paired-sample design is computed.}

\item{weighted}{logical: if \code{TRUE} (default), the weighted pooled standard deviation is used
to compute the standardized mean difference between two groups of a two-sample
design (i.e., \code{paired = FALSE}), while standard deviation of the difference
scores is used to compute the standardized mean difference in a paired-sample
design (i.e., \code{paired = TRUE}).}

\item{cor}{logical: if \code{TRUE} (default), \code{paired = TRUE}, and \code{weighted = FALSE},
Cohen's d for a paired-sample design while controlling for the correlation between
the two sets of measurement is computed. Note that this argument is only used in
a paired-sample design (i.e., \code{paired = TRUE}) when specifying \code{weighted = FALSE}.}

\item{ref}{character string \code{"x"} or \code{"y"} for specifying the reference reference
group when using the default \code{cohens.d()} function or a numeric value or
character string indicating the reference group in a two-sample design when using
the formula \code{cohens.d()} function. The standard deviation of the reference variable
or reference group is used to standardized the mean difference.
Note that this argument is only used in a two-sample design (i.e., \code{paired = FALSE}).}

\item{correct}{logical: if \code{TRUE}, correction factor to remove positive bias in small samples is
used.}

\item{alternative}{a character string specifying the alternative hypothesis, must be one of
\code{"two.sided"} (default), \code{"greater"} or \code{"less"}.}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence level of the interval.}

\item{group}{a numeric vector, character vector or factor as grouping variable.}

\item{split}{a numeric vector, character vector or factor as split variable.}

\item{sort.var}{logical: if \code{TRUE}, output table is sorted by variables when specifying \code{group}.}

\item{digits}{an integer value indicating the number of decimal places to be used for
displaying results.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting the analysis.
Note that \code{as.na()} function is only applied to \code{y} but not to \code{group}
in a two-sample design, while \code{as.na()} function is applied to \code{pre}
and \code{post} in a paired-sample design.}

\item{write}{a character string naming a text file with file extension
\code{".txt"} (e.g., \code{"Output.txt"}) for writing the
output into a text file.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown on the console.}

\item{formula}{a formula of the form \code{y ~ group} for one outcome variable or
\code{cbind(y1, y2, y3) ~ group} for more than one outcome variable where \code{y}
is a numeric variable giving the data values and \code{group} a numeric variable,
character variable or factor with two values or factor levels giving the
corresponding groups.}

\item{data}{a matrix or data frame containing the variables in the formula \code{formula}.}

\item{na.omit}{logical: if \code{TRUE}, incomplete cases are removed before conducting the analysis
(i.e., listwise deletion) when specifying more than one outcome variable.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:

\item{\code{call}}{function call}
\item{\code{type}}{type of analysis}
\item{\code{sample}}{type of sample, i.e., one-, two-, or, paired-sample}
\item{\code{data}}{matrix or data frame specified in \code{x}}
\item{\code{args}}{specification of function arguments}
\item{\code{result}}{result table}
}
\description{
This function computes Cohen's d for one-sample, two-sample (i.e., between-subject design),
and paired-sample designs (i.e., within-subject design) for one or more variables, optionally
by a grouping and/or split variable. In a two-sample design, the function computes the
standardized mean difference by dividing the difference between  means of the two groups
of observations by the weighted pooled standard deviation (i.e., Cohen's \eqn{d_s}
according to Lakens, 2013) by default. In a paired-sample design, the function computes the
standardized mean difference by dividing the mean of the difference scores by the standard
deviation of the difference scores (i.e., Cohen's \eqn{d_z} according to Lakens, 2013) by
default. Note that by default Cohen's d is computed without applying the correction factor
for removing the small sample bias (i.e., Hedges' g).
}
\details{
Cohen (1988, p.67) proposed to compute the standardized mean difference in a two-sample design
by dividing the mean difference by the unweighted pooled standard deviation (i.e.,
\code{weighted = FALSE}).

Glass et al. (1981, p. 29) suggested to use the standard deviation of the control group
(e.g., \code{ref = 0} if the control group is coded with 0) to compute the standardized
mean difference in a two-sample design (i.e., Glass's \eqn{\Delta}) since the standard deviation of the control group
is unaffected by the treatment and will therefore more closely reflect the population
standard deviation.

Hedges (1981, p. 110) recommended to weight each group's standard deviation by its sample
size resulting in a weighted and pooled standard deviation (i.e., \code{weighted = TRUE},
default). According to Hedges and Olkin (1985, p. 81), the standardized mean difference
based on the weighted and pooled standard deviation has a positive small sample bias,
i.e., standardized mean difference is overestimated in small samples (i.e., sample size
less than 20 or less than 10 in each group). However, a correction factor can be applied
to remove the small sample bias (i.e., \code{correct = TRUE}). Note that the function uses
a gamma function for computing the correction factor, while a approximation method is
used if computation based on the gamma function fails.

Note that the terminology is inconsistent because the standardized mean difference based
on the weighted and pooled standard deviation is usually called Cohen's d, but sometimes
called Hedges' g. Oftentimes, Cohen's d is called Hedges' d as soon as the small sample
correction factor is applied. Cumming and Calin-Jageman (2017, p.171) recommended to avoid
the term Hedges' g , but to report which standard deviation was used to standardized the
mean difference (e.g., unweighted/weighted pooled standard deviation, or the standard
deviation of the control group) and whether a small sample correction factor was applied.

As for the terminology according to Lakens (2013), in a two-sample design (i.e.,
\code{paired = FALSE}) Cohen's \eqn{d_s} is computed when using \code{weighted = TRUE} (default)
and Hedges's \eqn{g_s} is computed when using \code{correct = TRUE} in addition. In a
paired-sample design (i.e., \code{paired = TRUE}), Cohen's \eqn{d_z} is computed when using
\code{weighted = TRUE, default}, while Cohen's \eqn{d_{rm}} is computed when using
\code{weighted = FALSE} and \code{cor = TRUE, default} and Cohen's \eqn{d_{av}} is computed when
using \code{weighted = FALSE} and \code{cor = FALSE}. Corresponding Hedges' \eqn{g_z}, \eqn{g_{rm}},
and \eqn{g_{av}} are computed when using \code{correct = TRUE} in addition.
}
\examples{
#----------------------------------------------------------------------------
# One-sample design

# Example 1a: Cohen's d.z with two-sided 95% CI
# population mean = 3
cohens.d(mtcars$mpg, mu = 20)

# Example 1b: Cohen's d.z (aka Hedges' g.z) with two-sided 95% CI
# population mean = 3, with small sample correction factor
cohens.d(mtcars$mpg, mu = 20, correct = TRUE)

# Example 1c: Cohen's d.z with two-sided 95% CI
# population mean = 3, by 'vs' separately
cohens.d(mtcars$mpg, mu = 20, group = mtcars$vs)

# Example 1d: Cohen's d.z with two-sided 95% CI
# population mean = 20, split analysis by 'vs'
cohens.d(mtcars$mpg, mu = 20, split = mtcars$vs)

# Example 1e: Cohen's d.z with two-sided 95% CI
# population mean = 3, by 'vs' separately, split by 'am'
cohens.d(mtcars$mpg, mu = 20, group = mtcars$vs, split = mtcars$am)

#----------------------------------------------------------------------------
# Two-sample design

# Example 2a: Cohen's d.s with two-sided 95% CI
# weighted pooled SD
cohens.d(mpg ~ vs, data = mtcars)

# Example 2b: Cohen's d.s with two-sided 99% CI
# weighted pooled SD
cohens.d(mpg ~ vs, data = mtcars, conf.level = 0.99)

# Example 2c: Cohen's d.s with one-sided 99% CI
# weighted pooled SD
cohens.d(mpg ~ vs, data = mtcars, alternative = "greater", conf.level = 0.99)

# Example 2d: Cohen's d.s for more than one variable with two-sided 95% CI
# weighted pooled SD
cohens.d(cbind(mpg, disp, hp) ~ vs, data = mtcars)

# Example 2e: Cohen's d with two-sided 95% CI
# unweighted SD
cohens.d(mpg ~ vs, data = mtcars, weighted = FALSE)

# Example 2f: Cohen's d.s (aka Hedges' g.s) with two-sided 95% CI
# weighted pooled SD, with small sample correction factor
cohens.d(mpg ~ vs, data = mtcars, correct = TRUE)

# Example 2g: Cohen's d (aka Hedges' g) with two-sided 95% CI
# Unweighted SD, with small sample correction factor
cohens.d(mpg ~ vs, data = mtcars, weighted = FALSE, correct = TRUE)

# Example 2h: Cohen's d (aka Glass's delta) with two-sided 95% CI
# SD of reference group 1
cohens.d(mpg ~ vs, data = mtcars, ref = 0)

# Example 2i: Cohen's d.s with two-sided 95% CI
# weighted pooled SD, by 'am' separately
cohens.d(mpg ~ vs, data = mtcars, group = mtcars$am)

# Example 2j: Cohen's d.s with two-sided 95% CI
# weighted pooled SD, split analysis by 'am'
cohens.d(mpg ~ vs, data = mtcars, split = mtcars$am)

#----------------------------------------------------------------------------
# Paired-sample design

# Example 3a: Cohen's d.z with two-sided 95% CI
# SD of the difference scores
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE)

# Example 3b: Cohen's d.z with one-sided 99% CI
# SD of the difference scores
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE, alternative = "greater",
         conf.level = 0.99)

# Example 3c: Cohen's d.rm with two-sided 95% CI
# controlling for the correlation between measures
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE, weighted = FALSE)

# Example 3d: Cohen's d.av with two-sided 95% CI
# without controlling for the correlation between measures
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE, weighted = FALSE, cor = FALSE)

# Example 3e: Cohen's d.z (aka Hedges' g.z) with two-sided 95% CI
# SD of the differnece scores
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE, correct = TRUE)

# Example 3f: Cohen's d.rm (aka Hedges' g.rm) with two-sided 95% CI
# controlling for the correlation between measures
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE, weighted = FALSE, correct = TRUE)

# Example 3g: Cohen's d.av (aka Hedges' g.av) with two-sided 95% CI
# without controlling for the correlation between measures
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE, weighted = FALSE, cor = FALSE,
         correct = TRUE)

# Example 3h: Cohen's d.z with two-sided 95% CI
# SD of the difference scores, by 'vs' separately
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE, group = mtcars$vs)

# Example 3i:  Cohen's d.z with two-sided 95% CI
# SD of the difference scores, split analysis by 'vs'
cohens.d(mtcars$drat, mtcars$wt, paired = TRUE, split = mtcars$vs)
}
\references{
Cohen, J. (1988). \emph{Statistical power analysis for the behavioral sciences} (2nd ed.).
Academic Press.

Cumming, G., & Calin-Jageman, R. (2017). \emph{Introduction to the new statistics: Estimation, open science,
& beyond}. Routledge.

Glass. G. V., McGaw, B., & Smith, M. L. (1981). \emph{Meta-analysis in social research}. Sage Publication.

Goulet-Pelletier, J.-C., & Cousineau, D. (2018) A review of effect sizes and their confidence intervals,
Part I: The Cohen's d family. \emph{The Quantitative Methods for Psychology, 14}, 242-265.
https://doi.org/10.20982/tqmp.14.4.p242

Hedges, L. V. (1981). Distribution theory for Glass's estimator of effect size and related estimators.
\emph{Journal of Educational Statistics, 6}(3), 106-128.

Hedges, L. V. & Olkin, I. (1985). \emph{Statistical methods for meta-analysis}. Academic Press.

Lakens, D. (2013). Calculating and reporting effect sizes to facilitate cumulative science:
A practical primer for t-tests and ANOVAs. \emph{Frontiers in Psychology, 4}, 1-12.
https://doi.org/10.3389/fpsyg.2013.00863
}
\seealso{
\code{\link{test.t}}, \code{\link{test.z}}, \code{\link{effsize}}, \code{\link{cor.matrix}},
\code{\link{na.auxiliary}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
