% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blimp.plot.R
\name{blimp.plot}
\alias{blimp.plot}
\title{Blimp Trace Plots and Posterior Distribution Plots}
\usage{
blimp.plot(x, plot = c("none", "trace", "post"), param = NULL, labels = TRUE,
           burnin = TRUE, point = c("all", "none", "m", "med", "map"),
           ci = c("none", "eti", "hdi"), conf.level = 0.95, hist = TRUE,
           density = TRUE, area = TRUE, alpha = 0.4, fill = "gray85",
           facet.nrow = NULL, facet.ncol = NULL,
           facet.scales = c("fixed", "free", "free_x", "free_y"),
           xlab = NULL, ylab = NULL, xlim = NULL, ylim = NULL,
           xbreaks = ggplot2::waiver(), ybreaks = ggplot2::waiver(),
           xexpand = ggplot2::waiver(), yexpand = ggplot2::waiver(),
           palette = "Set 2", binwidth = NULL, bins = NULL,
           density.col = "#0072B2", shape = 21,
           point.col = c("#CC79A7", "#D55E00", "#009E73"),
           linewidth = 0.6, linetype = "dashed", line.col = "black",
           plot.margin = NULL, legend.title.size = 10, legend.text.size = 10,
           legend.box.margin = NULL, saveplot = c("all", "none", "trace", "post"),
           filename = "Blimp_Plot.pdf", file.plot = c("_TRACE", "_POST"),
           width = NA, height = NA, units = c("in", "cm", "mm", "px"), dpi = 600,
           check = TRUE)
}
\arguments{
\item{x}{a character string indicating the name of folder
containing the \code{posterior.*} file, e.g.,
\code{"Posterior_Ex4.3"} or the name of the
\code{posterior.*} file with or without any file
extension, e.g., \code{"Posterior_ExEx4.3/posterior.csv"}
or \code{"Posterior_ExEx4.3/posterior"}. Alternatively,
a \code{misty.object} of type \code{blimp} can be
specified, i.e., result object of the \code{blimp.plot()}
function. Note that if the \code{posterior} file is
specified without file extension while multiple
\code{posterior.*} files in different file formats
are available, then the file is read in following
order: \code{csv},\code{RData}, \code{rds},
and \code{xlsx}.}

\item{plot}{a character string indicating the type of plot to
display, i.e., \code{"none"} for not displaying any
plot, \code{"trace"} (default) for displaying trace
plots, and \code{post} for displaying posterior
distribution plots.}

\item{param}{a numeric vector indicating which parameters to print
for the trace plots or posterior distribution plots.
Note that the number of the parameter (\code{Param})
and the parameter specification (\code{L1}, \code{L2},
and \code{L3}) are provided in the text file
\code{"partable.txt"}. Note that parameters with zero
variance are excluded by default.}

\item{labels}{logical: if \code{TRUE} (default), parameter labels
(e.g., \code{y Beta x} for the slope of the regression
y on x) are shown in the facet labels. If \code{FALSE},
the numbers of the parameter (e.g., \code{Parameter 1}
are shown in the the facet labels.}

\item{burnin}{logical: if \code{FALSE}, the burn-in iterations are
discarded when displaying trace plots. The default
setting for \code{plot = "trace"} is \code{TRUE}.
Note that the burn-in iterations are always discarded
when displaying posterior distribution plots
(\code{plot = "post"}) regardless of the setting of
the argument \code{burnin}.}

\item{point}{a character vector indicating the point estimate(s)
to be displayed in the posterior distribution plots,
i.e., \code{"all"} for all point estimates, \code{"none"}
for not displaying any point estimates, \code{"m"}
for the posterior mean estimate, \code{"med"} (default)
for the posterior median estimate, and \code{"map"}
for the maximum a posterior estimate.}

\item{ci}{a character string indicating the type of credible
interval to be displayed in the posterior distribution
plots, i.e., \code{"none"} for not displaying any
credible intervals, \code{"eti"} (default) for displaying
the equal-tailed intervals and \code{"hdi"} for displaying
the highest density interval.}

\item{conf.level}{a numeric value between 0 and 1 indicating the
confidence level of the credible interval (default is
\code{0.95}).}

\item{hist}{logical: if \code{TRUE} (default), histograms are
drawn in the posterior probability plots.}

\item{density}{logical: if \code{TRUE} (default), density curves are
drawn in the posterior probability plots.}

\item{area}{logical: if \code{TRUE} (default), statistical not
significant and statistical significant area is
filled with a different color and vertical lines are
drawn.}

\item{alpha}{a numeric value between 0 and 1 for the \code{alpha}
argument (default is \code{0.4}) for the \code{annotate},
and \code{geom_histogram} function.}

\item{fill}{a character string indicating the color for the
\code{"fill"} argument (default is \code{"gray85"})
for the \code{annotate} and \code{geom_histogram}
functions.}

\item{facet.nrow}{a numeric value indicating the \code{nrow} argument
(default is \code{NULL}) for the \code{facet_wrap}
function.}

\item{facet.ncol}{a numeric value indicating the \code{ncol} argument
(default is \code{2}) for the \code{facet_wrap} function.}

\item{facet.scales}{a character string indicating the \code{scales} argument
(default is \code{"free"}) for the \code{facet_wrap}
function.}

\item{xlab}{a character string indicating the \code{name} argument
for the \code{scale_x_continuous} function.}

\item{ylab}{a character string indicating the \code{name} argument
for the \code{scale_y_continuous} function.}

\item{xlim}{a numeric vector with two elements indicating the
\code{limits} argument (default it \code{NULL}) for
the \code{scale_x_continuous} function.}

\item{ylim}{a numeric vector with two elements indicating the
\code{limits} argument (default it \code{NULL}) for
the \code{scale_y_continuous} function.}

\item{xbreaks}{a numeric vector indicating the \code{breaks} argument
(default is \code{ggplot2::waiver()}) for the
\code{scale_x_continuous} function.}

\item{ybreaks}{a numeric vector indicating the \code{breaks} argument
(default is \code{ggplot2::waiver()}) for the
\code{scale_y_continuous} function.}

\item{xexpand}{a numeric vector with two elements indicating the
\code{expand} argument (default is \code{(0.02, 0)})
for the \code{scale_x_continuous} function.}

\item{yexpand}{a numeric vector with two elements indicating the
\code{expand} argument for the \code{scale_y_continuous}
function. Note that the default setting depends
on the type of plot, e.g., \code{(0.02, 0)} for the
trace plots and \code{expansion(mult = c(0, 0.05))}
for the posterior distribution plots.}

\item{palette}{a character string indicating the palette name (default
is \code{"Set 2"}) for the \code{hcl.colors} function.
Note that the character string must be one of
\code{hcl.pals()}.}

\item{binwidth}{a numeric value indicating the \code{binwidth} argument
(default is to use the number of bins in \code{bins}
argument) for the \code{geom_histogram} function.}

\item{bins}{a numeric value indicating the \code{bins} argument
(default is \code{30}) for the \code{geom_histogram}
function.}

\item{density.col}{a character string indicating the \code{color} argument
(default is \code{"#0072B2"}) for the \code{geom_density}
function.}

\item{shape}{a numeric value indicating the \code{shape} argument
(default is \code{21}) for the \code{geom_point}
function.}

\item{point.col}{a character vector with three elements indicating the
\code{values} argument (default is
\code{c("#CC79A7", "#D55E00", "#009E73")}) for the
\code{scale_color_manual} function.}

\item{linewidth}{a numeric value indicating the \code{linewidth} argument
(default is \code{0.6}) for the \code{geom_vline} function.}

\item{linetype}{a numeric value indicating the \code{linetype} argument
(default is \code{"dashed"}) for the \code{geom_vline}
function.}

\item{line.col}{a character string indicating the \code{color} argument
(default is \code{"black"}) for the \code{geom_vline}
function.}

\item{plot.margin}{a numeric vector indicating the \code{plot.margin}
argument for the \code{theme} function. Note that the
default setting depends on the type of the plot, e.g.,
\code{c(4, 15, -10, 0)} for the trace plots, and
\code{c(4, 15, 4, 4)} for the autocorrelation plots.}

\item{legend.title.size}{a numeric value indicating the \code{legend.title}
argument (default is \code{element_text(size = 10)})
for the \code{theme} function.}

\item{legend.text.size}{a numeric value indicating the \code{legend.text}
argument (default is \code{element_text(size = 10)})
for the \code{theme} function.}

\item{legend.box.margin}{a numeric vector indicating the \code{legend.box.margin}
argument for the \code{theme} function. Note that the
default setting depends on the type of plot, e.g.,
\code{c(-16, 6, 6, 6)} for the trace plots, and
\code{c(-25, 6, 6, 6)} for the posterior distribution
plots with displaying point estimates.}

\item{saveplot}{a character vector indicating the plot to be saved,
i.e., \code{"all"} for saving all plots, \code{"none"}
(default) for not saving any plots, \code{"trace"}
for saving the trace plots and \code{post} for the saving
the posterior distribution plots.}

\item{filename}{a character string indicating the \code{filename}
argument (default is \code{"Blimp_Plot.pdf"}) including
the file extension for the \code{ggsave} function.
Note that one of \code{".eps"}, \code{".ps"},
\code{".tex"}, \code{".pdf"} (default), \code{".jpeg"},
\code{".tiff"}, \code{".png"}, \code{".bmp"},
\code{".svg"} or \code{".wmf"} needs to be specified
as file extension in the \code{filename} argument.}

\item{file.plot}{a character vector with two elements for distinguishing
different types of plots. By default, the character
string specified in the argument \code{"filename"}
(\code{"Blimp_Plot"}) is concatenated with \code{"_TRACE"}
(\code{"Blimp_Plot_TRACE"}) for the trace plots,
and \code{"_POST"} (\code{"Blimp_Plot_POST"}) for
the posterior distribution plots.}

\item{width}{a numeric value indicating the \code{width} argument
(default is the size of the current graphics device)
for the \code{ggsave} function.}

\item{height}{a numeric value indicating the \code{height} argument
(default is the size of the current graphics device)
for the \code{ggsave} function.}

\item{units}{a character string indicating the \code{units} argument
(default is \code{in}) for the \code{ggsave} function.}

\item{dpi}{a numeric value indicating the \code{dpi} argument
(default is \code{600}) for the \code{ggsave} function.}

\item{check}{logical: if \code{TRUE} (default), argument
specification is checked.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:

\item{\code{call}}{function call}
\item{\code{type}}{type of analysis}
\item{\code{x}}{a character string indicating the name of the \code{posterior.*} file}
\item{\code{args}}{specification of function arguments}
\item{\code{data}}{list with posterior distribution of each parameter estimate
                   in long format (\code{plotdat}), plot data for the trace
                   plots (\code{trace}), and plot data for the posterior
                   distribution plots (\code{post}).}
\item{\code{plot}}{list with the trace plots (\code{trace} and posterior distribution
                   plots (\code{post})}
}
\description{
This function reads the posterior distribution including burn-in and
post-burn-in phase for all parameters saved in long format in a file called
\code{posterior.*} by the function \code{blimp.run} or \code{blimp} when specifying
\code{posterior = TRUE} to display trace plots and posterior distribution plots.
}
\examples{
\dontrun{

#----------------------------------------------------------------------------
# Blimp Example 4.3: Linear Regression

#..........
# Trace Plots

# Example 1a: Default setting, specifying name of the folder
blimp.plot("Posterior_Ex4.3")

# Example 1b: Default setting, specifying the posterior file
blimp.plot("Posterior_Ex4.3/posterior.csv")

# Example 1c: Print parameters 2, 3, 4, and 5
blimp.plot("Posterior_Ex4.3", param = 2:5)

# Example 1e: Arrange panels in three columns
blimp.plot("Posterior_Ex4.3", ncol = 3)

# Example 1f: Specify "Pastel 1" palette for the hcl.colors function
blimp.plot("Posterior_Ex4.3", palette = "Pastel 1")

#..........
# Posterior Distribution Plots

# Example 2a: Default setting, i.e., posterior median and equal-tailed interval
blimp.plot("Posterior_Ex4.3", plot = "post")

# Example 2b: Display posterior mean and maximum a posteriori
blimp.plot("Posterior_Ex4.3", plot = "post", point = c("m", "map"))

# Example 2c: Display maximum a posteriori and highest density interval
blimp.plot("Posterior_Ex4.3", plot = "post", point = "map", ci = "hdi")

# Example 2d: Do not display any point estimates and credible interval
blimp.plot("Posterior_Ex4.3", plot = "post", point = "none", ci = "none")

# Example 2d: Do not display histograms
blimp.plot("Posterior_Ex4.3", plot = "post", hist = FALSE)

#..........
# Save Plots

# Example 3a: Save all plots in pdf format
blimp.plot("Posterior_Ex4.3", saveplot = "all")

# Example 3b: Save all plots in png format with 300 dpi
blimp.plot("Posterior_Ex4.3", saveplot = "all", filename = "Blimp_Plot.png", dpi = 300)

# Example 3a: Save posterior distribution plot, specify width and height of the plot
blimp.plot("Posterior_Ex4.3", plot = "none", saveplot = "post",
           width = 7.5, height = 7)

#----------------------------------------------------------------------------
# Plot from misty.object

# Create misty.object
object <- blimp.plot("Posterior_Ex4.3", plot = "none")

# Trace plot
blimp.plot(object, plot = "trace")

# Posterior distribution plot
blimp.plot(object, plot = "post")

#----------------------------------------------------------------------------
# Create Plots Manually

# Load ggplot2 package
library(ggplot2)

# Create misty object
object <- blimp.plot("Posterior_Ex4.3", plot = "none")

#..........
# Example 4: Trace Plots

# Extract data
data.trace <- object$data$trace

# Plot
ggplot(data.trace, aes(x = iter, y = value, color = chain)) +
  annotate("rect", xmin = 0, xmax = 1000, ymin = -Inf, ymax = Inf,
           alpha = 0.4, fill = "gray85") +
  geom_line() +
  facet_wrap(~ param, ncol = 2, scales = "free") +
  scale_x_continuous(name = "", expand = c(0.02, 0)) +
  scale_y_continuous(name = "", expand = c(0.02, 0)) +
  scale_colour_manual(name = "Chain",
                      values = hcl.colors(n = 2, palette = "Set 2")) +
  theme_bw() +
  guides(color = guide_legend(nrow = 1, byrow = TRUE)) +
  theme(plot.margin = margin(c(4, 15, -10, 0)),
        legend.position = "bottom",
        legend.title = element_text(size = 10),
        legend.text = element_text(size = 10),
        legend.box.margin = margin(c(-16, 6, 6, 6)),
        legend.background = element_rect(fill = "transparent"))

#..........
# Example 5: Posterior Distribution Plots

# Extract data
data.post <- object$data$post

# Plot
ggplot(data.post, aes(x = value)) +
  geom_histogram(aes(y = after_stat(density)), color = "black", alpha = 0.4,
                 fill = "gray85") +
  geom_density(color = "#0072B2") +
  geom_vline(data = data.frame(param = levels(data.post$param),
                               stat = tapply(data.post$value, data.post$param, median)),
             aes(xintercept = stat, color = "Median"), linewidth = 0.6) +
  geom_vline(data = data.frame(param = levels(data.post$param),
                               low = tapply(data.post$value, data.post$param,
                                            function(y) quantile(y, probs = 0.025))),
             aes(xintercept = low), linetype = "dashed", linewidth = 0.6) +
  geom_vline(data = data.frame(param = levels(data.post$param),
                               upp = tapply(data.post$value, data.post$param,
                                            function(y) quantile(y, probs = 0.975))),
             aes(xintercept = upp), linetype = "dashed", linewidth = 0.6) +
  facet_wrap(~ param, ncol = 2, scales = "free") +
  scale_x_continuous(name = "", expand = c(0.02, 0)) +
  scale_y_continuous(name = "Probability Density, f(x)",
                     expand = expansion(mult = c(0L, 0.05))) +
  scale_color_manual(name = "Point Estimate", values = c(Median = "#D55E00")) +
  labs(caption = "95\% Equal-Tailed Interval") +
  theme_bw() +
  theme(plot.margin = margin(c(4, 15, -8, 4)),
        plot.caption = element_text(hjust = 0.5, vjust = 7),
        legend.position = "bottom",
        legend.title = element_text(size = 10),
        legend.text = element_text(size = 10),
        legend.box.margin = margin(c(-30, 6, 6, 6)),
        legend.background = element_rect(fill = "transparent"))
}
}
\references{
Keller, B. T., & Enders, C. K. (2023). \emph{Blimp user’s guide} (Version 3).
Retrieved from www.appliedmissingdata.com/blimp
}
\seealso{
\code{\link{blimp}}, \code{\link{blimp.update}}, \code{\link{blimp.run}},
\code{\link{blimp.print}}, \code{\link{blimp.plot}}, \code{\link{blimp.bayes}}
}
\author{
Takuya Yanagida
}
