% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vlmc_simulate_cpp.R
\name{simulate.vlmc_cpp}
\alias{simulate.vlmc_cpp}
\title{Simulate a discrete time series for a vlmc}
\usage{
\method{simulate}{vlmc_cpp}(
  object,
  nsim = 1,
  seed = NULL,
  init = NULL,
  burnin = 0L,
  sample = c("fast", "slow", "R"),
  ...
)
}
\arguments{
\item{object}{a fitted vlmc object.}

\item{nsim}{length of the simulated time series (defaults to 1).}

\item{seed}{an optional random seed (see the dedicated section).}

\item{init}{an optional initial sequence for the time series.}

\item{burnin}{number of initial observations to discard or \code{"auto"} (see the
dedicated section).}

\item{sample}{specifies which implementation of \code{\link[base:sample]{base::sample()}} to use.
See the dedicated section.}

\item{...}{additional arguments.}
}
\value{
a simulated discrete time series of the same type as the one used to
build the vlmc with a \code{seed} attribute (see the Random seed section). The
results has also the \code{dts} class to hide the \code{seed} attribute when using
\code{print} or similar function.
}
\description{
This function simulates a time series from the distribution estimated by the
given vlmc object.
}
\details{
The time series can be initiated by a fixed sequence specified via the \code{init}
parameter.
}
\section{sampling method}{


The R backend for \code{\link[=vlmc]{vlmc()}} uses \code{\link[base:sample]{base::sample()}} to generate samples for each
context. Internally, this function sorts the probabilities of each state in
decreasing probability order (among other things), which is not needed in our
case. The C++ backend can be used with three different implementations:
\itemize{
\item \code{sample="fast"} uses a dedicated C++ implementation adapted to the data structures
used internally. In general, the simulated time series obtained with this
implementation will be different from the one generated with the R backend,
even using the same seed.
\item \code{sample="slow"} uses another C++ implementation that mimics \code{\link[base:sample]{base::sample()}} in
order to maximize the chance to provide identical simulation results regardless
of the backend (when using the same random seed). This process is not perfect
as we use the std::lib sort algorithm which is not guaranteed to give identical
results as the ones of R internal 'revsort'.
\item \code{sample="R"} uses direct calls to \code{\link[base:sample]{base::sample()}}. Results are guaranteed
to be identical between the two backends, but at the price of higher running
time.
}
}

\section{Burn in (Warm up) period}{


When using a VLMC for simulation purposes, we are generally interested in
the stationary distribution of the corresponding Markov chain. To reduce
the dependence of the samples from the initial values and get closer to
this stationary distribution (if it exists), it is recommended to discard
the first samples which are produced in a so-called "burn in" (or "warm
up") period. The \code{burnin} parameter can be used to implement this approach.
The VLMC is used to produce a sample of size \code{burnin + nsim} but the first
\code{burnin} values are discarded. Notice that this burn in values can be
partially given by the \code{init} parameter if it is specified.

If \code{burnin} is set to \code{"auto"}, the \code{burnin} period is set to \code{64 * context_number(object)}, following the heuristic proposed in Mächler and
Bühlmann (2004).
}

\section{Random seed}{


This function reproduce the behaviour of \code{\link[stats:simulate]{stats::simulate()}}. If \code{seed} is
\code{NULL} the function does not change the random generator state and returns
the value of \link{.Random.seed} as a \code{seed} attribute in the return value. This
can be used to reproduce exactly the simulation results by setting
\link{.Random.seed} to this value. Notice that if the random seed has not be
initialised by R so far, the function issues a call to \code{runif(1)} to
perform this initialisation (as is done in \code{\link[stats:simulate]{stats::simulate()}}).

It \code{seed} is an integer, it is used in a call to \code{\link[=set.seed]{set.seed()}} before the
simulation takes place. The integer is saved as a \code{seed} attribute in the
return value. The integer seed is completed by an attribute \code{kind} which
contains the value \verb{as.list([RNGkind()])} exactly as with
\code{\link[stats:simulate]{stats::simulate()}}. The random generator state is reset to its original
value at the end of the call.
}

\section{Extended contexts}{


As explained in details in \code{\link[=loglikelihood.vlmc]{loglikelihood.vlmc()}} documentation and in the
dedicated \code{vignette("likelihood", package = "mixvlmc")}, the first initial
values of a time series do not in general have a proper context for a VLMC
with a non zero order. In order to simulate something meaningful for those
values when \code{init} is not provided, we rely on the notion of extended
context defined in the documents mentioned above. This follows the same
logic as using \code{\link[=loglikelihood.vlmc]{loglikelihood.vlmc()}} with the parameter
\code{initial="extended"}. All vlmc functions that need to manipulate initial
values with no proper context use the same approach.
}

\examples{
pc <- powerconsumption[powerconsumption$week == 5, ]
dts <- cut(pc$active_power, breaks = c(0, quantile(pc$active_power, probs = c(0.25, 0.5, 0.75, 1))))
model <- vlmc(dts, min_size = 5)
new_dts <- simulate(model, 500, seed = 0)
new_dts_2 <- simulate(model, 500, seed = 0, init = dts[1:5])
new_dts_3 <- simulate(model, 500, seed = 0, burnin = 500)
}
\references{
Mächler, M. and Bühlmann, P. (2004) "Variable Length Markov
Chains: Methodology, Computing, and Software" Journal of Computational and
Graphical Statistics, 13 (2), 435-455, \doi{10.1198/1061860043524}
}
\seealso{
\code{\link[stats:simulate]{stats::simulate()}} for details and examples on the random number
generator setting
}
