% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_means.R
\name{estimate_means}
\alias{estimate_means}
\title{Estimate Marginal Means (Model-based average at each factor level)}
\usage{
estimate_means(
  model,
  at = "auto",
  fixed = NULL,
  transform = "response",
  ci = 0.95,
  ...
)
}
\arguments{
\item{model}{A statistical model.}

\item{at}{The predictor variable(s) \emph{at} which to evaluate the desired effect / mean / contrasts. Other predictors of the model that are not included here will be collapsed and "averaged" over (the effect will be estimated across them).}

\item{fixed}{A character vector indicating the names of the predictors to be
"fixed" (i.e., maintained), so that the estimation is made at these values.}

\item{transform}{Is passed to the \code{type} argument in
\code{emmeans::emmeans()}. See
\href{https://CRAN.R-project.org/package=emmeans/vignettes/transformations.html}{this vignette}. Can be \code{"none"} (default for contrasts), \code{"response"}
(default for means), \code{"mu"}, \code{"unlink"}, \code{"log"}.
\code{"none"} will leave the values on scale of the linear predictors.
\code{"response"} will transform them on scale of the response variable.
Thus for a logistic model, \code{"none"} will give estimations expressed in
log-odds (probabilities on logit scale) and \code{"response"} in terms of
probabilities.}

\item{ci}{Confidence Interval (CI) level. Default to \code{0.95} (\verb{95\%}).}

\item{...}{Other arguments passed for instance to \code{\link[=visualisation_matrix]{visualisation_matrix()}}.}
}
\value{
A dataframe of estimated marginal means.
}
\description{
Estimate average value of response variable at each factor levels. For
plotting, check the examples in \code{\link[=visualisation_recipe]{visualisation_recipe()}}. See also
other related functions such as \code{\link[=estimate_contrasts]{estimate_contrasts()}} and
\code{\link[=estimate_slopes]{estimate_slopes()}}.
}
\details{
The \code{\link[=estimate_slopes]{estimate_slopes()}}, \code{\link[=estimate_means]{estimate_means()}} and \code{\link[=estimate_contrasts]{estimate_contrasts()}} functions are forming a group, as they are all based on \emph{marginal} estimations (estimations based on a model). All three are also built on the \pkg{emmeans} package, so reading its documentation (for instance for \code{\link[emmeans:emmeans]{emmeans::emmeans()}} and \code{\link[emmeans:emtrends]{emmeans::emtrends()}}) is recommended to understand the idea behind these types of procedures.

\itemize{
\item Model-based \strong{predictions} is the basis for all that follows. Indeed, the first thing to understand is how models can be used to make predictions (see \code{\link[=estimate_link]{estimate_link()}}). This corresponds to the predicted response (or "outcome variable") given specific predictor values of the predictors (i.e., given a specific data configuration). This is why the concept of \code{\link[=visualisation_matrix]{reference grid()}} is so important for direct predictions.
\item \strong{Marginal "means"}, obtained via \code{\link[=estimate_means]{estimate_means()}}, are an extension of such predictions, allowing to "average" (collapse) some of the predictors, to obtain the average response value at a specific predictors configuration. This is typically used when some of the predictors of interest are factors. Indeed, the parameters of the model will usually give you the intercept value and then the "effect" of each factor level (how different it is from the intercept). Marginal means can be used to directly give you the mean value of the response variable at all the levels of a factor. Moreover, it can also be used to control, or average over predictors, which is useful in the case of multiple predictors with or without interactions.
\item \strong{Marginal contrasts}, obtained via \code{\link[=estimate_contrasts]{estimate_contrasts()}}, are themselves at extension of marginal means, in that they allow to investigate the difference (i.e., the contrast) between the marginal means. This is, again, often used to get all pairwise differences between all levels of a factor. It works also for continuous predictors, for instance one could also be interested in whether the difference at two extremes of a continuous predictor is significant.
\item Finally, \strong{marginal effects}, obtained via \code{\link[=estimate_slopes]{estimate_slopes()}}, are different in that their focus is not values on the response variable, but the model's parameters. The idea is to assess the effect of a predictor at a specific configuration of the other predictors. This is relevant in the case of interactions or non-linear relationships, when the effect of a predictor variable changes depending on the other predictors. Moreover, these effects can also be "averaged" over other predictors, to get for instance the "general trend" of a predictor over different factor levels.
}
\strong{Example:} let's imagine the following model \code{lm(y ~ condition * x)} where \code{condition} is a factor with 3 levels A, B and C and \code{x} a continuous variable (like age for example). One idea is to see how this model performs, and compare the actual response y to the one predicted by the model (using \code{\link[=estimate_response]{estimate_response()}}). Another idea is evaluate the average mean at each of the condition's levels (using \code{\link[=estimate_means]{estimate_means()}}), which can be useful to visualize them. Another possibility is to evaluate the difference between these levels (using \code{\link[=estimate_contrasts]{estimate_contrasts()}}). Finally, one could also estimate the effect of x averaged over all conditions, or instead within each condition (\code{using [estimate_slopes]}).
}
\examples{
library(modelbased)

# Frequentist models
# -------------------
model <- lm(Petal.Length ~ Sepal.Width * Species, data = iris)

estimate_means(model)
estimate_means(model, fixed = "Sepal.Width")
estimate_means(model, at = c("Species", "Sepal.Width"), length = 2)
estimate_means(model, at = "Species=c('versicolor', 'setosa')")
estimate_means(model, at = "Sepal.Width=c(2, 4)")
estimate_means(model, at = c("Species", "Sepal.Width=0"))
estimate_means(model, at = "Sepal.Width", length = 5)
estimate_means(model, at = "Sepal.Width=c(2, 4)")

# Methods that can be applied to it:
means <- estimate_means(model, fixed = "Sepal.Width")
plot(means) # which runs visualisation_recipe()
effectsize::standardize(means)
\donttest{
if (require("lme4")) {
  data <- iris
  data$Petal.Length_factor <- ifelse(data$Petal.Length < 4.2, "A", "B")

  model <- lmer(Petal.Length ~ Sepal.Width + Species + (1 | Petal.Length_factor), data = data)
  estimate_means(model)
  estimate_means(model, at = "Sepal.Width", length = 3)
}
}

}
