% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelsummary.R
\name{msummary}
\alias{msummary}
\title{\code{msummary()} is a shortcut to \code{modelsummary()}}
\usage{
msummary(
  models,
  output = "default",
  fmt = 3,
  estimate = "estimate",
  statistic = "std.error",
  vcov = NULL,
  conf_level = 0.95,
  stars = FALSE,
  coef_map = NULL,
  coef_omit = NULL,
  coef_rename = NULL,
  gof_map = NULL,
  gof_omit = NULL,
  group = term ~ model,
  group_map = NULL,
  add_rows = NULL,
  align = NULL,
  notes = NULL,
  title = NULL,
  ...
)
}
\arguments{
\item{models}{a model or (optionally named) list of models}

\item{output}{filename or object type (character string)
\itemize{
\item Supported filename extensions: .html, .tex, .md, .txt, .png, .jpg.
\item Supported object types: "default", "html", "markdown", "latex", "latex_tabular", "data.frame", "modelsummary_list", "gt", "kableExtra", "huxtable", "flextable".
\item To change the default output format, type \code{options(modelsummary_default = "latex")}, where \code{latex} can be any of the valid object types listed above.
\item Warning: users should not supply a file name to the \code{output} argument if they intend to customize the table with external packages.
\item See the 'Details' section below for more information.
}}

\item{fmt}{determines how to format numeric values
\itemize{
\item integer: the number of digits to keep after the period \code{format(round(x, fmt), nsmall=fmt)}
\item character: passed to the \code{sprintf} function (e.g., '\%.3f' keeps 3 digits with trailing zero). See \code{?sprintf}
\item function: returns a formatted character string.
}}

\item{estimate}{string or \code{glue} string of the estimate to display (or a
vector with one string per model). Valid entries include any column name of
the data.frame produced by \code{get_estimates(model)}. Examples:
\itemize{
\item \code{"estimate"}
\item \code{"{estimate} ({std.error}){stars}"}
\item \code{"{estimate} [{conf.low}, {conf.high}]"}
}}

\item{statistic}{vector of strings or \code{glue} strings which select uncertainty
statistics to report vertically below the estimate. NULL omits all
uncertainty statistics.
\itemize{
\item "conf.int", "std.error", "statistic", "p.value", "conf.low", "conf.high",
or any column name produced by: \code{get_estimates(model)}
\item \code{glue} package strings with braces, such as:
\itemize{
\item \code{"{p.value} [{conf.low}, {conf.high}]"}
\item \code{"Std.Error: {std.error}"}
}
\item Note: Parentheses are added automatically unless the string includes \code{glue} curly braces \code{{}}.
\item Note: To report uncertainty statistics \emph{next} to coefficients, you can #'   supply a \code{glue} string to the \code{estimate} argument.
}}

\item{vcov}{robust standard errors and other manual statistics. The \code{vcov}
argument accepts six types of input (see the 'Details' and 'Examples'
sections below):
\itemize{
\item NULL returns the default uncertainty estimates of the model object
\item string, vector, or (named) list of strings. The strings "classical", "iid" and "constant" are aliases for \code{NULL}, and they return the model's default uncertainty estimates. The strings "robust", "HC", "HC0", "HC1", "HC2", "HC3", "HC4", "HC4m", "HC5", "stata", "HAC", "NeweyWest", "Andrews", "panel-corrected", "outer-product", "weave" use variance-covariance matrices computed using functions from the \code{sandwich} package. The behavior of those functions can (and sometimes \emph{must}) be altered by passing arguments to \code{sandwich} directly from \code{modelsummary} through the ellipsis (\code{...}), but it is safer to define your own custom functions as described in the next bullet.
\item function or (named) list of functions which return variance-covariance matrices with row and column names equal to the names of your coefficient estimates (e.g., \code{stats::vcov}, \code{sandwich::vcovHC}, \code{function(x) vcovPC(x, cluster="country")}).
\item formula or (named) list of formulas with the cluster variable(s) on the right-hand side (e.g., ~clusterid).
\item (named) list of \code{length(models)} variance-covariance matrices with row and column names equal to the names of your coefficient estimates.
\item a (named) list of length(models) vectors with names equal to the names of your coefficient estimates. See 'Examples' section below. Warning: since this list of vectors can include arbitrary strings or numbers, \code{modelsummary} cannot automatically calculate p values. The \code{stars} argument may thus use incorrect significance thresholds when \code{vcov} is a list of vectors.
}}

\item{conf_level}{confidence level to use for confidence intervals}

\item{stars}{to indicate statistical significance
\itemize{
\item FALSE (default): no significance stars.
\item TRUE: +=.1, *=.05, **=.01, ***=0.001
\item Named numeric vector for custom stars such as \code{c('*' = .1, '+' = .05)}
\item Note: a legend will not be inserted at the bottom of the table when the \code{estimate} or \code{statistic} arguments use "glue strings" with \code{{stars}}.
}}

\item{coef_map}{character vector. Subset, rename, and reorder coefficients.
Coefficients omitted from this vector are omitted from the table. The order
of the vector determines the order of the table.  \code{coef_map} can be a named
or an unnamed character vector (see the Examples section below). If
\code{coef_map} is a named vector, its values define the labels that must appear
in the table, and its names identify the original term names stored in the
model object: \code{c("hp:mpg"="HPxM/G")}.}

\item{coef_omit}{string regular expression. Omits all matching coefficients
from the table using \code{grepl(perl=TRUE)}.}

\item{coef_rename}{named character vector or function which returns a named
vector. Values of the vector refer to the variable names that will appear
in the table. Names refer to the original term names stored in the model
object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.}

\item{gof_map}{rename, reorder, and omit goodness-of-fit statistics and other
model information. This argument accepts 3 types of values:
\itemize{
\item NULL (default): the \code{modelsummary::gof_map} dictionary is used for formatting, and all unknown statistic are included.
\item data.frame with 3 columns named "raw", "clean", "fmt". Unknown statistics are omitted. See the 'Examples' section below.
\item list of lists, each of which includes 3 elements named "raw", "clean", "fmt". Unknown statistics are omitted. See the 'Examples section below'.
}}

\item{gof_omit}{string regular expression. Omits all matching gof statistics from
the table (using \code{grepl(perl=TRUE)}).}

\item{group}{a two-sided formula with two or three components which describes
how groups of parameters should be displayed. The formula must include both
a "term" and a "model" component. In addition, a component can be used to
identify groups of parameters (e.g., outcome levels of a multinomial logit
model). This group identifier must be the name of a column in the
data.frame produced by \code{get_estimates(model)}.
\itemize{
\item \code{term ~ model} displays coefficients as rows and models as columns
\item \code{model ~ term} displays models as rows and coefficients as columns
\item \code{response + term ~ model} displays response levels and coefficients as rows and models as columns.
}}

\item{group_map}{named or unnamed character vector. Subset, rename, and
reorder coefficient groups specified in the \code{group} argument. See \code{coef_map}.}

\item{add_rows}{a data.frame (or tibble) with the same number of columns as
your main table. By default, rows are appended to the bottom of the table.
You can define a "position" attribute of integers to set the row positions.
See Examples section below.}

\item{align}{A character string of length equal to the number of columns in
the table.  "lcr" means that the first column will be left-aligned, the 2nd
column center-aligned, and the 3rd column right-aligned.}

\item{notes}{list or vector of notes to append to the bottom of the table.}

\item{title}{string}

\item{...}{all other arguments are passed through to the extractor and
table-making functions. This allows users to pass arguments directly to
\code{modelsummary} in order to affect the behavior of other functions behind
the scenes. Examples include:
\itemize{
\item \code{broom::tidy(exponentiate=TRUE)} to exponentiate logistic regression
\item \code{kableExtra::kbl(escape=FALSE)} to avoid escaping math characters in \code{kableExtra} tables.
\item \code{performance::model_performance(metrics="RMSE")} to select goodness-of-fit statistics to extract using the \code{performance} package (must have set \code{options(modelsummary_get="easystats")} first).
}}
}
\value{
a regression table in a format determined by the \code{output} argument.
}
\description{
The content of the tables can be altered with the function's arguments, or by
calling \code{options}, as described in the \emph{Details} section below. The look of
the tables can be customized by specifying the \code{output} argument, and by
using functions from one of the supported table customization packages:
\code{kableExtra}, \code{gt}, \code{flextable}, \code{huxtable}.
}
\details{
\code{options}

\code{modelsummary} supports 4 table-making packages: \code{kableExtra}, \code{gt},
\code{flextable}, and \code{huxtable}. Some of these packages have overlapping
functionalities. For example, 3 of those packages can export to LaTeX. To
change the default backend used for a specific file format, you can use
the \code{options} function:

\code{options(modelsummary_html = 'kableExtra')}
\code{options(modelsummary_latex = 'gt')}
\code{options(modelsummary_word = 'huxtable')}
\code{options(modelsummary_png = 'gt')}

\code{modelsummary} can use two sets of packages to extract information from
statistical models: \code{broom} and the \code{easystats} family (\code{performance} and
\code{parameters}). By default, it uses \code{broom} first and \code{easystats} as a
fallback if \code{broom} fails. You can change the order of priorities
or include goodness-of-fit extracted by \emph{both} packages by setting:

\code{options(modelsummary_get = "broom")}
\code{options(modelsummary_get = "easystats")}
\code{options(modelsummary_get = "all")}

\code{output} argument:

The \code{modelsummary_list} output type is a lightweight representation of the
model results. The \code{modelsummary} function can export to this format by
setting the \code{output} argument, and it can accept objects of this format
as input models to create a table. This can be useful to save raw
results, in order to print a table later, without having to save and
extract from the entire model object. Note that the confidence intervals
are only stored in a \code{modelsummary_list} if explicitly requested:

\verb{backup <- modelsummary(models, output = "modelsummary_list"}
\verb{                       statistic = "conf.int")}
\code{modelsummary(backup)}

When a file name with a valid extension is supplied to the \code{output} argument,
the table is written immediately to file. If you want to customize your table
by post-processing it with an external package, you need to choose a
different output format and saving mechanism. Unfortunately, the approach
differs from package to package:
\itemize{
\item \code{gt}: set \code{output="gt"}, post-process your table, and use the \code{gt::gtsave} function.
\item \code{kableExtra}: set \code{output} to your destination format (e.g., "latex", "html", "markdown"), post-process your table, and use \code{kableExtra::save_kable} function.
}

\code{vcov} argument:

To use a string such as "robust" or "HC0", your model must be supported
by the \code{sandwich} package. This includes objects such as: lm, glm,
survreg, coxph, mlogit, polr, hurdle, zeroinfl, and more.

NULL, "classical", "iid", and "constant" are aliases which do not modify
uncertainty estimates and simply report the default standard errors stored
in the model object.

One-sided formulas such as \code{~clusterid} are passed to the \code{sandwich::vcovCL}
function.

Matrices and functions producing variance-covariance matrices are first
passed to \code{lmtest}. If this does not work, \code{modelsummary} attempts to take
the square root of the diagonal to adjust "std.error", but the other
uncertainty estimates are not be adjusted.

Numeric vectors are formatted according to \code{fmt} and placed in brackets.
Character vectors printed as given, without parentheses.

If your model type is supported by the \code{lmtest} package, the
\code{vcov} argument will try to use that package to adjust all the
uncertainty estimates, including "std.error", "statistic", "p.value", and
"conf.int". If your model is not supported by \code{lmtest}, only the "std.error"
will be adjusted by, for example, taking the square root of the matrix's
diagonal.
}
\examples{
\dontrun{

# The `modelsummary` website includes \emph{many} examples and tutorials:
# https://vincentarelbundock.github.io/modelsummary

library(modelsummary)

# load data and estimate models
data(trees)
models <- list()
models[['Bivariate']] <- lm(Girth ~ Height, data = trees)
models[['Multivariate']] <- lm(Girth ~ Height + Volume, data = trees)

# simple table
modelsummary(models)

# statistic
modelsummary(models, statistic = NULL)
modelsummary(models, statistic = 'p.value')
modelsummary(models, statistic = 'statistic')
modelsummary(models, statistic = 'conf.int', conf_level = 0.99)
modelsummary(models, statistic = c("t = {statistic}",
                                   "se = {std.error}",
                                   "conf.int"))

# estimate
modelsummary(models,
  statistic = NULL,
  estimate = "{estimate} [{conf.low}, {conf.high}]")
modelsummary(models,
  estimate = c("{estimate}{stars}",
               "{estimate} ({std.error})"))

# vcov
modelsummary(models, vcov = "robust")
modelsummary(models, vcov = list("classical", "stata"))
modelsummary(models, vcov = sandwich::vcovHC)
modelsummary(models,
  vcov = list(stats::vcov, sandwich::vcovHC))
modelsummary(models,
  vcov = list(c("(Intercept)"="", "Height"="!"),
              c("(Intercept)"="", "Height"="!", "Volume"="!!")))

# vcov with custom names
modelsummary(
  models,
  vcov = list("Stata Corp" = "stata",
              "Newey Lewis & the News" = "NeweyWest"))

# coef_rename
modelsummary(models, coef_map = c('Volume' = 'Large', 'Height' = 'Tall'))

# coef_map
modelsummary(models, coef_map = c('Volume' = 'Large', 'Height' = 'Tall'))
modelsummary(models, coef_map = c('Volume', 'Height'))

# title
modelsummary(models, title = 'This is the title')

# title with LaTeX label (for numbering and referencing)
modelsummary(models, title = 'This is the title \\\\label{tab:description}')

# add_rows
rows <- tibble::tribble(~term, ~Bivariate, ~Multivariate,
  'Empty row', '-', '-',
  'Another empty row', '?', '?')
attr(rows, 'position') <- c(1, 3)
modelsummary(models, add_rows = rows)

# notes
modelsummary(models, notes = list('A first note', 'A second note'))

# gof_map: data.frame
gm <- modelsummary::gof_map
gof_custom$omit[gof_custom$raw == 'deviance'] <- FALSE
gof_custom$fmt[gof_custom$raw == 'r.squared'] <- "\%.5f"
modelsummary(models, gof_map = gof_custom)

# gof_map: list of lists
f1 <- function(x) format(round(x, 3), big.mark=",")
f2 <- function(x) format(round(x, 0), big.mark=",")
gm <- list(
  list("raw" = "nobs", "clean" = "N", "fmt" = f2),
  list("raw" = "AIC", "clean" = "aic", "fmt" = f1))
modelsummary(models,
  fmt = f1,
  gof_map = gm)

}

}
\keyword{internal}
