% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parsnip-naive_reg.R
\name{naive_reg}
\alias{naive_reg}
\title{General Interface for NAIVE Forecast Models}
\usage{
naive_reg(mode = "regression", id = NULL, seasonal_period = NULL)
}
\arguments{
\item{mode}{A single character string for the type of model.
The only possible value for this model is "regression".}

\item{id}{An optional quoted column name (e.g. "id") for
identifying multiple time series (i.e. panel data).}

\item{seasonal_period}{SNAIVE only. A seasonal frequency. Uses "auto" by default.
A character phrase of "auto" or time-based phrase of "2 weeks"
can be used if a date or date-time variable is provided.
See Fit Details below.}
}
\description{
\code{naive_reg()} is a way to generate a \emph{specification} of an NAIVE or SNAIVE model
before fitting and allows the model to be created using
different packages.
}
\details{
The data given to the function are not saved and are only used
to determine the \emph{mode} of the model. For \code{naive_reg()}, the
mode will always be "regression".

The model can be created using the \code{fit()} function using the
following \emph{engines}:
\itemize{
\item "naive" (default) - Performs a NAIVE forecast
\item "snaive" - Performs a Seasonal NAIVE forecast
}
}
\section{Engine Details}{


\strong{naive (default engine)}
\itemize{
\item The engine uses \code{\link[=naive_fit_impl]{naive_fit_impl()}}
\item The NAIVE implementation uses the last observation and forecasts this value forward.
\item The \code{id} can be used to distinguish multiple time series contained in
the data
\item The \code{seasonal_period} is not used but provided for consistency with the SNAIVE
implementation
}

\strong{snaive (default engine)}
\itemize{
\item The engine uses \code{\link[=snaive_fit_impl]{snaive_fit_impl()}}
\item The SNAIVE implementation uses the last seasonal series in the data
and forecasts this sequence of observations forward
\item The \code{id} can be used to distinguish multiple time series contained in
the data
\item The \code{seasonal_period} is used to determine how far back to define the repeated
series. This can be a numeric value (e.g. 28) or a period (e.g. "1 month")
}
}

\section{Fit Details}{


\strong{Date and Date-Time Variable}

It's a requirement to have a date or date-time variable as a predictor.
The \code{fit()} interface accepts date and date-time features and handles them internally.
\itemize{
\item \code{fit(y ~ date)}
}

\strong{ID features (Multiple Time Series, Panel Data)}

The \code{id} parameter is populated using the \code{fit()} or \code{fit_xy()} function:

\emph{ID Example:} Suppose you have 3 features:
\enumerate{
\item \code{y} (target)
\item \code{date} (time stamp),
\item \code{series_id} (a unique identifer that identifies each time series in your data).
}

The \code{series_id} can be passed to the \code{naive_reg()} using
\code{fit()}:
\itemize{
\item \code{naive_reg(id = "series_id")} specifes that the \code{series_id} column should be used
to identify each time series.
\item \code{fit(y ~ date + series_id)} will pass \code{series_id} on to the underlying
naive or snaive functions.
}

\strong{Seasonal Period Specification (snaive)}

The period can be non-seasonal (\verb{seasonal_period = 1 or "none"}) or
yearly seasonal (e.g. For monthly time stamps, \code{seasonal_period = 12}, \code{seasonal_period = "12 months"}, or \code{seasonal_period = "yearly"}).
There are 3 ways to specify:
\enumerate{
\item \code{seasonal_period = "auto"}: A seasonal period is selected based on the periodicity of the data (e.g. 12 if monthly)
\item \code{seasonal_period = 12}: A numeric frequency. For example, 12 is common for monthly data
\item \code{seasonal_period = "1 year"}: A time-based phrase. For example, "1 year" would convert to 12 for monthly data.
}

\strong{External Regressors (Xregs)}

These models are univariate. No xregs are used in the modeling process.
}

\examples{
library(dplyr)
library(parsnip)
library(rsample)
library(timetk)
library(modeltime)

# Data
m750 <- m4_monthly \%>\% filter(id == "M750")
m750

# Split Data 80/20
splits <- initial_time_split(m750, prop = 0.8)

# ---- NAIVE ----

# Model Spec
model_spec <- naive_reg() \%>\%
    set_engine("naive")

# Fit Spec
model_fit <- model_spec \%>\%
    fit(log(value) ~ date, data = training(splits))
model_fit


# ---- SEASONAL NAIVE ----

# Model Spec
model_spec <- naive_reg(
        id = "id",
        seasonal_period = 12
    ) \%>\%
    set_engine("snaive")

# Fit Spec
model_fit <- model_spec \%>\%
    fit(log(value) ~ date + id, data = training(splits))
model_fit

}
\seealso{
\code{\link[=fit.model_spec]{fit.model_spec()}}, \code{\link[=set_engine]{set_engine()}}
}
