% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bimodality.R
\name{mt_check_bimodality}
\alias{mt_check_bimodality}
\title{Assess bimodality of mouse-tracking measure distributions.}
\usage{
mt_check_bimodality(
  data,
  use = "measures",
  use_variables = NULL,
  methods = c("BC", "HDS"),
  B = 2000,
  grouping_variables = NULL,
  ...
)
}
\arguments{
\item{data}{a mousetrap data object created using one of the mt_import 
functions (see \link{mt_example} for details).}

\item{use}{a character string specifying which data should be used. By
default, points to the \code{measures} data.frame created using
\link{mt_measures}.}

\item{use_variables}{a vector specifying for which mouse-tracking measures
bimodality should be assessed.}

\item{methods}{a character string (or vector) specifying which methods should
be used for assessing bimodality (see Details).}

\item{B}{an integer specifying the number of replicates used in the Monte
Carlo test (only relevant if "HDS_sim" is included in methods, see
Details).}

\item{grouping_variables}{a character string (or vector) specifying one or
more variables in \code{data[["data"]]}. If specified, bimodality will be
assessed separately for each level of the variable. If unspecified (the
default), bimodality is checked across all trials.}

\item{...}{additional arguments passed on to \link{mt_reshape} (such as
\code{subset}).}
}
\value{
A list of several data.frames. Each data.frame contains the value
  returned by the respective method for assessing bimodality (see Details) - 
  separately per condition (specified in the row dimension) and measure
  (specified in the column dimension).
}
\description{
Assess bimodality of the distribution of mouse-tracking measures using the 
bimodality coefficient and Hartigan's dip statistic (see Details). If 
bimodality should be assessed separately for different conditions, the 
corresponding variables can be specified under \code{grouping_variables}.
}
\details{
Different methods have been suggested for assessing the bimodality of 
mouse-tracking measure distributions, each of which has advantages and
disadvantages (see Freeman & Dale, 2013).

Hehman et al. (2015) focus on two specific methods (bimodality coefficient 
and Hartigan's dip statistic) which are implemented here.

If \code{methods} include \code{BC}, the bimodality coefficient is calculated
using the \link{bimodality_coefficient} function in this package. According 
to Freeman and Ambady (2010), a distribution is considered bimodal if 
\code{BC > 0.555}.

Note that MouseTracker (Freeman & Ambady, 2010) standardizes variables within
each subject before computing the BC. This is also possible here using
\link{mt_standardize} (see Examples).

If \code{methods} include \code{HDS}, Hartigan's dip statistic is calculated 
using the \link[diptest]{dip.test} function of the \code{diptest} package. 
The corresponding p value (computed via linear interpolation) is returned.

If \code{methods} include \code{HDS_sim}, Hartigan's dip statistic is 
calculated using the \link[diptest]{dip.test} function with the additional 
argument \code{simulate.p.values=TRUE}. In this case, the p value is computed
from a Monte Carlo simulation of a uniform distribution with B (default: 
2000) replicates.
}
\examples{
# Calculate measures
mt_example <- mt_measures(mt_example)

# Assess bimodality for untransformed variables
mt_check_bimodality(mt_example,
  use_variables=c("MAD", "AD"))

# Standardize variables per participant
mt_example <- mt_standardize(mt_example,
  use_variables=c("MAD", "AD"), within="subject_nr")
  
# Assess bimodality for standardized variables
mt_check_bimodality(mt_example,
  use_variables=c("z_MAD", "z_AD"))

# Assess bimodality with simulated p values for HDS
mt_check_bimodality(mt_example,
  use_variables=c("z_MAD", "z_AD"),
  methods=c("BC", "HDS_sim"))
 
# Assess bimodality per condition
mt_check_bimodality(mt_example,
  use_variables=c("z_MAD", "z_AD"),
  grouping_variables="Condition")
                    
}
\references{
Freeman, J. B., & Ambady, N. (2010). MouseTracker: Software for studying
real-time mental processing using a computer mouse-tracking method.
\emph{Behavior Research Methods, 42}(1), 226-241.

Freeman, J. B., & Dale, R. (2013).
Assessing bimodality to detect the presence of a dual cognitive process.
\emph{Behavior Research Methods, 45}(1), 83-97.

Hehman, E., Stolier, R. M., & Freeman, J. B. (2015). Advanced mouse-tracking
analytic techniques for enhancing psychological science. \emph{Group
Processes & Intergroup Relations, 18}(3), 384-401.
}
\seealso{
\link{bimodality_coefficient} for more information about the bimodality
  coefficient.

  \link[diptest]{dip.test} for more information about Hartigan's dip test.
}
\author{
Pascal J. Kieslich

Felix Henninger
}
