% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_sa.R
\name{fit_sa}
\alias{fit_sa}
\title{Stochastic Approximation fitting of MRFs on 2d lattices}
\usage{
fit_sa(
  Z,
  mrfi,
  family = "onepar",
  gamma_seq,
  init = 0,
  cycles = 5,
  refresh_each = length(gamma_seq) + 1,
  refresh_cycles = 60,
  verbose = interactive()
)
}
\arguments{
\item{Z}{A \code{matrix} object containing the observed MRF. \code{NA} values can be
used to create a subregion of the lattice for non-rectangular data.}

\item{mrfi}{A \code{\link[=mrfi-class]{mrfi}} object representing the
interaction structure.}

\item{family}{The family of parameter restrictions to potentials. Families
are:
\code{'onepar'}, \code{'oneeach'}, \code{'absdif'}, \code{'dif'} or \code{'free'}.
See \code{\link[=mrf2d-family]{mrf2d-familiy}}.}

\item{gamma_seq}{A \code{numeric} vector with the sequence of constants
used in each step \eqn{\gamma_t}.}

\item{init}{The initial value to be used in the optimization. It can be:
\itemize{
\item A valid \code{array} of parameter values according to \code{family}.
\item \code{0}. If set to \code{0} an array with `0`` in all entries is created.
}}

\item{cycles}{The number of updates to be done (for each each pixel).}

\item{refresh_each}{An integer with the number of iterations taken before a
complete refresh (restart from a random state). This prevents the sample from
being stuck in a mode for too long. Defaults to \code{length(gamma_seq) + 1} (no
refresh happens).}

\item{refresh_cycles}{An integer indicating how many Gibbs Sampler cycles are
performed when a refresh happens. Larger is usually better, but slower.}

\item{verbose}{\code{logical} indicating whether the iteration number is printed
during execution.}
}
\value{
A \code{mrfout} object with the following elements:
\itemize{
\item \code{theta}: The estimated \code{array} of potentials.
\item \code{mrfi}: The interaction structure considered.
\item \code{family}: The parameter restriction family considered.
\item \code{method}: The estimation method (\code{"Stochastic Approximation"}).
\item \code{metrics}: A \code{data.frame} containing the the euclidean distance between
the sufficient statics computed for \code{Z} and the current sample.
}
}
\description{
Estimates the parameters of a MRF by successively sampling from
a parameter configuration and updating it by comparing the sufficient statistics
of the sampled field and the observed field.

This method aims to find the parameter value where the gradient of the
likelihood function is equal to zero.
}
\details{
The stochastic approximation method consists of, given an observed field \code{Z},
and a starting parameters configuration \eqn{\theta_0}, successively sample
a field \eqn{Z_t} from the current parameter configuration and estimate the
direction of the  gradient of the likelihood function by comparing the
sufficient statistics in the current sample and the observed field.

The solution is updated by moving in the estimated direction with a predefined
step size \eqn{\gamma_t}, a new field \eqn{Z_{t+1}} is sampled using the new
parameter configuration and \eqn{Z_t} as an initial value, and the process is
repeated.

\deqn{\theta_{t+1} = \theta_t - \gamma_t(T(Z_t) - T(Z)),}
where \eqn{T(Z)} is the sufficient statistics for the reference field,
\eqn{T(Z_t)} is the sufficient statistics for a field sampled from
\eqn{\theta_t}.

\code{gamma_seq} is normalized internally by diving values by \code{length(Z)}, so the
choice of the sequence is invariant to the lattice dimensions. Typically, a
sequence like \code{seq(from = 1, to = 0, length.out = 1000)} should be used for
defining a sequence with \code{1000} steps. Some tuning of this sequence is
required.
}
\note{
Stochastic Approximation is called "Controllable Simulated Annealing" in
some references.

Examples where Stochastic Approximation is used with MRFs are
\insertCite{gimel_sa}{mrf2d}, \insertCite{CR}{mrf2d}.
}
\examples{
\donttest{
set.seed(2)
fit1 <- fit_sa(Z_potts, mrfi(1), family = "oneeach", gamma_seq = seq(1, 0, length.out = 50))
# Estimated parameters
fit1$theta
# A visualization of estimated gradient norm over iterations.
plot(fit1$metrics, type = "l")

fit_sa(Z_potts, mrfi(1), family = "oneeach", gamma_seq = seq(1, 0, length.out = 50))
}

}
\references{
\insertRef{wiki_sa}{mrf2d}

\insertAllCited{}
}
\seealso{
A paper with detailed description of the package can be found at
\url{https://arxiv.org/abs/2006.00383}
}
\author{
Victor Freguglia
}
