# Copyright (C) 2022 Hibiki AI Limited <info@hibiki-ai.com>
#
# This file is part of nanonext.
#
# nanonext is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# nanonext is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# nanonext. If not, see <https://www.gnu.org/licenses/>.

# nanonext - Utilities ---------------------------------------------------------

#' NNG Library Version
#'
#' Returns the versions of the 'libnng' and 'libmbedtls' libraries used.
#'
#' @return A character vector of length 2.
#'
#' @examples
#' nng_version()
#'
#' @export
#'
nng_version <- function() .Call(rnng_version)

#' Translate Error Codes
#'
#' Translate integer exit codes generated by the NNG library. All package
#'     functions return an integer exit code on error rather than the expected
#'     return value. These are classed 'errorValue' and may be checked by
#'     \code{\link{is_error_value}}.
#'
#' @param xc integer exit code to translate.
#'
#' @return A character vector.
#'
#' @examples
#' nng_error(1L)
#'
#' @export
#'
nng_error <- function(xc) .Call(rnng_strerror, xc)

#' Clock Utility
#'
#' Provides the number of elapsed milliseconds since an arbitrary reference time
#'     in the past. The reference time will be the same for a given program, but
#'     may differ between programs.
#'
#' @details A convenience function for building concurrent applications. The
#'     resolution of the clock depends on the underlying system timing facilities
#'     and may not be particularly fine-grained. This utility should however be
#'     faster than using base \code{Sys.time()}.
#'
#' @return A double.
#'
#' @examples
#' time <- mclock(); msleep(100); mclock() - time
#'
#' @export
#'
mclock <- function() .Call(rnng_clock)

#' Sleep Utility
#'
#' Sleep function. May block for longer than requested, with the actual wait
#'     time determined by the capabilities of the underlying system.
#'
#' @param msec integer number of milliseconds to block the caller.
#'
#' @return Invisible NULL.
#'
#' @details If 'msec' is non-integer, it will be coerced to integer. Non-numeric
#'     input will be ignored and return immediately.
#'
#' @examples
#' time <- mclock(); msleep(100); mclock() - time
#'
#' @export
#'
msleep <- function(msec) invisible(.Call(rnng_sleep, msec))

#' NNG Random Number Generator
#'
#' Strictly not for statistical analysis. Not reproducible. No ability to set
#'     a seed value. Provides random numbers suitable for system functions such
#'     as cryptographic key generation. Random values are obtained using
#'     platform-specific strong cryptographic random number facilities where
#'     available.
#'
#' @param n [default 1L] length of vector to return.
#'
#' @return A length 'n' vector of random positive doubles.
#'
#' @details If 'n' is non-integer, it will be coerced to integer; if a vector,
#'     only the first element will be used.
#'
#' @examples
#' random()
#' random(n = 3L)
#'
#' @export
#'
random <- function(n = 1L) .Call(rnng_random, n)

#' Create Device
#'
#' Creates a device which is a socket forwarder or proxy. Provides for improved
#'     horizontal scalability, reliability, and isolation.
#'
#' @param s1 a raw mode Socket.
#' @param s2 a raw mode Socket.
#'
#' @return NULL. If the device was successfully created, this function does not
#'     return.
#'
#' @details Only raw mode sockets may be used with this function. Sockets s1 and
#'     s2 must be compatible with each other, i.e. be opposite halves of a two
#'     protocol pattern, or both the same protocol for a single protocol pattern.
#'
#' @section Usage:
#'
#'     Warning: this function is designed to be called in an isolated process
#'     with the two sockets. Once called, it will block with no ability to
#'     interrupt. Kill the process to terminate the device.
#'
#' @export
#'
device <- function(s1, s2) .Call(rnng_device, s1, s2)

#' Is Nano
#'
#' Is the object an object created by the nanonext package i.e. a nanoSocket,
#'     nanoContext, nanoStream, nanoListener, nanoDialer or nano Object.
#'
#' @param x an object.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @details Note: does not include Aio objects, for which there is a separate
#'     function \code{\link{is_aio}}.
#'
#' @examples
#' s <- socket()
#' is_nano(s)
#' n <- nano()
#' is_nano(n)
#'
#' close(s)
#' n$close()
#'
#' @export
#'
is_nano <- function(x) inherits(x, c("nano", "nanoObject"))

#' Is Aio
#'
#' Is the object an Aio (sendAio or recvAio).
#'
#' @param x an object.
#'
#' @return Logical value TRUE if 'x' is of class 'recvAio' or 'sendAio', FALSE
#'     otherwise.
#'
#' @examples
#' sock <- socket(listen = "inproc://isaio")
#' r <- recv_aio(sock)
#' s <- send_aio(sock, "test")
#'
#' is_aio(r)
#' is_aio(s)
#'
#' close(sock)
#'
#' @export
#'
is_aio <- function(x) inherits(x, c("recvAio", "sendAio"))

#' Is Nul Byte
#'
#' Is the object a nul byte.
#'
#' @param x an object.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @examples
#' is_nul_byte(as.raw(0L))
#' is_nul_byte(raw(length = 1L))
#' is_nul_byte(writeBin("", con = raw()))
#'
#' is_nul_byte(0L)
#' is_nul_byte(NULL)
#' is_nul_byte(NA)
#'
#' @export
#'
is_nul_byte <- function(x) identical(x, as.raw(0L))

#' Is Error Value
#'
#' Is the object an error value generated by NNG. All returned integer error
#'     codes are classed as 'errorValue' to be distinguishable from integer
#'     message values. Includes user-specified errors such as 'aio' timeouts.
#'
#' @param x an object.
#'
#' @return Logical value TRUE if 'x' is of class 'errorValue', FALSE otherwise.
#'
#' @examples
#' is_error_value(1L)
#'
#' @export
#'
is_error_value <- function(x) inherits(x, "errorValue")

#' nanonext Initialise
#'
#' Initialise global options. This function prints a deprecation message and no
#'     longer has any effect.
#'
#' @param warn Deprecated.
#'
#' @return Invisible NULL.
#'
#' @export
#'
nano_init <- function(warn) {

  message("nanonext 0.7.0 | 'nano_init()' is deprecated as 'errorValues' no longer generate warnings")

}

#' Translate HTTP Status Codes
#'
#' Provides an explanation for HTTP response status codes (in the range 100 to
#'     599). If the status code is not defined as per RFC 9110, 'Non-standard
#'     Response' is returned, which may be a custom code used by the server.
#'
#' @param x numeric HTTP status code to translate.
#'
#' @return A character vector.
#'
#' @examples
#' status_code(200)
#' status_code(404)
#'
#' @export
#'
status_code <- function(x) .Call(rnng_status_code, x)

