% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run.simulation.R
\name{run.simulation}
\alias{run.simulation}
\title{Run a direct N-body simulation}
\usage{
run.simulation(sim, measure.time = TRUE)
}
\arguments{
\item{sim}{structured list of simulation settings, which must contain the following sublists:\cr\cr

\code{ics} is the sublist of initial conditions. It must contain the items:\cr
\code{m} = N-vector with the masses of the N particles. Negative mass values are considered as positive masses belonging to a background field, which is not subject to any forces. Therefore particles with negative mass will have a normal effect on particles with positive masses, but they will not, themselves, be accelerated by any other particle.\cr
\code{x} = N-by-3 matrix specifying the initial position in Cartesian coordinates\cr
\code{v} = N-by-3 matrix specifying the initial velocities\cr\cr

\code{para} is an optional sublist of optional simulation parameters. It contains the items:\cr
\code{t.max} = final simulation time in simulation units (see details). If not given, a characteristic time is computed as \code{t.max = 2*pi*sqrt(R^3/GM)}, where \code{R} is the RMS radius and \code{M} is the total mass.\cr
\code{dt.max} = maximum time step. If not given, no maximum time step is imposed, meaning that the maximum time step is either equal to \code{dt.out} or the adaptive time step, whichever is smaller.\cr
\code{dt.min} = minimum time step used, unless a smaller time step is required to save an output or to land precisely on the final time \code{t.max}.
\code{dt.out} = output time step, i.e. time step between successive snapshots in the \code{output} sublist returned by \code{run.simulation}. If not given, \code{dt.max=t.max/100} is assumed.\cr
\code{eta} = accuracy parameter of adaptive time step. Smaller values lead to proportionally smaller adaptive time steps. Typical values range between 0.001 and 0.1. If not given, a default value of 0.01 is assumed. To use fixed time steps, set \code{eta=1e99} and set a time step \code{dt.max}.\cr
\code{integrator} = character string specifying the integrator to be used. Currently implemented integrators are 'euler' (1st order), 'leapfrog' (2nd order), 'yoshida' (4th order), 'yoshida6' (6th order). If not given, 'leapfrog' is the default integrator.\cr
\code{rsmooth} = optional smoothing radius. If not given, no smoothing is assumed.\cr
\code{afield} = a function(x,t) of positions \code{x} (N-by-3 matrix) and time \code{t} (scalar), specifying the external acceleration field. It must return an N-by-3 matrix. If not given, no external field is assumed.\cr
\code{G} = gravitational constant in simulation units (see details). If not given, the measured value in SI units is used.\cr
\code{box.size} = scalar>=0. If 0, open boundary conditions are adopted. If >0, the simulation is run in a cubic box of side length box.size with periodic boundary conditions. In this case, the cubic box is contained in the interval [0,box.size) in all three Cartesian coordinates, and all initial positions must be contained in this interval. For periodic boundary conditions, the force between any two particles is always calculated along their shortest separation, which may cross 0-3 boundaries. The exception is GADGET-4, which also evaluates the forces from the periodic repetitions.\cr
\code{include.bg} = logical argument. If FALSE (default), only foreground particles, i.e. particles with masses >=0, are contained in the output vectors \code{x} and \code{v}. If TRUE, all particles are included.

\code{code} is an optional sublist to force the use of an external simulation code (see details). It contains the items:\cr
\code{name} = character string specifying the name of the code, currently available options are "R" (default), "nbodyx" (a simple, but fast N-body simulator in Fortran) and "gadget4" (a powerful N-body+SPH simulator, not very adequate for small direct N-body simulations).\cr
\code{file} = character string specifying the path+filename of the external compiled simulation code.\cr
\code{interface} = optional character string specifying a temporary working path used as interface with external codes. If not given, the current working directory is used by default.\cr
\code{kind} = optional number of bytes per floating-point number used in nbodyx output files (has no bearing on computation accuracy)\cr
\code{gadget.np} = number of processors used with GADGET-4 (defaults to 1, which is normally best for small direct N-body runs)}

\item{measure.time}{logical flag that determines whether time computation time will be measured and displayed.}
}
\value{
The routine returns the structured list of the input argument, with one sublist \code{output} added. This sublist contains the items:
\item{t}{k-vector with the simulation times of the k snapshots.}
\item{x}{k-by-N-by-3 array giving the 3D coordinates of the N particles in k snapshots.}
\item{v}{k-by-N-by-3 array giving the 3D velocities of the N particles in k snapshots.}
\item{n.snapshots}{total number of snapshots.}
\item{n.iterations}{total number of iterations used to run the simulation.}
}
\description{
Run direct N-body simulations using an adaptive block timestep.
}
\details{
UNITS: The initial conditions (in the sublist \code{ics}) can be provided in any units. The units of mass, length and velocity then fix the other units.
For instance, [unit of time in seconds] = [unit of length in meters] / [unit of velocity in m/s]. E.g., if initial positions are given in units of 1AU=1.49598e11m and velocities in units of 1km/s, one unit of time is 1.49598e8s=4.74yrs.
Likewise, units of the gravitational constant \code{G} are given via [unit of G in m^3*kg^(-1)*s^(-2)] = [unit of length in meters] * [unit of velocity in m/s]^2 / [unit of mass in kg]. E.g., for length units of 1AU=1.49598e11m, velocity units of 1km/s=1e3m/s and mass units of 1Msun=1.98847e30kg, a unit of G is
7.523272e-14 m^3*kg^(-1)*s^(-2). In these units the true value of G is about 887.154.\cr\cr

NBODYX simulator:\cr
Can be downloaded from github via\cr
\code{git clone https://github.com/obreschkow/nbodyx}\cr
Details on installing, compiling and running the code are given in the README file.\cr
Note: To run very high-accuracy simulations, such as the Pythagorean three-body problem, you can use 128-bit floating-point numbers by compiling the code as\cr
\code{make kind=16}\cr\cr

GADGET-4 simulator:\cr
This his a very powerful N-body+SPH simulator used primarily for large astrophysical simulations. GADGET-4 is not particularly suitable for small direct N-body problems, but it can nonetheless be used for such simulations for the sake of comparison, at least if not too much accuracy is needed and if a massively increased computational overhead is acceptable.
Please refer to https://wwwmpa.mpa-garching.mpg.de/gadget4 for details on how to download and compile the code. In order to use GADGET-4 with this R-package, it must be compiled with the following compile-time options (in the file Config.sh):\cr
\code{NTYPES=2}\cr
\code{GADGET2_HEADER}\cr
\code{SELFGRAVITY}\cr
\code{ALLOW_DIRECT_SUMMATION}\cr
\code{HIERARCHICAL_GRAVITY}\cr
\code{DOUBLEPRECISION=1}\cr
\code{ENLARGE_DYNAMIC_RANGE_IN_TIME}\cr
If and only if periodic boundary conditions are used, you also need to add the option\cr
\code{PERIODIC}\cr
If you plan to often switch between runs with open and periodic boundaries, it may be advisable to compile two versions of GADGET-4, with and without this option. To do so, one needs to create two sub-directories with the respective Config.sh files and compile them via\cr
\code{make -j [number of cores] DIR=[path containing Config.sh with PERIODIC]}\cr
\code{make -j [number of cores] DIR=[path containing Config.sh without PERIODIC]}\cr
The runtime parameter file (param.txt) needed by GADGET-4 is written automatically when calling \code{run.simulation}. The gravitational softening length in GADGET-4 is computed as sim$para$rsmooth/2.8, which ensures that the particles behave like point masses at separations beyond sim$para$rsmooth. If rsmooth is not provided, it is computed as \code{stats::sd(apply(sim$ics$x,2,sd))*1e-5}. The accuracy parameter ErrTolIntAccuracy is set equal to sim$para$eta/sim$para$rsmooth*1e-3, which gives roughly comparable accuracy to in-built simulator for the Leapfrog integrator.
}
\examples{
sim = setup.halley()
sim = run.simulation(sim)
AU = 149597870700 # Astronomical unit in meters
plot(sim, units=AU, xlim=c(-20,60), ylim=c(-40,40), xlab='[AU]', ylab='[AU]')
cat(sprintf('This simulation was run with \%d iterations.\n',sim$output$n.iterations))

}
\author{
Danail Obreschkow
}
\keyword{N-body}
\keyword{simulation}
