% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rdiffnet.r
\name{rdiffnet}
\alias{rdiffnet}
\alias{rdiffnet_multiple}
\title{Random diffnet network}
\usage{
rdiffnet_multiple(R, statistic, ..., ncpus = 1L, cl = NULL)

rdiffnet(
  n,
  t,
  seed.nodes = "random",
  seed.p.adopt = 0.05,
  seed.graph = "scale-free",
  rgraph.args = list(),
  rewire = TRUE,
  rewire.args = list(),
  threshold.dist = runif(n),
  exposure.args = list(),
  name = "A diffusion network",
  behavior = "Random contagion",
  stop.no.diff = TRUE
)
}
\arguments{
\item{R}{Integer scalar. Number of simulations to be done.}

\item{statistic}{A Function to be applied to each simulated diffusion network.}

\item{...}{Further arguments to be passed to \code{rdiffnet}.}

\item{ncpus}{Integer scalar. Number of processors to be used (see details).}

\item{cl}{An object of class \code{\link[parallel:makeCluster]{c("SOCKcluster", "cluster")}}
(see details).}

\item{n}{Integer scalar. Number of vertices.}

\item{t}{Integer scalar. Time length.}

\item{seed.nodes}{Either a character scalar or a vector. Type of seed nodes (see details).}

\item{seed.p.adopt}{Numeric scalar. Proportion of early adopters.}

\item{seed.graph}{Baseline graph used for the simulation (see details).}

\item{rgraph.args}{List. Arguments to be passed to rgraph.}

\item{rewire}{Logical scalar. When TRUE, network slices are generated by rewiring
(see \code{\link{rewire_graph}}).}

\item{rewire.args}{List. Arguments to be passed to \code{\link{rewire_graph}}.}

\item{threshold.dist}{Either a function to be applied via \code{\link{sapply}},
a numeric scalar, or a vector/matrix with \eqn{n} elements. Sets the adoption
threshold for each node.}

\item{exposure.args}{List. Arguments to be passed to \code{\link{exposure}}.}

\item{name}{Character scalar. Passed to \code{\link{as_diffnet}}.}

\item{behavior}{Character scalar. Passed to \code{\link{as_diffnet}}.}

\item{stop.no.diff}{Logical scalar. When \code{TRUE}, the function will return
with error if there was no diffusion. Otherwise it throws a warning.}
}
\value{
A random \code{\link{diffnet}} class object.

\code{rdiffnet_multiple} returns either a vector or an array depending
on what \code{statistic} is (see \code{\link{sapply}} and
\code{\link[parallel:parSapply]{parSapply}}).
}
\description{
Simulates a diffusion network by creating a random dynamic network and
adoption threshold levels.
}
\details{
Instead of randomizing whether an individual adopts the innovation or not, this
toy model randomizes threshold levels, seed adopters and network structure, so
an individual adopts the innovation in time \eqn{T} iff his exposure is above or
equal to his threshold. The simulation is done in the following steps:

\enumerate{
 \item Using \code{seed.graph}, a baseline graph is created.
 \item Given the baseline graph, the set of initial adopters is defined
 using \code{seed.nodes}.
 \item Afterwards, if \code{rewire=TRUE} \eqn{t-1} slices of the network are created
 by iteratively rewiring the baseline graph.
 \item The \code{threshold.dist} function is applied to each node in the graph.
 \item Simulation starts at \eqn{t=2} assigning adopters in each time period
 accordingly to each vertex's threshold and exposure.
}

When \code{seed.nodes} is a character scalar it can be \code{"marginal"}, \code{"central"} or \code{"random"},
So each of these values sets the initial adopters using the vertices with lowest
degree, with highest degree or completely randomly. The number of early adoptes
is set as \code{seed.p.adopt * n}. Please note that when marginal nodes are
set as seed it may be the case that no diffusion process is attained as the
chosen set of first adopters can be isolated. Any other case will be considered
as an index (via \code{\link{[<-}} methods), hence the user can manually set the set of initial adopters, for example
if the user sets \code{seed.nodes=c(1, 4, 7)} then nodes 1, 4 and 7 will be
selected as initial adopters.

The argument \code{seed.graph} can be either a function that generates a graph
(Any class of accepted graph format (see \code{\link{netdiffuseR-graphs}})), a
graph itself or a character scalar in which the user sets the algorithm used to
generate the first network (network in t=1), this can be either "scale-free"
(Barabasi-Albert model using the \code{\link{rgraph_ba}} function, the default),
\code{"bernoulli"} (Erdos-Renyi model using the \code{\link{rgraph_er}} function),
or \code{"small-world"} (Watts-Strogatz model using the \code{\link{rgraph_ws}}
function). The list \code{rgraph.args} passes arguments to the chosen algorithm.

When \code{rewire=TRUE}, the networks that follow t=1 will be generated using the
\code{\link{rewire_graph}} function as \eqn{G(t) = R(G(t-1))}, where \eqn{R}
is the rewiring algorithm.

If a function, the argument \code{threshold.dist} sets the threshold for each vertex in the graph.
It is applied using \code{sapply} as follows

\preformatted{
sapply(1:n, threshold.dist)
}

By default sets the threshold to be random for each node in the graph.

If \code{seed.graph} is provided, no random graph is generated and the simulation
is applied using that graph instead.

\code{rewire.args} has the following default options:

\tabular{ll}{
  \code{p}          \tab \code{.1} \cr
  \code{undirected} \tab \code{getOption("diffnet.undirected", FALSE)} \cr
  \code{self}       \tab \code{getOption("diffnet.self", FALSE)}
}

\code{exposure.args} has the following default options:

\tabular{ll}{
  \code{outgoing} \tab \code{TRUE} \cr
  \code{valued} \tab \code{getOption("diffnet.valued", FALSE)} \cr
  \code{normalized} \tab \code{TRUE}
}

The function \code{rdiffnet_multiple} is a wrapper of \code{rdiffnet} wich allows
simulating multiple diffusion networks with the same parameters and apply
the same function to all of them. This function is designed to allow the user
to perform larger simulation studies in which the distribution of a particular
statistic is observed.

When \code{cl} is provided, then simulations are done via
\code{\link[parallel:parSapply]{parSapply}}. If \code{ncpus} is greater than
1, then the function creates a cluster via \code{\link[parallel:makeCluster]{makeCluster}}
which is stopped (removed) once the process is complete.
}
\examples{
# Asimple example -----------------------------------------------------------
set.seed(123)
z <- rdiffnet(100,10)
z
summary(z)

# A more complex example: Adopt if at least one neighbor has adopted --------
y <- rdiffnet(100, 10, threshold.dist=function(x) 1,
    exposure.args=list(valued=FALSE, normalized=FALSE))

# Re thinking the Adoption of Tetracycline ----------------------------------
newMI <- rdiffnet(seed.graph = medInnovationsDiffNet$graph,
 threshold.dist = threshold(medInnovationsDiffNet), rewire=FALSE)


# Simulation study comparing the diffusion with diff sets of seed nodes -----

# Random seed nodes
set.seed(1)
ans0 <- rdiffnet_multiple(R=50, statistic=function(x) sum(!is.na(x$toa)),
    n = 100, t = 4, seed.nodes = "random", stop.no.diff=FALSE)

# Central seed nodes
set.seed(1)
ans1 <- rdiffnet_multiple(R=50, statistic=function(x) sum(!is.na(x$toa)),
    n = 100, t = 4, seed.nodes = "central", stop.no.diff=FALSE)

boxplot(cbind(Random = ans0, Central = ans1), main="Number of adopters")
}
\seealso{
Other simulation functions: 
\code{\link{permute_graph}()},
\code{\link{rewire_graph}()},
\code{\link{rgraph_ba}()},
\code{\link{rgraph_er}()},
\code{\link{rgraph_ws}()},
\code{\link{ring_lattice}()}
}
\author{
George G. Vega Yon
}
\concept{simulation functions}
