% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/amsr.R
\docType{methods}
\name{[[,amsr-method}
\alias{[[,amsr-method}
\title{Extract Something From an amsr Object}
\usage{
\S4method{[[}{amsr}(x, i, j, ...)
}
\arguments{
\item{x}{An \code{amsr} object, i.e. one inheriting from \code{\link{amsr-class}}.}

\item{i}{Character string indicating the name of item to extract.}

\item{j}{Optional additional information on the \code{i} item.}

\item{...}{Optional additional information (ignored).}
}
\value{
In all cases, the returned value is a matrix with 
with \code{NA} values inserted at locations where
the raw data equal \code{as.raw(251:255)}, as explained
in \dQuote{Details}.
}
\description{
Extract something from the \code{metadata} or \code{data} slot of an
\code{\link{amsr-class}} object.

The \code{[[} method works for all \code{oce} objects, i.e.
objects inheriting from \code{\link{oce-class}}.  The purpose,
as with the related replacement method, \code{\link{[[<-}}, 
is to insulate users from the internal details of \code{oce}
objects, by looking for items within the various storage
slots of the object. Items not actually stored can also be 
extracted, including derived data, units of measurement,
and data-quality flags.

The method uses a two-step process to try to find the
requested information. First, a class-specific function 
is used to try to access the requested information (see
\dQuote{Details of the specialized ... method}). 
Second, if no match is found, a general function is used
(see \sQuote{Details of the general method}). If neither
method can locates the requested item, \code{NULL} is returned.
}
\details{
Partial matches for \code{i}
are permitted for \code{metadata}, and \code{j} is ignored for
\code{metadata}.

Data within the \code{data} slot may be found directly, e.g.
\code{j="SSTDay"} will yield sea-surface temperature in the daytime
satellite, and \code{j="SSTNight"} is used to access the nighttime data. In
addition, \code{j="SST"} yields an average of the night and day values
(using just one of these, if the other is missing). This scheme works for
all the data stored in \code{amsr} objects, namely:
\code{time}, \code{SST}, \code{LFwind}, \code{MFwind},
\code{vapor}, \code{cloud} and \code{rain}.  In each case, the default
is to calculate values in scientific units, unless \code{j="raw"}, in
which case the raw data are returned.

The \code{"raw"} mode can be useful
in decoding the various types of missing value that are used by \code{amsr}
data, namely \code{as.raw(255)} for land, \code{as.raw(254)} for
a missing observation, \code{as.raw(253)} for a bad observation,
\code{as.raw(252)} for sea ice, or \code{as.raw(251)} for missing SST
due to rain or missing water vapour due to heavy rain. Note that
something special has to be done for e.g. \code{d[["SST", "raw"]]}
because the idea is that this syntax (as opposed to specifying
\code{"SSTDay"}) is a request to try to find good
data by looking at both the Day and Night measurements. The scheme
employed is quite detailed. Denote by "A" the raw value of the desired field
in the daytime pass, and by "B" the corresponding value in the 
nighttime pass. If either A or B is 255, the code for land, then the
result will be 255. If A is 254 (i.e. there is no observation),
then B is returned, and the reverse holds also. Similarly, if either
A or B equals 253 (bad observation), then the other is returned.
The same is done for code 252 (ice) and code 251 (rain).
}
\section{Details of the general method}{


If the specialized method produces no matches, the following generalized
method is applied. As with the specialized method, the procedure hinges
first on the value of \code{i}.

First, a check is made as to whether \code{i} names one of the standard
\code{oce} slots, and returns the slot contents if so. Thus, 
\code{x[["metadata"]]} will retrieve the \code{metadata} slot, 
while \code{x[["data"]]} and \code{x[["processingLog"]]} return
those slots.

Next, if \code{i} is a string ending in the
\code{"Unit"}, then the characters preceding that string
are taken to be the name of an item in the data object, and a list
containing the unit is returned. This list consists of an item
named \code{unit}, which is an \code{\link{expression}}, and
an item named \code{scale}, which is a string describing the
measurement scale.  If the string ends in \code{" unit"}, e.g.
\code{x[["temperature unit"]]}, then just the expression is returned, 
and if it ends in \code{" scale"}, then just the scale is returned.

Next, if \code{i} is a string ending in \code{"Flag"}, then the corresponding
data-quality flag is returned (or \code{NULL} if there is no such flag). 
For example, \code{x[["salinityFlag"]]} returns a vector of salinity
flags if \code{x} is a ctd object.

If none of the preceding conditions are met, a check is done
to see if the \code{metadata} slot contains an item with
the provided name, and that is returned, if so. A direct match
is required for this condition.

Finally, the \code{data} slot is checked to see if it contains 
an item with the name indicated by \code{i}. In this case, a partial
match will work; this is accomplished by using \code{\link{pmatch}}.

If none of the above-listed conditions holds, then \code{NULL} is returned.
}
\examples{
\dontrun{
# Show a daytime SST image, along with an indication of whether
# the NA values are from rain.
library(oce)
earth <- read.amsr("f34_20160102v7.2.gz")
fclat <- subset(earth , 35 <= latitude & latitude <= 55)
fc <- subset(fclat , -70 <= longitude & longitude <= -30)
par(mfrow=c(2,1))
plot(fc, "SSTDay")
rainy <- fc[["SSTDay", "raw"]] == as.raw(0xfb)
lon <- fc[["longitude"]]
lat <- fc[["latitude"]]
asp <- 1 / cos(pi*mean(lat)/180)
imagep(lon, lat, rainy, asp=asp)
mtext("red: too rainy to sense SSTDay")
}
}
\author{
Dan Kelley
}
\seealso{
Other functions that extract parts of \code{oce} objects: \code{\link{[[,adp-method}},
  \code{\link{[[,adv-method}},
  \code{\link{[[,argo-method}},
  \code{\link{[[,bremen-method}},
  \code{\link{[[,cm-method}},
  \code{\link{[[,coastline-method}},
  \code{\link{[[,ctd-method}},
  \code{\link{[[,echosounder-method}},
  \code{\link{[[,g1sst-method}},
  \code{\link{[[,gps-method}},
  \code{\link{[[,ladp-method}},
  \code{\link{[[,lisst-method}},
  \code{\link{[[,lobo-method}},
  \code{\link{[[,met-method}}, \code{\link{[[,odf-method}},
  \code{\link{[[,rsk-method}},
  \code{\link{[[,sealevel-method}},
  \code{\link{[[,section-method}},
  \code{\link{[[,tidem-method}},
  \code{\link{[[,topo-method}},
  \code{\link{[[,windrose-method}},
  \code{\link{[[<-,adv-method}}
}

