/* vim: set expandtab shiftwidth=2 softtabstop=2 tw=70: */

#include <Rcpp.h>
using namespace Rcpp;

// Cross-reference work:
// 1. update ../src/registerDynamicSymbol.c with an item for this
// 2. main code should use the autogenerated wrapper in ../R/RcppExports.R

//#define USE_APPROX_EXP 1
#ifdef USE_APPROX_EXP
// Compute exp(-x) approximately, as efficiency measure.
// See Dan Kelley notebook [1997/1/25] for demonstration
// of factor of 3 speedup, with 1000 column data and a
// 10 by 10 grid, and demonstration that the
// error is < 0.1% in the final grid.
inline double exp_approx(double x)
{
  return 1.0 / (0.999448
      + x * (1.023820
        + x * (0.3613967
          + x * (0.4169646
            + x * (-0.1292509
              + x * 0.0499565)))));
}
#endif

static time_t start;

static double interpolate_barnes(double xx, double yy, double zz, /* interpolate to get zz value at (xx,yy) */
    int skip, /* value in (x,y,z) to skip, or -1 if no skipping */
    unsigned int nx, double *x, double *y, double *z, double *w, /* data num, locations, values, weights */
    double *z_last, /* last estimate of z at (x,y) */
    double xr, double yr, int debug) /* influence radii */
{
  double sum = 0.0, sum_w = 0.0;
  for (unsigned int k = 0; k < nx; k++) {
    // R trims NA (x,y values so no need to check here
    if ((int)k != skip) {
      double dx, dy, d, weight;
      dx = (xx - x[k]) / xr;
      dy = (yy - y[k]) / yr;
      d = dx*dx + dy*dy;
#ifdef USE_APPROX_EXP
      weight = w[k] * exp_approx(-d);
#else
      weight = w[k] * exp(-d);
#endif
      sum += weight * (z[k] - z_last[k]);
      sum_w += weight;
    }
  }
  return ((sum_w > 0.0) ? (zz + sum / sum_w) : NA_REAL);
}

// next is modelled on interpolate_barnes()
static double weight_barnes(double xx, double yy,
    int skip,
    unsigned int n, double *x, double *y, double *z, double *w,
    double xr, double yr)
{
  double sum_w, dx, dy, d, weight;
  sum_w = 0.0;
  for (unsigned int k = 0; k < n; k++) {
    if ((int)k != skip) {
      dx = (xx - x[k]) / xr;
      dy = (yy - y[k]) / yr;
      d = dx*dx + dy*dy;
#ifdef USE_APPROX_EXP
      weight = w[k] * exp_approx(-d);
#else
      weight = w[k] * exp(-d);
#endif
      sum_w += weight;
    }
  }
  return ((sum_w > 0.0) ? sum_w : NA_REAL);
}


// [[Rcpp::export]]
List do_interp_barnes(NumericVector x, NumericVector y, NumericVector z, NumericVector w, NumericVector xg, NumericVector yg, NumericVector xr, NumericVector yr, NumericVector gamma, NumericVector iterations)
{
  start = time(NULL);
  int nx = x.size();
  int nxg = xg.size();
  int nyg = yg.size();
  NumericMatrix zg(nxg, nyg), wg(nxg, nyg); // predictions on the grid
  NumericVector zd(nx); // predictions at the data

  //Rprintf("xg[0]=%f; size=%d\n", xg[0], xg.size());
  //Rprintf("yg[0]=%f; size=%d\n", yg[0], yg.size());

  double rgamma = gamma[0]; // gamma
  if (rgamma < 0.0)
    ::Rf_error("gamma=%f cannot be non-positive", rgamma);
  int niter = floor(0.5 + iterations[0]); // number of iterations
  if (niter < 0)
    ::Rf_error("cannot have a negative number of iterations.  Got %d ", niter);
  if (niter > 20)
    ::Rf_error("cannot have more than 20 iterations.  Got %d ", niter);
  if (xr[0] <= 0)
    ::Rf_error("cannot have xr<=0 but it is %f", xr[0]);
  if (yr[0] <= 0)
    ::Rf_error("cannot have yr<=0 but it is %f", yr[0]);
  double xr2 = xr[0]; // local radius, which will vary with iteration
  double yr2 = yr[0]; // local radius, which will vary with iteration

  /* previous values and working matrix */
  NumericVector z_last(nx); // .c had nx+100000 (??!!??)
  NumericMatrix zz(nxg, nyg);

  // initialize (Q: needed? I can't find docs on initialization)
  std::fill(zz.begin(), zz.end(), 0.0);
  std::fill(z_last.begin(), z_last.end(), 0.0);
  std::fill(zd.begin(), zd.end(), 0.0);


  for (int iter = 0; iter < niter; iter++) {
    //Rprintf("iter=%d xr2=%f yr2=%f\n", iter, xr2, yr2);
    /* update grid */
    for (int i = 0; i < nxg; i++) {
      for (int j = 0; j < nyg; j++) {
        zz(i, j) = interpolate_barnes(xg[i], yg[j], zz(i, j),
            -1, /* no skip */
            nx, &x[0], &y[0], &z[0], &w[0],
            &z_last(0),
            xr2, yr2, i==(nxg-1)&&j==(nyg-1));
        R_CheckUserInterrupt();
      }
    }
    /* interpolate grid back to data locations */
    for (int k = 0; k < nx; k++) {
      //Rprintf("  zd[%d] = %f (iter %d)\n", k, zd[k], iter);
      zd[k] = interpolate_barnes(x[k], y[k], z_last[k],
          -1, /* BUG: why not skip? */
          nx, &x[0], &y[0], &z[0], &w[0],
          &z_last(0),
          xr2, yr2, 0);
      //Rprintf("  -> zd[%d] = %f (iter %d)\n", k, zd[k], iter);
    }
    R_CheckUserInterrupt();
    for (int k = 0; k < nx; k++)
      z_last[k] = zd[k];
    if (rgamma > 0.0) {
      // refine search range for next iteration
      xr2 *= sqrt(rgamma);
      yr2 *= sqrt(rgamma);
    }
  }

  // copy matrix to return value
  for (int i = 0; i < nxg; i++)
    for (int j = 0; j < nyg; j++)
      zg(i, j) = zz(i, j);

  // weights at final region-of-influence radii
  for (int i = 0; i < nxg; i++) {
    for (int j = 0; j < nyg; j++) {
      wg(i, j) = weight_barnes(xg[i], yg[j],
          -1, /* no skip */
          nx, &x[0], &y[0], &z[0], &w[0],
          xr2, yr2);
    }
    R_CheckUserInterrupt();
  }
  return(List::create(Named("zg")=zg, Named("wg")=wg, Named("zd")=zd));
}
