% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctd.R
\name{setFlags,ctd-method}
\alias{setFlags,ctd-method}
\title{Set data-quality flags within a ctd object}
\usage{
\S4method{setFlags}{ctd}(
  object,
  name = NULL,
  i = NULL,
  value = NULL,
  debug = getOption("oceDebug")
)
}
\arguments{
\item{object}{An oce object.}

\item{name}{Character string indicating the name of the variable to be flagged. If
this variable is not contained in the object's \code{data} slot, an error is reported.}

\item{i}{Indication of where to insert the flags; see \dQuote{Description} for
general rules and \dQuote{Details} for rules for \linkS4class{ctd}
objects.}

\item{value}{The value to be inserted in the flag.}

\item{debug}{Integer set to 0 for quiet action or to 1 for some debugging.}
}
\value{
An object with flags set as indicated.
}
\description{
This function changes specified entries in the data-quality
flags of a ctd object, which are stored within
a list named \code{flags} that resides in the \code{metadata}
slot. If the object already has a flag set up for \code{name},
then only the specified entries are altered. If not, the flag
entry is first created and its entries set to \code{default},
after which the entries specified by \code{i}
are changed to \code{value}.

The specification is made with \code{i}, the form of which
is determined by the data item in question. Generally,
the rules are as follows:
\enumerate{
\item If the data item is a vector, then \code{i} must be (a)
an integer vector specifying indices to be set to \code{value},
(b) a logical vector of length matching the data item, with
\code{TRUE} meaning to set the flag to \code{value}, or (c)
a function that takes an \code{oce} object as its single
argument, and returns a vector in either of the forms
just described.
\item If the data item is an array, then \code{i} must be
(a) a data frame of integers whose rows specify spots to change
(where the number of columns matches the number of dimensions
of the data item), (b) a logical array that has dimension equal to
that of the data item, or (c) a function that takes an \code{oce}
object as its single input and returns such a data frame or array.
}

See \dQuote{Details} for the particular case of \linkS4class{ctd} objects.
}
\details{
Since all the entries in the \code{data} slot of ctd objects are vectors, \code{i} must be a vector (either logical as in Example 1 or integer as in Example 2), or a function taking a \code{ctd} object and returning such a vector (see \dQuote{Indexing rules}).
}
\examples{
library(oce)
# Example 1: Range-check salinity
data(ctdRaw)
# Salinity and temperature range checks
qc <- ctdRaw
# Initialize flags to 2, meaning good data in the default
# scheme for handleFlags(ctd).
qc <- initializeFlags(qc, "salinity", 2)
qc <- initializeFlags(qc, "temperature", 2)
# Flag bad salinities as 4
oddS <- with(qc[["data"]], salinity < 25 | 40 < salinity)
qc <- setFlags(qc, name="salinity", i=oddS, value=4)
# Flag bad temperatures as 4
oddT <- with(qc[["data"]], temperature < -2 | 40 < temperature)
qc <- setFlags(qc, name="temperature", i=oddT, value=4)
# Compare results in TS space
par(mfrow=c(2, 1))
plotTS(ctdRaw)
plotTS(handleFlags(qc, flags=c(1, 3:9)))

# Example 2: Interactive flag assignment based on TS plot, using
# WHP scheme to define 'acceptable' and 'bad' codes
\dontrun{
options(eos="gsw")
data(ctd)
qc <- ctd
qc <- initializeFlagScheme(qc, "WHP CTD")
qc <- initializeFlags(qc, "salinity", 2)
Sspan <- diff(range(qc[["SA"]]))
Tspan <- diff(range(qc[["CT"]]))
n <- length(qc[["SA"]])
par(mfrow=c(1, 1))
plotTS(qc, type="o")
message("Click on bad points; quit by clicking to right of plot")
for (i in seq_len(n)) {
    xy <- locator(1)
    if (xy$x > par("usr")[2])
        break
    i <- which.min(abs(qc[["SA"]] - xy$x)/Sspan + abs(qc[["CT"]] - xy$y)/Tspan)
    qc <- setFlags(qc, "salinity", i=i, value=4)
    qc <- handleFlags(qc, flags=list(salinity=4))
    plotTS(qc, type="o")
}
}

}
\seealso{
Other functions relating to data-quality flags: 
\code{\link{defaultFlags}()},
\code{\link{handleFlags,adp-method}},
\code{\link{handleFlags,argo-method}},
\code{\link{handleFlags,ctd-method}},
\code{\link{handleFlags,oce-method}},
\code{\link{handleFlags,section-method}},
\code{\link{handleFlags}()},
\code{\link{initializeFlagScheme,ctd-method}},
\code{\link{initializeFlagScheme,oce-method}},
\code{\link{initializeFlagScheme,section-method}},
\code{\link{initializeFlagSchemeInternal}()},
\code{\link{initializeFlagScheme}()},
\code{\link{initializeFlags,adp-method}},
\code{\link{initializeFlags,oce-method}},
\code{\link{initializeFlagsInternal}()},
\code{\link{initializeFlags}()},
\code{\link{setFlags,adp-method}},
\code{\link{setFlags,oce-method}},
\code{\link{setFlags}()}

Other things related to ctd data: 
\code{\link{CTD_BCD2014666_008_1_DN.ODF.gz}},
\code{\link{[[,ctd-method}},
\code{\link{[[<-,ctd-method}},
\code{\link{as.ctd}()},
\code{\link{cnvName2oceName}()},
\code{\link{ctd-class}},
\code{\link{ctd.cnv}},
\code{\link{ctdDecimate}()},
\code{\link{ctdFindProfiles}()},
\code{\link{ctdRaw}},
\code{\link{ctdRepair}()},
\code{\link{ctdTrim}()},
\code{\link{ctd}},
\code{\link{d200321-001.ctd}},
\code{\link{d201211_0011.cnv}},
\code{\link{handleFlags,ctd-method}},
\code{\link{initialize,ctd-method}},
\code{\link{initializeFlagScheme,ctd-method}},
\code{\link{oceNames2whpNames}()},
\code{\link{oceUnits2whpUnits}()},
\code{\link{plot,ctd-method}},
\code{\link{plotProfile}()},
\code{\link{plotScan}()},
\code{\link{plotTS}()},
\code{\link{read.ctd.itp}()},
\code{\link{read.ctd.odf}()},
\code{\link{read.ctd.odv}()},
\code{\link{read.ctd.sbe}()},
\code{\link{read.ctd.woce.other}()},
\code{\link{read.ctd.woce}()},
\code{\link{read.ctd}()},
\code{\link{subset,ctd-method}},
\code{\link{summary,ctd-method}},
\code{\link{woceNames2oceNames}()},
\code{\link{woceUnit2oceUnit}()},
\code{\link{write.ctd}()}
}
\concept{functions relating to data-quality flags}
\concept{things related to ctd data}
