% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sw.R
\name{swTFreeze}
\alias{swTFreeze}
\title{Seawater freezing temperature}
\usage{
swTFreeze(
  salinity,
  pressure = NULL,
  longitude = NULL,
  latitude = NULL,
  saturation_fraction = 1,
  eos = getOption("oceEOS", default = "gsw")
)
}
\arguments{
\item{salinity}{Either practical salinity (PSU) or a \code{ctd} object from which
practical salinity and pressure (plus in the \code{eos="gsw"} case,
longitude and latitude) are inferred.}

\item{pressure}{Seawater pressure (dbar).}

\item{longitude}{Longitude of observation (only used if \code{eos="gsw"};
see \sQuote{Details}).}

\item{latitude}{Latitude of observation (only used if \code{eos="gsw"}; see
\sQuote{Details}).}

\item{saturation_fraction}{The saturation fraction of dissolved air in seawater,
ignored if \code{eos="unesco"}).}

\item{eos}{The equation of state, either \code{"unesco"} (Fofonoff and Millard,
1983; Gill 1982) or \code{"gsw"} (IOC, SCOR and IAPSO 2010; McDougall and Barker
2011).}
}
\value{
Temperature (degC), defined on the ITS-90 scale.
}
\description{
Compute in-situ freezing temperature of seawater, using either the UNESCO
formulation (computed as in Section 5 of Fofonoff and Millard, 1983) or the
GSW formulation (computed by using \code{\link[gsw:gsw_SA_from_SP]{gsw::gsw_SA_from_SP()}} to get Absolute
Salinity, and then \code{\link[gsw:gsw_t_freezing]{gsw::gsw_t_freezing()}} to get the freezing temperature).
}
\details{
If the first argument is an \code{oce} object, and if the \code{pressure} argument is
\code{NULL}, then the pressure is sought within the first argument. In the case of
\code{eos="gsw"}, then a similar procedure also applies to the \code{longitude} and
\code{latitude} arguments.
}
\examples{
# 1. Test for a check-value given in reference 1. This value, -2.588567 degC,
# is in the 1968 temperature scale (IPTS-68), but swTFreeze reports
# in the newer ITS-90 scale, so we must convert before checking.
Tcheck <- -2.588567 # IPTS-68
T <- swTFreeze(salinity=40, pressure=500, eos="unesco")
stopifnot(abs(Tcheck - T68fromT90(T)) < 1e-6)

# 2. Compare unesco and gsw formulations.
data(ctd)
p <- ctd[["pressure"]]
par(mfrow=c(1, 2), mar=c(3, 3, 1, 2), mgp=c(2, 0.7, 0))
plot(swTFreeze(ctd, eos="unesco"),
     p, xlab="unesco", ylim=rev(range(p)))
plot(swTFreeze(ctd, eos="unesco") - swTFreeze(ctd, eos="gsw"),
     p, xlab="unesco-gsw", ylim=rev(range(p)))

}
\references{
Fofonoff, N. P., and R. C. Millard. Algorithms for Computation of
Fundamental Properties of Seawater. UNESCO Technical Papers in Marine
Research. SCOR working group on Evaluation of CTD data; UNESCO/ICES/SCOR/IAPSO
Joint Panel on Oceanographic Tables and Standards, 1983.

Gill, A E. Atmosphere-Ocean Dynamics. New York, NY, USA: Academic Press,
1982.

IOC, SCOR, and IAPSO (2010). The international thermodynamic equation of
seawater-2010: Calculation and use of thermodynamic properties.  Technical
Report 56, Intergovernmental Oceanographic Commission, Manuals and Guide, 2010.

McDougall, Trevor J., and Paul M. Barker. Getting Started with TEOS-10 and
the Gibbs Seawater (GSW) Oceanographic Toolbox. SCOR/IAPSO WG127, 2011.
}
\seealso{
Other functions that calculate seawater properties: 
\code{\link{T68fromT90}()},
\code{\link{T90fromT48}()},
\code{\link{T90fromT68}()},
\code{\link{computableWaterProperties}()},
\code{\link{locationForGsw}()},
\code{\link{swAbsoluteSalinity}()},
\code{\link{swAlphaOverBeta}()},
\code{\link{swAlpha}()},
\code{\link{swBeta}()},
\code{\link{swCSTp}()},
\code{\link{swConservativeTemperature}()},
\code{\link{swDepth}()},
\code{\link{swDynamicHeight}()},
\code{\link{swLapseRate}()},
\code{\link{swN2}()},
\code{\link{swPressure}()},
\code{\link{swRho}()},
\code{\link{swRrho}()},
\code{\link{swSCTp}()},
\code{\link{swSR}()},
\code{\link{swSTrho}()},
\code{\link{swSigma0}()},
\code{\link{swSigma1}()},
\code{\link{swSigma2}()},
\code{\link{swSigma3}()},
\code{\link{swSigma4}()},
\code{\link{swSigmaTheta}()},
\code{\link{swSigmaT}()},
\code{\link{swSigma}()},
\code{\link{swSoundAbsorption}()},
\code{\link{swSoundSpeed}()},
\code{\link{swSpecificHeat}()},
\code{\link{swSpice}()},
\code{\link{swSstar}()},
\code{\link{swTSrho}()},
\code{\link{swThermalConductivity}()},
\code{\link{swTheta}()},
\code{\link{swViscosity}()},
\code{\link{swZ}()}
}
\author{
Dan Kelley
}
\concept{functions that calculate seawater properties}
