% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/f_hyperEstimation.R
\name{autoHyper}
\alias{autoHyper}
\title{Semi-automated hyperparameter estimation}
\usage{
autoHyper(data, theta_init, squashed = TRUE, zeroes = FALSE,
  N_star = 1, tol = c(0.05, 0.05, 0.2, 0.2, 0.025), min_conv = 1,
  param_limit = 100, max_pts = 20000, conf_ints = FALSE,
  conf_level = c("95", "80", "90", "99"))
}
\arguments{
\item{data}{A data frame from \code{\link{processRaw}} containing columns
named \emph{N}, \emph{E}, and (if squashed) \emph{weight}.}

\item{theta_init}{A data frame of initial hyperparameter guesses with
columns ordered as:
\eqn{\alpha_1, \beta_1, \alpha_2, \beta_2, P}.}

\item{squashed}{A scalar logical (\code{TRUE} or \code{FALSE}) indicating
whether or not data squashing was used.}

\item{zeroes}{A scalar logical specifying if zero counts are included.}

\item{N_star}{A positive scalar whole number value for the minimum count
size to be used for hyperparameter estimation. If zeroes are used, set
\code{N_star} to \code{NULL}.}

\item{tol}{A numeric vector of tolerances for determining how close the
chosen estimate must be to at least \code{min_conv} convergent solutions.
Order is \eqn{\alpha_1}, \eqn{\beta_1}, \eqn{\alpha_2}, \eqn{\beta_2},
\eqn{P}.}

\item{min_conv}{A scalar positive whole number for defining the minimum
number of convergent solutions that must be close to the convergent
solution with the smallest negative log-likelihood. Must be at least one
and at most one less than the number of rows in \code{theta_init}.}

\item{param_limit}{A scalar numeric value for the largest acceptable value
for the \eqn{\alpha} and \eqn{\beta} estimates. Used to help protect
against unreasonable/erroneous estimates.}

\item{max_pts}{A scalar whole number for the largest number of data points
allowed. Used to help prevent extremely long run times.}

\item{conf_ints}{A scalar logical indicating if confidence intervals and
standard errors should be returned.}

\item{conf_level}{A scalar string for the confidence level used if confidence
intervals are requested.}
}
\value{
A list containing the following elements:
  \itemize{
    \item{\emph{method}: }{A scalar character string for the method used to
      find the hyperparameter estimate (possibilities are
      \dQuote{\code{nlminb}}, \dQuote{\code{nlm}}, and
      \dQuote{\code{bfgs}}).}
    \item{\emph{estimates}: }{A named numeric vector of length 5 for the
      hyperparameter estimate corresponding to the smallest log-likelihood.}
    \item{\emph{conf_int}: }{A data frame including the standard errors and
      confidence limits. Only included if \code{conf_ints = TRUE}.}
    \item{\emph{num_close}: }{A scalar integer for the number of other
      convergent solutions that were close (within tolerance) to the chosen
      estimate.}
    \item{\emph{theta_hats}: }{A data frame for the estimates corresponding
      to the initial starting points defined by \code{theta_init}. See
      \code{\link{exploreHypers}}}.
  }
}
\description{
\code{autoHyper} finds a single hyperparameter estimate using an algorithm
that evaluates results from multiple starting points (see
\code{\link{exploreHypers}}). The algorithm verifies that the optimization
converges within the bounds of the parameter space and that the chosen
estimate (smallest negative log-likelihood) is similar to at least
one (see \code{min_conv} argument) of the other convergent solutions.
}
\details{
The algorithm first attempts to find a consistently convergent
  solution using \code{\link[stats]{nlminb}}. If it fails, it will next try
  \code{\link[stats]{nlm}}. If it still fails, it will try
  \code{\link[stats]{optim}} (\emph{method = "BFGS"}). If all three
  approaches fail, the function returns an error message.

Since this function runs multiple optimization procedures, it is
  best to start with 5 or less initial starting points (rows in
  \code{theta_init}). If the function runs in a reasonable amount of time,
  this number can be increased.

This function should not be used with very large data sets since
  each optimization call will take a long time. \code{\link{squashData}} can
  be used first to reduce the size of the data.

It is recommended to use \code{N_star = 1} when practical. Data
  squashing (see \code{\link{squashData}}) can be used to further reduce the
  number of data points.

Asymptotic normal confidence intervals, if requested, use standard
  errors calculated from the observed Fisher information matrix as discussed
  in DuMouchel (1999).
}
\examples{
#Start with 2 or more guesses
theta_init <- data.frame(
  alpha1 = c(0.2, 0.1),
  beta1  = c(0.1, 0.1),
  alpha2 = c(2,   10),
  beta2  = c(4,   10),
  p      = c(1/3, 0.2)
)
data(caers)
proc <- processRaw(caers)
squashed <- squashData(proc, bin_size = 100, keep_pts = 100)
squashed <- squashData(squashed, count = 2, bin_size = 10, keep_pts = 20)
suppressWarnings(
  autoHyper(squashed, theta_init = theta_init)
)

}
\references{
DuMouchel W (1999). "Bayesian Data Mining in Large Frequency
  Tables, With an Application to the FDA Spontaneous Reporting System."
  \emph{The American Statistician}, 53(3), 177-190.
}
\seealso{
\code{\link[stats]{nlminb}}, \code{\link[stats]{nlm}}, and
  \code{\link[stats]{optim}} for optimization details

\code{\link{squashData}} for data preparation

Other hyperparameter estimation functions: \code{\link{exploreHypers}},
  \code{\link{hyperEM}}
}
\concept{hyperparameter estimation functions}
\keyword{openEBGM}
