% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_attributes_sites_approx.R
\name{calc_attributes_sites_approx}
\alias{calc_attributes_sites_approx}
\title{Calculate attributes of the sites.}
\usage{
calc_attributes_sites_approx(
  sites_map = "sites",
  input_attr_name = NULL,
  output_attr_name = NULL,
  stat = NULL,
  round_dig = 2,
  calc_basin_area = TRUE
)
}
\arguments{
\item{sites_map}{character; name of the sites (observation or prediction) 
attributes shall be calculated for. "sites" (default) refers to the observation sites.}

\item{input_attr_name}{character vector; input column name(s) in the
attribute table of the vector map "edges".}

\item{output_attr_name}{character vector (optional); output column name(s)
appended to the site attribute data table. If not provided it is set to
\code{input_attr_name}. Attribute names must not be longer than 10
characters.}

\item{stat}{name or character vector giving the statistics to be calculated.
See details below.}

\item{round_dig}{integer; number of digits to round results to.}

\item{calc_basin_area}{boolean; shall the catchment area be calculated? (Useful
to set to FALSE if the function has been called before.)}
}
\value{
Nothing. The function appends new columns to the \code{sites_map}
  attribute table
\itemize{
 \item{'H2OAreaA':} {Total watershed area of the watershed upstream of each site.}
 \item{attr_name:} {Additional optional attributes calculated based on \code{input_attr_name}.}
}
}
\description{
For each site (observations or predictions) attributes (potential predictor variables)
are derived based on the values calculated for the edge the site lies on.
This function calculates approximate values for site catchments as described
in Peterson & Ver Hoef, 2014: STARS: An ArcGIS Toolset Used to Calculate the
Spatial Information Needed to Fit Spatial Statistical Models to Stream
Network Data. J. Stat. Softw., 56 (2).
}
\details{
The approximate total catchment area (H2OAreaA) is always calculated
  if \code{calc_basin_area} is TRUE. If \code{stat} is one of 
  "min", "max", "mean" or "percent" the function assigns the value of the edge the site lies on. 
  Otherwise, the value is calculated as the sum of all edges upstream of the previous
  confluence and the proportional value of the edge the site lies on (based on
  the distance ratio 'ratio'); this is useful e.g. for counts of dams or waste water
  treatment plants or total catchment area.
  
  \code{input_attr_name} must give the column names of the edges attribute table
  for that the statistics should be calculated.
}
\note{
\code{\link{import_data}}, \code{\link{derive_streams}},
  \code{\link{calc_edges}}, \code{\link{calc_sites}} or
  \code{\link{calc_prediction_sites}} and \code{\link{calc_attributes_edges}}
  must be run before.
}
\examples{
\donttest{
# Initiate GRASS session
if(.Platform$OS.type == "windows"){
  gisbase = "c:/Program Files/GRASS GIS 7.6"
  } else {
  gisbase = "/usr/lib/grass74/"
  }
initGRASS(gisBase = gisbase,
    home = tempdir(),
    override = TRUE)

# Load files into GRASS
dem_path <- system.file("extdata", "nc", "elev_ned_30m.tif", package = "openSTARS")
sites_path <- system.file("extdata", "nc", "sites_nc.shp", package = "openSTARS")
pred_path <- system.file("extdata", "nc", "geology.shp", package = "openSTARS")
setup_grass_environment(dem = dem_path)
import_data(dem = dem_path, sites = sites_path,
 predictor_vector = pred_path)
gmeta()

# Derive streams from DEM
derive_streams(burn = 0, accum_threshold = 700, condition = TRUE, clean = TRUE)

# Check and correct complex confluences (there are no complex confluences in this
# example date set; set accum_threshold in derive_streams to a smaller value
# to create complex confluences)
cj <- check_compl_confluences()
if(cj){
  correct_compl_confluences()
}

# Prepare edges
calc_edges()

# Derive slope from the DEM as an example raster map to calculate attributes from
execGRASS("r.slope.aspect", flags = c("overwrite","quiet"),
parameters = list(
  elevation = "dem",
    slope = "slope"
    ))
calc_attributes_edges(input_raster = "slope", stat_rast = "max", attr_name_rast = "maxSlo",
                     input_vector = "geology", stat_vect = "percent", attr_name_vect = "GEO_NAME")

calc_sites() 
 
# approximate potential predictor variables for each site based on edge values
calc_attributes_sites_approx(
input_attr_name = c('maxSlo', 'CZamp', 'CZbgp', 'CZfgp', 'CZgp', 'CZigp', 'CZlgp', 'CZvep', 'Kmp'), 
  stat = c("max", rep("percent", 8)))

# plot share of a certain geology in the sampling point's catchment as
# point size
library(sp)
edges <- readVECT('edges', ignore.stderr = TRUE)
sites <- readVECT('sites', ignore.stderr = TRUE)
geo <- readVECT("geology", ignore.stderr = TRUE)
plot(geo, col = adjustcolor(1:8, alpha.f = 0.5)[as.factor(geo$GEO_NAME)])
plot(edges, col = "blue", add = TRUE)
plot(sites, col = 1, add = TRUE, pch = 19, cex = (sites$CZbgp + 0.15) * 2)
legend("left", col = adjustcolor(1:8, alpha.f = 0.5), bty = "n",
legend = unique(geo$GEO_NAME), pch = 15, title = "geology")
legend("right", col = 1, pch = 19, legend = seq(0, 1, 0.2), bty = "n",
title = "share CZbg\nin catchment", pt.cex =  (seq(0, 1, 0.2) + 0.15) * 2)
}
}
\author{
Mira Kattwinkel, \email{mira.kattwinkel@gmx.net}
}
