% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ordinalNetTune.R
\name{ordinalNetTune}
\alias{ordinalNetTune}
\title{Uses K-fold cross validation to obtain out-of-sample log-likelihood and
misclassification rates for a sequence of lambda values.}
\usage{
ordinalNetTune(x, y, lambdaVals = NULL, folds = NULL, nFolds = 5,
  printProgress = TRUE, warn = TRUE, ...)
}
\arguments{
\item{x}{Covariate matrix.}

\item{y}{Response variable. Can be a factor, ordered factor, or a matrix
where each row is a multinomial vector of counts. A weighted fit can be obtained
using the matrix option, since the row sums are essentially observation weights.
Non-integer matrix entries are allowed.}

\item{lambdaVals}{An optional user-specified lambda sequence (vector). If \code{NULL},
a sequence will be generated using the model fit to the full training data.
This default sequence is based on \code{nLambda} and \code{lambdaMinRatio},
which can be passed as additional arguments (otherwise \code{ordinalNet} default
values are used). The maximum lambda is the smallest value that sets all penalized
coefficients to zero, and the minimum lambda is the maximum value multiplied
by the factor \code{lambdaMinRatio}.}

\item{folds}{An optional list, where each element is a vector of row indices
corresponding to a different cross validation fold. Indices correspond to rows
of the \code{x} matrix. Each index number should be used in exactly one fold.
If \code{NULL}, the data will be randomly divided into equal-sized partitions.
It is recommended to use \code{set.seed} before calling this function to make
results reproducible.}

\item{nFolds}{Numer of cross validation folds. Only used if \code{folds=NULL}.}

\item{printProgress}{Logical. If \code{TRUE} the fitting progress is printed
to the terminal.}

\item{warn}{Logical. If \code{TRUE}, the following warning message is displayed
when fitting a cumulative probability model with \code{nonparallelTerms=TRUE}
(i.e. nonparallel or semi-parallel model).
"Warning message: For out-of-sample data, the cumulative probability model
with nonparallelTerms=TRUE may predict cumulative probabilities that are not
monotone increasing."
The warning is displayed by default, but the user may wish to disable it.}

\item{...}{Other arguments (besides \code{x}, \code{y}, \code{lambdaVals}, and \code{warn})
passed to \code{ordinalNet}.}
}
\value{
A list containing the following:
\describe{
  \item{loglik}{Matrix of out of sample log-likelihood values. Each row corresponds
  to a different lambda value, and each column corresponds to a different fold.}
  \item{misclass}{Matrix of out of sample misclassificaton rates. Each row corresponds
  to a different lambda value, and each column corresponds to a different fold.}
  \item{lambdaVals}{The sequence of lambda values used for all cross validation folds.}
  \item{folds}{A list containing the index numbers of each fold.}
  \item{fit}{An object of class "\code{ordinalNetFit}", resulting from fitting
  \code{ordinalNet} to the entire dataset.}
}
}
\description{
The data is divided into K folds. \code{ordinalNet} is fit \eqn{K} times (\code{K=nFolds}),
each time leaving out one fold as a test set. The same sequence of lambda values is used
each time. The out-of-sample log-likelihood and misclassification rate
are obtained for each lambda value from the held out test set.
It is up to the user to determine how to tune the model using this information.
}
\details{
\itemize{
  \item The fold partition splits can be passed by the user via the \code{folds}
  argument. By default, the data are randomly divided into equally-sized partitions.
  The \code{set.seed} function should be called prior to \code{ordinalNetCV} for reproducibility.
  \item A sequence of lambda values can be passed by the user via the
  \code{lambdaVals} argument. By default, the sequence is generated by first
  fitting the model to the full data set (this sequence is determined by the
  \code{nLambda} and \code{lambdaMinRatio} arguments of \code{ordinalNet}).
  \item The \code{standardize} argument of \code{ordinalNet} can be modified through
  the additional arguments (...). If \code{standardize=TRUE}, then the data are scaled
  within each cross validation fold. This is done because scaling is part of
  the statistical procedure and should be repeated each time the procedure is applied.
}
}
\examples{
\dontrun{
# Simulate x as independent standard normal
# Simulate y|x from a parallel cumulative logit (proportional odds) model
set.seed(1)
n <- 50
intercepts <- c(-1, 1)
beta <- c(1, 1, 0, 0, 0)
ncat <- length(intercepts) + 1  # number of response categories
p <- length(beta)  # number of covariates
x <- matrix(rnorm(n*p), ncol=p)  # n x p covariate matrix
eta <- c(x \%*\% beta) + matrix(intercepts, nrow=n, ncol=ncat-1, byrow=TRUE)
invlogit <- function(x) 1 / (1+exp(-x))
cumprob <- t(apply(eta, 1, invlogit))
prob <- cbind(cumprob, 1) - cbind(0, cumprob)
yint <- apply(prob, 1, function(p) sample(1:ncat, size=1, prob=p))
y <- as.factor(yint)

# Fit parallel cumulative logit model; select lambda by cross validation
tunefit <- ordinalNetTune(x, y)
bestLambdaIndex <- which.max(rowMeans(tunefit$loglik))
coef(tunefit$fit, whichLambda=bestLambdaIndex, matrix=TRUE)
predict(tunefit$fit, whichLambda=bestLambdaIndex)
}

}
