% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/osm_cont_inhibit.R
\name{osm_contin_inhibit_sample}
\alias{osm_contin_inhibit_sample}
\title{OSM Continuous Inhibitory sampling.}
\usage{
osm_contin_inhibit_sample(
  bounding_geom = NULL,
  boundary = 0,
  buff_dist = 0,
  buff_epsg = 4326,
  sample_size,
  plotit = TRUE,
  plotit_leaflet = TRUE,
  delta,
  delta.fix = FALSE,
  k = 0,
  rho = NULL,
  ntries = 10000
)
}
\arguments{
\item{bounding_geom}{a \code{sf} or \code{sp} object (with \eqn{N \geq
 \code{size}}) where each line corresponds to one spatial location. It
should contain values of 2D coordinates, data and, optionally, covariate(s)
value(s) at the locations. This argument must be provided when sampling
from a \code{'discrete'} set of points, see \code{'type'} below for
details.}

\item{boundary}{categorical variable to determine whether the exact boundary
provided (\code{boundary = 0}), the bounding box (\code{boundary = 1}) or a
buffer around the boundary (\code{boundary = 2}) is used for sampling.
Default is \code{boundary = 0}.}

\item{buff_dist}{if \code{boundary = 2} then this value determines the size
of the buffer by distance. The default is \code{buff_dist is NULL}).}

\item{buff_epsg}{if \code{boundary = 2} then this value determines the local
geographic grid reference so that the buffer can be calculated in meters.
The default is \code{buff_epsg = 4326} which will use decimal degrees
instead of meters. As an example, 27700 relates to the British National
Grid.}

\item{sample_size}{a non-negative integer giving the total number of
locations to be sampled.}

\item{plotit}{'logical' specifying if graphical output is required. Default
is \code{plotit = TRUE}.}

\item{plotit_leaflet}{'logical' specifying if leaflet (html) graphical output
is required. This is prioritised over plotit if both are selected. Default
is \code{plotit_leaflet = TRUE}.}

\item{delta}{minimum permissible distance between any two locations in
preliminary sample. This can be allowed to vary with the number of
\code{'close pairs'} if a \bold{simple inhibitory} design is compared to
one of the \bold{inhibitory plus close pairs} design.}

\item{delta.fix}{'logical' specifies whether \code{delta} is fixed or allowed
to vary with number of close pairs \eqn{k}. Default is \code{delta.fix =
 FALSE}.}

\item{k}{number of locations in preliminary sample to be replaced by near
neighbours of other preliminary sample locations to form \code{close pairs}
(integer between 0 and \code{size/2}). A \bold{simple inhibitory} deisgn is
generated when \eqn{k = 0}.}

\item{rho}{maximum distance between the two locations in a
\code{'close-pair'}.}

\item{ntries}{number of rejected proposals after which the algorithm will
terminate.}
}
\value{
a list with the following four components:

\code{size:} the total number of sampled locations.

\code{delta:} the value of \eqn{\delta} after taking into account the
number of close pairs \eqn{k}. If \code{delta.fix = TRUE}, this will be
\eqn{\delta} input by the user.

\eqn{k:} the number of close pairs included in the sample (for
\bold{inhibitory plus close pairs} design).

\code{sample.locs:} a \code{sf} or \code{sp} object containing
coordinates of dimension \code{n} by 2 containing the sampled locations.
}
\description{
Draws a spatially continous sample of locations within a
polygonal sampling region according to an \bold{'inhibitory plus close
pairs'} specification. The region can be defined using OSM data or a user
defined polygon.
}
\details{
To draw a simple inhibitory (\bold{SI}) sample of size \code{n}
from a spatially continuous region \eqn{A}, with the property that the
distance between any two sampled locations is at least \code{delta}, the
following algorithm is used. \itemize{ \item{Step 1.} Set \eqn{i  = 1} and
generate a point \eqn{x_{1}}  uniformly distributed on \eqn{{\cal D}}.
\item{Step 2.} Generate a point \eqn{x}  uniformly distributed on
\eqn{{\cal D}} and calculate the minimum, \eqn{d_{\min}}, of the distances
from \eqn{x_{i}} to all \eqn{x_{j}: j \leq i }. \item{Step 3.} If
\eqn{d_{\min} \ge \delta}, increase \eqn{i}  by 1, set \eqn{x_{i} = x} and
return to step 2 if \eqn{i \le n}, otherwise stop; \item{Step 4.} If
\eqn{d_{\min} < \delta}, return to step 2 without increasing \eqn{i}. }

\bold{Sampling close pairs of points.}

For some purposes, it is desirable that a spatial sampling scheme include
pairs of closely spaced points, resulting in an inhibitory plus close pairs
(\bold{ICP}) design. In this case, the above algorithm requires the
following additional steps to be taken. Let \eqn{k}  be the required number
of close pairs. Choose a value \code{rho}  such that a close pair  of
points will be a pair of points separated by a distance of at most
\code{rho}. \itemize{ \item{Step 5.} Set \eqn{j  = 1} and draw a random
sample of size 2 from integers \eqn{1, 2, \ldots, n}, say \eqn{(i_1, i_2)};
\item{Step 6.} Replace \eqn{x_{i_{1}}} by \eqn{x_{i_{2}} + u} , where
\eqn{u}  is uniformly distributed on the disc with centre \eqn{x_{i_{2}}}
and radius \code{rho}, increase \eqn{i} by 1 and return to step 5 if \eqn{i
 \le k}, otherwise stop. }

When comparing a \bold{SI} design to one of the \bold{ICP} designs, the
inhibitory components should have the same degree of spatial regularity.
This requires \eqn{\delta} to become a function of \eqn{k} namely
\deqn{\delta_{k} = \delta_{0}\sqrt{n/(n - k)}} with \eqn{\delta_{0}} held
fixed.
}
\note{
If \code{'delta'} is set to 0, a completely random sample is generated.
In this case, \code{'close pairs'} are not permitted and \code{rho} is
irrelevant.
}
\examples{
\dontrun{
library(sp)
bounding_geom<-
SpatialPolygonsDataFrame(
   SpatialPolygons(list(Polygons(list(Polygon(
       cbind(
           c(3.888959,3.888744,3.888585,3.888355,3.887893,3.887504,
           3.886955,3.886565,3.886303,3.886159,3.885650,3.885650,
           3.885595,3.885404,3.885444,3.885897,3.886692,3.887241,
           3.888068,3.888323,3.888697,3.889150,3.889548,3.889890,
           3.890184,3.890828,3.891258,3.891807,3.892061,3.892292,
           3.892689,3.893294,3.893008,3.893676,3.888959),
           c(7.379483,7.379785,7.380024,7.380294,7.380629,7.380986,
           7.381448,7.381861,7.382243,7.382474,7.383277,7.383468,
           7.383890,7.384263,7.384669,7.385258,7.385313,7.385194,
           7.384868,7.384900,7.385051,7.385067,7.384955,7.384749,
           7.384526,7.384120,7.384009,7.384080,7.384430,7.384478,
           7.384629,7.384772,7.383269,7.380963,7.379483)))), ID=1))),
   data.frame( ID=1))
proj4string(bounding_geom) <- CRS('+proj=longlat +datum=WGS84')

set.seed(15892)
osm_contin_inhibit_sample(bounding_geom = bounding_geom, boundary = 0,
buff_dist=NULL,
buff_epsg = NULL, sample_size = 50, plotit = TRUE, plotit_leaflet = TRUE,
                  delta=50, delta.fix = FALSE,k=7,rho=1, ntries = 10)
}

}
\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern
analysis code in S-Plus. Computers and Geosciences, 19, 627-655 Chipeta  M
G, Terlouw D J, Phiri K S and Diggle P J. (2016b). Inhibitory
geostatistical designs for spatial prediction taking account of uncertain
covariance structure, \emph{Enviromentrics}, pp. 1-11.
https://wiki.openstreetmap.org/wiki/Map_Features
}
\seealso{
\code{\link[osmgeosample:osm_random_sample]{osm_random_sample}} and
osm_discrete_inhibit_sample
}
\author{
Henry J. Crosby \email{henry.crosby@warwick.ac.uk}

Godwin Yeboah \email{godwin.yeboah@warwick.ac.uk}

J. Porto De Albuquerque \email{J.Porto@warwick.ac.uk}
}
