% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rexp.var.R
\name{rexp.var}
\alias{rexp.var}
\title{General rate exponential and Weibull waiting times}
\usage{
rexp.var(n, rate, now = 0, tMax = Inf, shape = NULL, TS = 0, fast = FALSE)
}
\arguments{
\item{n}{The number of waiting times to return. Usually 1, but we allow for
a higher \code{n} to be consistent with the \code{rexp} function.}

\item{rate}{The rate parameter for the exponential distribution. If
\code{shape} is not \code{NULL}, \code{rate} is a scale for a Weibull
distribution. In both cases we allow for any time-varying function. Note
\code{rate} can be constant.}

\item{now}{The current time. Needed if one wants to consider only the 
interval between the current time and the maximum time for the time-varying
rate. Note this does means the waiting time is \code{>= now}, so we also 
subtract \code{now} from the result before returning. The default is 
\code{0}.}

\item{tMax}{The simulation ending time. If the waiting time would be too
high, we return \code{tMax + 0.01} to signify the event never happens, if
\code{fast == TRUE}. Otherwise we return the true waiting time. By default,
\code{tMax} will be \code{Inf}, but if \code{FAST == TRUE} one must supply
a finite value.}

\item{shape}{Shape of the Weibull distribution. Can be a \code{numeric} for 
constant shape or a \code{function(t)} for time-varying. When smaller than
one, rate will decrease along species' age (negative age-dependency). When 
larger than one, rate will increase along each species' age (positive 
age-dependency). Default is \code{NULL}, so the function acts as an 
exponential distribution. For \code{shape != NULL} (including when equal to 
one), \code{rate} will be considered a scale (= 1/rate), and \code{rexp.var} 
will draw a Weibull distribution instead of an exponential. This means 
Weibull(rate, 1) = Exponential(1/rate). Notice even when 
\code{Shape != NULL}, \code{rate} may still be time-dependent. 

Note: Time-varying shape is within expectations for most cases, but if it is
lower than 1 and varies too much (e.g. \code{0.5 + 0.5*t}), it can be 
slightly biased for higher waiting times due to computational error. Slopes 
(or equivalent, since it can be any function of time) of the order of 0.01 
are advisable. It rarely also displays small biases for abrupt variations. 
In both cases, error is still quite low for the purposes of the package.

Note: We do not test for shape < 0 here since as we allow shape to be a 
function this would severely slow the rest of the package. It is tested on
the birth-death functions, and the user should make sure not to use any
functions that become negative eventually.}

\item{TS}{Speciation time, used to account for the scaling between simulation 
and species time. The default is \code{0}. Supplying a \code{TS > now} will
return an error.}

\item{fast}{If set to \code{FALSE}, waiting times larger than \code{tMax} will
not be thrown away. This argument is needed so one can test the function
without bias.}
}
\value{
A vector of waiting times following the exponential or Weibull 
distribution with the given parameters.
}
\description{
Generates a waiting time following an exponential or Weibull distribution 
with constant or varying rates. Output can be used as the waiting time to an
extinction, speciation, or fossil sampling event. Allows for an optional 
shape parameter, in which case \code{rate} will be taken as a Weibull scale.
Allows for further customization by restricting possible waiting time 
outputs with arguments for (1) current time, to consider only the rates
starting at that time, (2) maximum time, to bound the output and therefore
allow for faster calculations if one only cares about waiting times lower 
than a given amount, and (3) speciation time, necessary to scale rates in
the case where the output is to follow a Weibull distribution, i.e. for 
age-dependent processes. This function is used in birth-death and sampling
functions, but can also be convenient for any user looking to write their
own code requiring exponential or Weibull distributions with varying rates.
}
\examples{

###
# let us start by checking a simple exponential variable

# rate
rate <- 0.1

# set seed
set.seed(1)

# find the waiting time
t <- rexp.var(n = 1, rate)
t

# this is the same as t <- rexp(1, rate)

###
# now let us try a linear function for the rate

# rate
rate <- function(t) {
  return(0.01*t + 0.1)
}

# set seed
set.seed(1)

# find the waiting time
t <- rexp.var(n = 1, rate)
t

###
# what if rate is exponential?

# rate
rate <- function(t) {
  return(0.01 * exp(0.1*t) + 0.02)
}

# set seed
set.seed(1)

# find the waiting time
t <- rexp.var(n = 1, rate)
t

###
# if we give a shape argument, we have a Weibull distribution

# scale - note that this is equivalent to 1/rate if shape were NULL
rate <- 2

# shape
shape <- 1

# speciation time
TS <- 0

# set seed
set.seed(1)

# find the vector of waiting time
t <- rexp.var(n = 1, rate, shape = shape, TS = TS)
t

###
# when shape = 1, the Weibull is an exponential, we could do better

# scale
rate <- 10

# shape
shape <- 2

# speciation time
TS <- 0

# set seed
set.seed(1)

# find the vector of waiting times - it doesn't need to be just one
t <- rexp.var(n = 5, rate, shape = shape, TS = TS)
t

###
# shape can be less than one, of course

# scale
rate <- 10

# shape
shape <- 0.5

# note we can supply now (default 0) and tMax (default Inf)

# now matters when we wish to consider only waiting times
# after that time, important when using time-varying rates
now <- 3

# tMax matters when fast = TRUE, so that higher times will be
# thrown out and returned as tMax + 0.01
tMax <- 40

# speciation time - it will be greater than 0 frequently during a 
# simulation, as it is used to represent where in the species life we 
# currently are and rescale accordingly
TS <- 2.5

# set seed
set.seed(1)

# find the vector of waiting times
t <- rexp.var(n = 10, rate, now, tMax,
              shape = shape, TS = TS, fast = TRUE)
t

# note how some results are tMax + 0.01, since fast = TRUE

###
# both rate and shape can be time varying for a Weibull

# scale
rate <- function(t) {
  return(0.25*t + 5)
}

# shape
shape <- 3

# current time
now <- 0

# maximum time to check
tMax <- 40

# speciation time
TS <- 0

# set seed
set.seed(1)

# find the vector of waiting times
t <- rexp.var(n = 5, rate, now, tMax,
              shape = shape, TS = TS, fast = TRUE)
t

}
\author{
Bruno do Rosario Petrucci.
}
